<?php
    // Props: data (array), rows (array of uniqueName strings), measures (array of {uniqueName, aggregation, format}), formats (array), height (int|string|'auto')
    $componentId = 'pivot_' . uniqid();
    $dataProp = $data ?? [];
    $rowsProp = $rows ?? [];
    $measuresProp = $measures ?? [];
    $formatsProp = $formats ?? [];
    $heightProp = $height ?? 'auto';
    $heightStyle = '';
    if ($heightProp !== 'auto' && $heightProp !== null && $heightProp !== '') {
        $heightStyle = 'height: ' . (is_numeric($heightProp) ? intval($heightProp) . 'px' : htmlspecialchars($heightProp, ENT_QUOTES)) . ';';
    }
?>
<div id="<?= $componentId ?>" class="pivot-container"<?= $heightStyle ? ' style="' . $heightStyle . '"' : '' ?>>
    <style>
        /* Container */
        #<?= $componentId ?> .pivot-toolbar { background: var(--bs-body-tertiary-bg, var(--bs-tertiary-bg, #f8f9fa)); }
        #<?= $componentId ?> .table thead th { position: sticky; top: 0; z-index: 2; background: var(--bs-body-bg); text-transform: uppercase; font-size: .75rem; letter-spacing: .02em; }
        #<?= $componentId ?> .table tbody tr.pivot-row:hover { background-color: var(--bs-light); }
        #<?= $componentId ?> .pivot-dim { white-space: nowrap; }
        #<?= $componentId ?> th.pivot-col-dim, #<?= $componentId ?> td.pivot-col-dim { white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
        #<?= $componentId ?> th.pivot-col-dim { font-weight: 600; }
        #<?= $componentId ?> .pivot-col-dim { width: 160px; max-width: 200px; }
        #<?= $componentId ?> .pivot-col-dim.pivot-col-dim-narrow { width: 110px; max-width: 130px; }
        #<?= $componentId ?> table { border-radius: .5rem; }
        #<?= $componentId ?> .table > :not(caption) > * > * { box-shadow: inset 0 -1px 0 rgba(0,0,0,.05); }
        #<?= $componentId ?> .table thead th { box-shadow: inset 0 -1px 0 rgba(0,0,0,.1); }
        /* Full height layout */
        html, body, #page-content, #page-content-container, #<?= $componentId ?> { height: 100%; }
        #<?= $componentId ?> .table-responsive { max-height: none; }
    </style>
    <div class="pivot-toolbar d-flex justify-content-between align-items-center mb-2 border rounded px-2 py-2 shadow-sm">
        <div class="btn-group btn-group-sm" role="group" aria-label="Pivot controls">
            <button type="button" class="btn btn-outline-secondary" id="<?= $componentId ?>_expand_all">Expand all</button>
            <button type="button" class="btn btn-outline-secondary" id="<?= $componentId ?>_collapse_all">Collapse all</button>
            <button type="button" class="btn btn-outline-secondary" id="<?= $componentId ?>_reset">Reset</button>
            <button type="button" class="btn btn-primary" id="<?= $componentId ?>_export_xlsx">Export Excel</button>
            <?php if (!empty($exportUrl ?? '')): ?>
            <button type="button" class="btn btn-success" id="<?= $componentId ?>_export_pivot_xlsx">Export Pivot (XLSX)</button>
            <?php endif; ?>
        </div>
        <div class="d-flex align-items-center gap-2">
            <span class="small text-muted">Filter</span>
            <select id="<?= $componentId ?>_filter_factory" class="form-select form-select-sm" style="min-width: 12rem;"></select>
            <select id="<?= $componentId ?>_filter_customer" class="form-select form-select-sm" style="min-width: 12rem;"></select>
        </div>
    </div>
    <div class="table-responsive border rounded shadow-sm">
        <table class="table table-sm align-middle table-hover table-striped mb-0" aria-label="Pivot table">
            <thead></thead>
            <tbody></tbody>
        </table>
    </div>

    <!-- Drill-down Modal -->
    <div class="modal fade" id="<?= $componentId ?>_drill" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-xl modal-dialog-scrollable">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="d-flex justify-content-between align-items-center mb-2">
                        <div id="<?= $componentId ?>_drill_summary" class="small text-muted"></div>
                        <input id="<?= $componentId ?>_drill_search" class="form-control form-control-sm" placeholder="Search..." />
                    </div>
                    <div class="table-responsive">
                        <table class="table table-sm table-bordered align-middle" id="<?= $componentId ?>_drill_table">
                            <thead></thead>
                            <tbody></tbody>
                        </table>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
    (function(){
        const rootId = "<?= $componentId ?>";
        const rawData = <?= json_encode($dataProp) ?>;
        const rowDims = <?= json_encode(array_values($rowsProp)) ?>;
        const measures = <?= json_encode($measuresProp) ?>;
        const formatsList = <?= json_encode($formatsProp) ?>;
        const exportPivotUrl = <?= isset($exportUrl) ? json_encode($exportUrl) : 'null' ?>;

        const formatsByName = {};
        (formatsList || []).forEach(function(fmt){
            if (fmt && fmt.name) formatsByName[fmt.name] = fmt;
        });

        function formatNumber(value, fmtName){
            if (value == null || isNaN(value)) return '';
            const fmt = formatsByName[fmtName];
            const options = { minimumFractionDigits: (fmt && typeof fmt.decimalPlaces === 'number') ? fmt.decimalPlaces : 0, maximumFractionDigits: (fmt && typeof fmt.decimalPlaces === 'number') ? fmt.decimalPlaces : 2 };
            const parts = Number(value).toLocaleString(undefined, options);
            if (!fmt) return parts;
            const sym = fmt.currencySymbol || '';
            if (!sym) return parts;
            return fmt.currencySymbolAlign === 'left' ? sym + parts : parts + sym;
        }

        function buildGroups(data, dims, level, path){
            if (level >= dims.length) {
                return { path: path.slice(), rows: data.slice(), sums: sumMeasures(data) };
            }
            const dim = dims[level];
            const map = new Map();
            data.forEach(function(row){
                const key = row[dim];
                const arr = map.get(key) || [];
                arr.push(row);
                map.set(key, arr);
            });
            const children = [];
            map.forEach(function(rows, key){
                const child = buildGroups(rows, dims, level + 1, path.concat([{ dim: dim, value: key }]));
                child.dim = dim;
                child.value = key;
                child.level = level;
                children.push(child);
            });
            // Aggregate sums for this group from children
            const sums = {};
            measures.forEach(function(m){ sums[m.uniqueName] = 0; });
            children.forEach(function(ch){
                measures.forEach(function(m){ sums[m.uniqueName] += Number(ch.sums[m.uniqueName] || 0); });
            });
            return { path: path.slice(), children: children, sums: sums, dim: dim, level: level };
        }

        function sumMeasures(rows){
            const sums = {};
            measures.forEach(function(m){ sums[m.uniqueName] = 0; });
            rows.forEach(function(r){
                measures.forEach(function(m){ sums[m.uniqueName] += Number(r[m.uniqueName] || 0); });
            });
            return sums;
        }

        const expanded = new Set(); // store JSON stringified path keys for expanded nodes
        let sortKey = null; // current measure sort key
        let sortDir = 'desc'; // 'asc' | 'desc'
        let lastRoot = null;

        function pathKey(node){
            const parts = (node.path || []).map(function(p){ return { dim: p.dim, value: p.value }; });
            if (node.dim !== undefined) parts.push({ dim: node.dim, value: node.value });
            return JSON.stringify(parts);
        }

        function getVisibleDimsCount(){
            if (!expanded || expanded.size === 0) return Math.min(1, rowDims.length);
            let maxLevel = -1;
            expanded.forEach(function(k){
                try {
                    const parts = JSON.parse(k);
                    if (Array.isArray(parts)) {
                        const level = parts.length - 1;
                        if (level > maxLevel) maxLevel = level;
                    }
                } catch(e) {}
            });
            // Show columns up to the deepest expanded level + 1
            const count = (maxLevel + 2);
            return Math.max(1, Math.min(rowDims.length, count));
        }

        function render(){
            const container = document.getElementById(rootId);
            if (!container) return;
            const thead = container.querySelector('thead');
            const tbody = container.querySelector('tbody');

            // Header
            thead.innerHTML = '';
            const headerRow = document.createElement('tr');
            const visibleDims = getVisibleDimsCount();
            rowDims.slice(0, visibleDims).forEach(function(dim, idx){
                const th = document.createElement('th');
                th.className = 'pivot-col-dim' + (idx > 0 ? ' pivot-col-dim-narrow' : '');
                th.textContent = dim;
                headerRow.appendChild(th);
            });
            measures.forEach(function(m){
                const th = document.createElement('th');
                th.style.cursor = 'pointer';
                if ((m.uniqueName || '').toLowerCase().includes('base')) {
                    th.classList.add('fw-bold');
                }
                let title = m.uniqueName;
                if (sortKey === m.uniqueName) title += sortDir === 'asc' ? ' ▲' : ' ▼';
                th.textContent = title;
                th.addEventListener('click', function(){
                    if (sortKey === m.uniqueName) {
                        sortDir = (sortDir === 'asc') ? 'desc' : 'asc';
                    } else {
                        sortKey = m.uniqueName; sortDir = 'desc';
                    }
                    render();
                });
                headerRow.appendChild(th);
            });
            thead.appendChild(headerRow);

            // Body
            tbody.innerHTML = '';
            // Apply select filters (Factory, Customer)
            const ff = document.getElementById(rootId + '_filter_factory');
            const fc = document.getElementById(rootId + '_filter_customer');
            const factoryVal = ff && ff.value ? ff.value : '';
            const customerVal = fc && fc.value ? fc.value : '';
            const filtered = rawData.filter(function(r){
                if (factoryVal && r['Factory'] != factoryVal) return false;
                if (customerVal && r['Customer'] != customerVal) return false;
                return true;
            });

            const root = buildGroups(filtered, rowDims, 0, []);
            lastRoot = root;
            // Depth-first render with collapsible nodes (collapsed by default)
            function renderNode(node){
                if (node.level != null && node.value !== undefined) {
                    const tr = document.createElement('tr');
                    tr.className = 'pivot-row';
                    // Row dims: show only the current node's dimension label, not parents/children
                    const visibleDims = getVisibleDimsCount();
                    rowDims.slice(0, visibleDims).forEach(function(dim, idx){
                        const td = document.createElement('td');
                        td.className = 'pivot-col-dim' + (idx > 0 ? ' pivot-col-dim-narrow' : '');
                        const isNodeDim = (idx === node.level);
                        if (isNodeDim) {
                            const hasChildren = Array.isArray(node.children) && node.children.length > 0;
                            if (hasChildren) {
                                const key = pathKey(node);
                                const isOpen = expanded.has(key);
                                const toggle = document.createElement('span');
                                toggle.textContent = isOpen ? '▾' : '▸';
                                toggle.style.cursor = 'pointer';
                                toggle.className = 'me-1 text-secondary';
                                toggle.addEventListener('click', function(e){
                                    e.stopPropagation();
                                    if (expanded.has(key)) expanded.delete(key); else expanded.add(key);
                                    render();
                                });
                                td.appendChild(toggle);
                            }
                            const text = document.createElement('span');
                            text.className = 'pivot-dim';
                            text.style.paddingLeft = (node.level * 12) + 'px';
                            const val = node.value;
                            text.textContent = (val === null || typeof val === 'undefined') ? '' : val;
                            td.appendChild(text);
                        }
                        tr.appendChild(td);
                    });
                    // Measures
                    measures.forEach(function(m){
                        const td = document.createElement('td');
                        td.className = 'text-end';
                        if ((m.uniqueName || '').toLowerCase().includes('base')) {
                            td.classList.add('fw-bold');
                        }
                        const v = node.sums[m.uniqueName] || 0;
                        td.textContent = formatNumber(v, m.format);
                        td.style.cursor = 'pointer';
                        td.addEventListener('click', function(){ onDrill(node, m, v); });
                        tr.appendChild(td);
                    });
                    tbody.appendChild(tr);
                }
                const key = pathKey(node);
                const isOpen = expanded.has(key);
                if (Array.isArray(node.children) && isOpen) {
                    const children = node.children.slice();
                    if (sortKey) {
                        children.sort(function(a, b){
                            const av = Number(a.sums[sortKey] || 0);
                            const bv = Number(b.sums[sortKey] || 0);
                            return av - bv;
                        });
                        if (sortDir === 'desc') children.reverse();
                    } else {
                        // Default sorting for dimensions - special handling for Money In Week
                        children.sort(function(a, b){
                            // Special handling for Money In Week dimension to sort by date
                            if (a.dim === 'Money In Week' && b.dim === 'Money In Week') {
                                // Parse the Y: W format (e.g., "2024: 15") to create comparable dates
                                const parseWeekString = function(weekStr) {
                                    if (!weekStr || typeof weekStr !== 'string') return new Date(0);
                                    const parts = weekStr.split(':');
                                    if (parts.length !== 2) return new Date(0);
                                    const year = parseInt(parts[0].trim());
                                    const week = parseInt(parts[1].trim());
                                    if (isNaN(year) || isNaN(week)) return new Date(0);
                                    
                                    // Use ISO week calculation - week 1 is the first week with at least 4 days in the year
                                    // Find the first Thursday of the year, then go back to Monday
                                    const jan4 = new Date(year, 0, 4); // January 4th is always in week 1
                                    const jan4Day = jan4.getDay(); // 0 = Sunday, 1 = Monday, etc.
                                    const daysToMonday = jan4Day === 0 ? 6 : jan4Day - 1; // Days to subtract to get to Monday
                                    const week1Monday = new Date(jan4.getTime() - daysToMonday * 24 * 60 * 60 * 1000);
                                    
                                    // Add (week - 1) * 7 days to get to the desired week
                                    const daysToAdd = (week - 1) * 7;
                                    return new Date(week1Monday.getTime() + daysToAdd * 24 * 60 * 60 * 1000);
                                };
                                const aDate = parseWeekString(a.value);
                                const bDate = parseWeekString(b.value);
                                return aDate - bDate;
                            }
                            // For other dimensions, sort alphabetically
                            const aVal = (a.value || '').toString();
                            const bVal = (b.value || '').toString();
                            return aVal.localeCompare(bVal);
                        });
                    }
                    children.forEach(renderNode);
                }
            }
            if (Array.isArray(root.children)) {
                // Apply the same sorting logic to root level children
                const rootChildren = root.children.slice();
                if (sortKey) {
                    rootChildren.sort(function(a, b){
                        const av = Number(a.sums[sortKey] || 0);
                        const bv = Number(b.sums[sortKey] || 0);
                        return av - bv;
                    });
                    if (sortDir === 'desc') rootChildren.reverse();
                } else {
                    // Default sorting for dimensions - special handling for Money In Week
                    rootChildren.sort(function(a, b){
                        // Special handling for Money In Week dimension to sort by date
                        if (a.dim === 'Money In Week' && b.dim === 'Money In Week') {
                            // Parse the Y: W format (e.g., "2024: 15") to create comparable dates
                            const parseWeekString = function(weekStr) {
                                if (!weekStr || typeof weekStr !== 'string') return new Date(0);
                                const parts = weekStr.split(':');
                                if (parts.length !== 2) return new Date(0);
                                const year = parseInt(parts[0].trim());
                                const week = parseInt(parts[1].trim());
                                if (isNaN(year) || isNaN(week)) return new Date(0);
                                
                                // Use ISO week calculation - week 1 is the first week with at least 4 days in the year
                                // Find the first Thursday of the year, then go back to Monday
                                const jan4 = new Date(year, 0, 4); // January 4th is always in week 1
                                const jan4Day = jan4.getDay(); // 0 = Sunday, 1 = Monday, etc.
                                const daysToMonday = jan4Day === 0 ? 6 : jan4Day - 1; // Days to subtract to get to Monday
                                const week1Monday = new Date(jan4.getTime() - daysToMonday * 24 * 60 * 60 * 1000);
                                
                                // Add (week - 1) * 7 days to get to the desired week
                                const daysToAdd = (week - 1) * 7;
                                return new Date(week1Monday.getTime() + daysToAdd * 24 * 60 * 60 * 1000);
                            };
                            const aDate = parseWeekString(a.value);
                            const bDate = parseWeekString(b.value);
                            return aDate - bDate;
                        }
                        // For other dimensions, sort alphabetically
                        const aVal = (a.value || '').toString();
                        const bVal = (b.value || '').toString();
                        return aVal.localeCompare(bVal);
                    });
                }
                rootChildren.forEach(renderNode);
            }

            // Grand total row (align with visible dimension columns)
            const totalRow = document.createElement('tr');
            const totalVisibleDims = getVisibleDimsCount();
            for (let idx = 0; idx < totalVisibleDims; idx++) {
                const td = document.createElement('td');
                td.className = 'pivot-col-dim' + (idx > 0 ? ' pivot-col-dim-narrow' : '');
                if (idx === 0) td.textContent = 'Grand Total';
                totalRow.appendChild(td);
            }
            measures.forEach(function(m){
                const td = document.createElement('td');
                td.className = 'text-end fw-bold';
                td.textContent = formatNumber(root.sums[m.uniqueName] || 0, m.format);
                totalRow.appendChild(td);
            });
            tbody.appendChild(totalRow);

            // No free-text filter anymore
        }

        function onDrill(node, measure, cellValue){
            // Build filters from node.path + node current
            const filters = {};
            (node.path || []).forEach(function(p){ filters[p.dim] = p.value; });
            if (node.dim) filters[node.dim] = node.value;
            const rows = rawData.filter(function(r){
                for (const k in filters) {
                    if (r[k] != filters[k]) return false;
                }
                return true;
            });
            showDrill(rows, measure, cellValue, filters);
        }

        function showDrill(rows, measure, cellValue, filters){
            const modalId = rootId + '_drill';
            const modalEl = document.getElementById(modalId);
            const summaryEl = document.getElementById(rootId + '_drill_summary');
            const thead = document.querySelector('#' + rootId + '_drill_table thead');
            const tbody = document.querySelector('#' + rootId + '_drill_table tbody');

            // Columns
            const preferred = [
                'Money In Week', 'Month', 'Week No', 'Date', 'Factory', 'Customer', 'Order', 'Style', 'Colourway', 'Percent Of Total Sale',
                'Sales GBP', 'Sales Euro', 'Sales USD', 'CMT GBP', 'CMT Euro', 'CMT USD', 'Yarn GBP', 'Yarn Euro', 'Yarn USD',
                'Base Sales', 'Base CMT', 'Base Yarn', 'Factoring', 'drop'
            ];
            const columns = (function(){
                if (!rows.length) return preferred;
                const keys = Object.keys(rows[0]);
                const ordered = preferred.filter(k => keys.includes(k));
                const rest = keys.filter(k => !ordered.includes(k));
                return ordered.concat(rest);
            })();

            thead.innerHTML = '';
            const hr = document.createElement('tr');
            columns.forEach(function(c){ const th = document.createElement('th'); th.textContent = c; hr.appendChild(th); });
            thead.appendChild(hr);

            tbody.innerHTML = '';
            rows.slice(0, 1000).forEach(function(r){
                const tr = document.createElement('tr');
                columns.forEach(function(c){ const td = document.createElement('td'); const v = r[c]; td.textContent = (v==null? '': v); tr.appendChild(td); });
                tbody.appendChild(tr);
            });

            // Summary text
            const filterText = Object.keys(filters || {}).map(k => k + ': ' + filters[k]).join(' • ');
            const mName = measure && (measure.uniqueName || measure.caption || 'Value');
            summaryEl.textContent = `${rows.length} record(s) • ${mName} • Cell: ${formatNumber(cellValue, measure && measure.format)}${filterText ? ' • ' + filterText : ''}`;

            // Show modal (Bootstrap 5 / 4)
            if (window.bootstrap && bootstrap.Modal) {
                const instance = bootstrap.Modal.getOrCreateInstance(modalEl);
                instance.show();
            } else if (window.$ && $(modalEl).modal) {
                $(modalEl).modal('show');
            } else {
                modalEl.style.display = 'block';
            }

            const search = document.getElementById(rootId + '_drill_search');
            if (search) {
                search.value = '';
                search.oninput = function(e){
                    const term = (e.target.value || '').toLowerCase();
                    Array.from(tbody.rows).forEach(function(row){
                        row.style.display = row.textContent.toLowerCase().includes(term) ? '' : 'none';
                    });
                };
            }
        }

        // Toolbar controls
        document.getElementById(rootId + '_expand_all').addEventListener('click', function(){
            if (!lastRoot) return;
            function collect(n){
                if (!n) return;
                if (n.level != null) expanded.add(pathKey(n));
                (n.children || []).forEach(collect);
            }
            expanded.clear();
            (lastRoot.children || []).forEach(collect);
            render();
        });
        document.getElementById(rootId + '_collapse_all').addEventListener('click', function(){
            expanded.clear();
            render();
        });
        document.getElementById(rootId + '_reset').addEventListener('click', function(){
            expanded.clear(); sortKey = null; sortDir = 'desc';
            const f1 = document.getElementById(rootId + '_filter_factory'); if (f1) f1.value = '';
            const f2 = document.getElementById(rootId + '_filter_customer'); if (f2) f2.value = '';
            render();
        });
        const ff = document.getElementById(rootId + '_filter_factory');
        const fc = document.getElementById(rootId + '_filter_customer');
        function populateSelect(selectEl, options){
            if (!selectEl) return;
            const val = selectEl.value;
            selectEl.innerHTML = '';
            const optAny = document.createElement('option');
            optAny.value = ''; optAny.textContent = 'All';
            selectEl.appendChild(optAny);
            options.forEach(function(o){ const opt = document.createElement('option'); opt.value = o; opt.textContent = o; selectEl.appendChild(opt); });
            if (val) selectEl.value = val;
        }
        // Build option sets from raw data
        const factories = Array.from(new Set((rawData || []).map(r => r['Factory']).filter(Boolean))).sort();
        const customers = Array.from(new Set((rawData || []).map(r => r['Customer']).filter(Boolean))).sort();
        populateSelect(ff, factories);
        populateSelect(fc, customers);
        ff && ff.addEventListener('change', render);
        fc && fc.addEventListener('change', render);

        render();

        // Export to Excel (current rendered table)
        function ensureSheetJS(callback){
            if (window.XLSX) return callback();
            const s = document.createElement('script');
            s.src = 'https://cdn.jsdelivr.net/npm/xlsx@0.19.3/dist/xlsx.full.min.js';
            s.onload = callback;
            document.head.appendChild(s);
        }
        document.getElementById(rootId + '_export_xlsx').addEventListener('click', function(){
            ensureSheetJS(function(){
                try {
                    const table = document.querySelector('#' + rootId + ' table');
                    if (!table) return;
                    const wb = XLSX.utils.table_to_book(table, { sheet: 'Pivot' });
                    const now = new Date();
                    const pad = (n) => (n<10?'0':'') + n;
                    const fname = 'pivot_export_' + now.getFullYear() + pad(now.getMonth()+1) + pad(now.getDate()) + '_' + pad(now.getHours()) + pad(now.getMinutes()) + '.xlsx';
                    XLSX.writeFile(wb, fname);
                } catch(e) {
                    console.error('Export failed', e);
                }
            });
        });

        // Server-side pivot export
        const expPivotBtn = document.getElementById(rootId + '_export_pivot_xlsx');
        if (expPivotBtn && exportPivotUrl) {
            expPivotBtn.addEventListener('click', function(){
                const ff = document.getElementById(rootId + '_filter_factory');
                const fc = document.getElementById(rootId + '_filter_customer');
                const payload = {
                    factory: ff && ff.value ? ff.value : '',
                    customer: fc && fc.value ? fc.value : '',
                    rows: rowDims,
                    measures: measures,
                    data: rawData
                };
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = exportPivotUrl;
                const csrf = document.querySelector('meta[name="csrf-token"]');
                if (csrf) {
                    const input = document.createElement('input'); input.type = 'hidden'; input.name = '_token'; input.value = csrf.getAttribute('content'); form.appendChild(input);
                }
                Object.keys(payload).forEach(function(k){
                    const input = document.createElement('input');
                    input.type = 'hidden';
                    input.name = k;
                    input.value = typeof payload[k] === 'string' ? payload[k] : JSON.stringify(payload[k]);
                    form.appendChild(input);
                });
                document.body.appendChild(form);
                form.submit();
                document.body.removeChild(form);
            });
        }
    })();
</script>


