<?php

use App\Models\Price;
use App\Models\Sizes;
use GuzzleHttp\Client;
use App\Models\Seasons;
use Symfony\Component\DomCrawler\Crawler;

if (!function_exists('currencyToRate')) {
    function currencyToRate($currency = "")
    {
        if($currency == '€')
            return 'euro_rate';
        elseif($currency == '$')
            return 'usd_rate';
        elseif($currency == '£')
            return 'gbp_rate';
        else
            return '';
    }
}

if (!function_exists('calculateYarnValueEur')) {
    function calculateYarnValueEur(array $price)
    {
        if($price['model'] == 'standard')
            $yarnCostKgEur = (float)($price['yarn_cost_kg_eur'] ?? 0);
        elseif($price['model'] == 'manual'){
            $yarnCostKgEur = (float)convertCurrency($price['yarnkg_currency'], '€', $price['yarn'], $price['seasons_id'], $price[currencyToRate($price['yarnkg_currency'])] ?? 0);
        }
        else{
            $yarnCostKgEur = 0;
        }

        // Convert yarn_trans to Euro based on its currency
        $yarnTransCurrency = $price['yarn_trans_currency'] ?? '€';
        $yarnTransRate = $price[currencyToRate($yarnTransCurrency)] ?? 0;
        $yarnTransEur = (float)convertCurrency($yarnTransCurrency, '€', $price['yarn_trans'], $price['seasons_id'], $yarnTransRate);

        // Calculate base yarn value
        $val = ($yarnCostKgEur + $yarnTransEur) * ((float)($price['costed_weight'] ?? 0));
        
        // Apply yarn interest if applicable
        $yarnInterest = (float)($price['yarn_interest'] ?? 0);
        if ($yarnInterest > 0) {
            // Get number of days from season (default to 365)
            $noDays = (int)($price['no_days_yarn_interest'] ?? 365);
            
            // Apply AER formula: A = P * (1 + r)^(days/365)
            // Convert percentage to decimal (e.g., 5% becomes 0.05)
            $interestRate = $yarnInterest / 100;
            $val = $val * pow(1 + $interestRate, $noDays / 365);
        }
        
        return number_format($val, 2, '.', '');
    }
}

if (!function_exists('calculateSubtotal')) {
    function calculateSubtotal(array $price)
    {
        // Cache conversion rates for the price's fact_currency to minimize multiple calls.
        $factCurrencyRate = $price[currencyToRate($price['fact_currency'])] ?? null;
        $euroRate = $price[currencyToRate('€')] ?? null;
        $poundRate = $price[currencyToRate('£')] ?? null;

        // Cast values to float to handle string values from MySQL format()
        $cmt = (float)($price['cmt'] ?? 0);
        $aoc = (float)($price['aoc'] ?? 0);
        $duty = (float)($price['duty'] ?? 0);

        // Calculate the subtotal in Euro using the cached rates
        $subtotalEuro = (float)calculateYarnValueEur($price) +
            $cmt +
            $cmt * ($aoc / 100) +
            (float)convertCurrency($price['accessory_currency'], $price['fact_currency'], $price['accessory'], $price['seasons_id'], $factCurrencyRate) +
            (float)convertCurrency($price['embroidery_currency'], $price['fact_currency'], $price['embroidery'], $price['seasons_id'], $factCurrencyRate) +
            (float)convertCurrency($price['label_currency'], $price['fact_currency'], $price['label'], $price['seasons_id'], $factCurrencyRate) +
            (float)convertCurrency($price['gmt_trans_currency'], $price['fact_currency'], $price['gmt_trans'], $price['seasons_id'], $factCurrencyRate) +
            (float)convertCurrency('£', $price['fact_currency'], $price['testing'], $price['seasons_id'], $factCurrencyRate) +
            (float)convertCurrency('£', $price['fact_currency'], $price['testing_courier'], $price['seasons_id'], $factCurrencyRate);

        // Apply duty rate if provided
        $subtotalEuro *= (($duty / 100) + 1);

        // Add customs if applicable
        if ($price['customer_customs'] || $price['factory_customs']) {
            $subtotalEuro += (float)convertCurrency('£', $price['fact_currency'], env('CUSTOMS'), $price['seasons_id'], $factCurrencyRate);
        }

        // Add factoring cost (assuming calculateFactoring function is defined elsewhere)
        $subtotalEuro += (float)calculateFactoring($price);

        // Return the subtotal formatted to 2 decimal places
        return number_format($subtotalEuro, 2, '.', '');
    }
}

if (!function_exists('calculateSubtotalGBP')) {
    function calculateSubtotalGBP(array $price)
    {
        $cost = (float)calculateSubtotal($price);

        return number_format(convertCurrency($price['fact_currency'], '£', $cost, $price['seasons_id'], $price[currencyToRate($price['fact_currency'])] ?? null), 2);
    }
}

if (!function_exists('calculateCustoms')) {
    function calculateCustoms(array $price)
    {
        if($price['customer_customs'] || $price['factory_customs'])
            return number_format(env('CUSTOMS'), 2);
        else
            return 0;
    }
}

if (!function_exists('fetchBaseRate')) {
    function fetchBaseRate()
    {
        // Define the cache key
        $cacheKey = 'boe_base_rate_value';

        // Check if the rate is already cached
        return Cache::remember($cacheKey, now()->addWeek(), function () {
            // If not cached, fetch the value from the website

            // Create a Guzzle HTTP client
            $client = new Client();

            // Make a GET request to the URL
            $response = $client->request('GET', 'https://www.bankofengland.co.uk/boeapps/database/Bank-Rate.asp#');

            // Check if the response is successful
            if ($response->getStatusCode() === 200) {
                // Get the response body (HTML)
                $html = $response->getBody()->getContents();

                // Create a new Crawler instance
                $crawler = new Crawler($html);

                // Use the crawler to find the element by its class name
                $rate = $crawler->filter('.stat-figure')->text();

                // Remove the percentage symbol
                $rate = str_replace('%', '', $rate);

                // Trim & Convert to number
                $rate = (float) trim($rate);

                // Return as percentage
                return $rate / 100;
            }

            // If the response wasn't successful, return null
            return null;
        });
    }
}

if (!function_exists('calculateFactoring')) {
    function calculateFactoring(array $price)
    {
        // Cast quote to float to handle string values from MySQL format()
        $quote = (float)($price['quote'] ?? 0);
        
        // Calculate the base item fee
        $itemFee = (float)env('FACTORING_CHARGE', 0);
        $itemFee += ($quote * 1.2 * (float)env('FACTORING_CHARGE_PERCENTAGE', 0));

        // Annual Equivalent Rate (AER)
        $aer = fetchBaseRate() + env('FACTORING_INTEREST');
        $days_in_year = 365;

        // Decode JSON terms and set up total interest fee
        if(is_string($price['cust_terms'])){
            $terms = json_decode($price['cust_terms'], true);
        }
        else{
            $terms = json_decode(json_encode($price['cust_terms']), true);
        }

        $total_interest_fee = 0;

        foreach ($terms ?? [] as $term) {
            // Ensure term has both 'days' and 'percentage' keys
            if (!isset($term['days']) || !isset($term['percentage'])) {
                continue; // Skip invalid terms
            }

            $days = $term['days'];
            $percentage = $term['percentage'] / 100;

            // Step 1: Calculate the daily interest rate from AER
            $daily_rate = pow(1 + $aer, 1 / $days_in_year) - 1;

            // Step 2: Calculate the interest for the specified days
            $interest_for_days = pow(1 + $daily_rate, $days) - 1;

            // Step 3: Calculate the interest fee for this term
            $interest_fee = ($quote * 1.2) * $interest_for_days * $percentage; // Adjust to include VAT

            // Add to total interest fee
            $total_interest_fee += $interest_fee;
        }

        // Calculate total factoring cost
        $total_factoring_cost = $itemFee + $total_interest_fee;

        // Format the result to 2 decimal places
        return number_format($total_factoring_cost, 2);
    }
}

if (!function_exists('calculateQuoteGBP')) {
    function calculateQuoteGBP(array $price)
    {
        $quote = (float)($price['quote'] ?? 0);
        
        return number_format(convertCurrency($price['cust_currency'], '£', $quote, $price['seasons_id'], $price[currencyToRate($price['fact_currency'])] ?? null), 2);
    }
}

if (!function_exists('calculateMargin')) {
    function calculateMargin(array $price)
    {
        $quote = (float)($price['quote'] ?? 0);
        if($quote > 0) {
            $discountPriceGBP = (float)discountPriceGBPRaw($price);
            $subtotalGBP = (float)calculateSubtotalGBP($price);
            $quoteGBP = (float)calculateQuoteGBP($price);
            
            if ($discountPriceGBP > 0 && $quoteGBP > 0) {
                return number_format((100 * ($discountPriceGBP - $subtotalGBP)) / $discountPriceGBP, 1);
            }
        }
        return '0.0';
    }
}

if (!function_exists('discountPriceRaw')) {
    function discountPriceRaw(array $price)
    {
        $quote = (float)($price['quote'] ?? 0);
        $custDiscount = (float)($price['cust_discount'] ?? 0);
        if($quote && array_key_exists('cust_discount', $price) && $custDiscount)
            return $quote * ((100 - $custDiscount) / 100);
        elseif($quote)
            return $quote;
        return null;
    }
}

if (!function_exists('discountPrice')) {
    function discountPrice(array $price)
    {
        $rawValue = discountPriceRaw($price);
        return $rawValue !== null ? number_format($rawValue, 2) : '';
    }
}

if (!function_exists('discountPriceGBPRaw')) {
    function discountPriceGBPRaw(array $price)
    {
        $quoteGBP = (float)calculateQuoteGBP($price);
        $custDiscount = (float)($price['cust_discount'] ?? 0);
        if($quoteGBP && array_key_exists('cust_discount', $price) && $custDiscount)
            return $quoteGBP * ((100 - $custDiscount) / 100);
        elseif($quoteGBP)
            return $quoteGBP;
        return null;
    }
}

if (!function_exists('discountPriceGBP')) {
    function discountPriceGBP(array $price)
    {
        $rawValue = discountPriceGBPRaw($price);
        return $rawValue !== null ? number_format($rawValue, 2) : '';
    }
}





if (!function_exists('convertCurrency')) {
    function convertCurrency($currencyIn, $currencyOut, $value, $season, $rate = 0)
    {
        // If no rate is provided, and season is an ID, find the season.
        if (!$rate && is_int($season)) {
            $season = Seasons::allCached()->where('id', $season)->first();
        }

        // Check if required data is present.
        if ($currencyIn !== null && $currencyOut !== null && $value !== null && $value != 0 && $value !== '') {
            // If no rate is provided, use season rates.
            $euroRate = $rate ?: ($season->euro_rate ?? 1);
            $usdRate = $rate ?: ($season->usd_rate ?? 1);

            switch ($currencyOut) {
                case '£':
                    switch ($currencyIn) {
                        case '£':
                            return $value;
                        case '€':
                            return $value / $euroRate;
                        case '$':
                            return $value / $usdRate;
                    }
                case '€':
                    switch ($currencyIn) {
                        case '£':
                            return $value * $euroRate;
                        case '€':
                            return $value;
                        case '$':
                            return $value * $euroRate / $usdRate;
                    }
                case '$':
                    switch ($currencyIn) {
                        case '£':
                            return $value * $usdRate;
                        case '€':
                            return $value * $usdRate / $euroRate;
                        case '$':
                            return $value;
                    }
            }
        }

        // Return 0 if any required data is missing or invalid.
        return 0;
    }
}




if (!function_exists('getFirstNonEmpty')) {
    function getFirstNonEmpty(...$values) {
        foreach ($values as $value) {
            if ($value !== null && $value !== '') {
                return $value;
            }
        }
        return null;
    }
}



if (!function_exists('arrayToCsv')) {
    function arrayToCsv(array $data): string {
        // Ensure there is data
        if (empty($data)) {
            return '';
        }

        // Open a memory stream for writing CSV
        $output = fopen('php://memory', 'r+');

        // Write the headings (keys from the first array element)
        fputcsv($output, array_keys($data[0]));

        // Write each data row
        foreach ($data as $row) {
            fputcsv($output, $row);
        }

        // Rewind the memory stream and fetch the CSV as a string
        rewind($output);
        $csv = stream_get_contents($output);

        // Close the memory stream
        fclose($output);

        return $csv;
    }
}


if (!function_exists('getSizeName')) {
    function getSizeName($id) {
        return Sizes::allCached()->find($id)->name;
    }
}



if (!function_exists('humanizeAttribute')) {
    function humanizeAttribute($attribute) {
        $parts = explode('.', $attribute);
        $result = '';
        foreach ($parts as $i => $part) {
            if (is_numeric($part)) {
                // For numeric parts, append them in parentheses.
                $result .= " (" . $part . ")";
            } else {
                // For the first segment, simply title-case it.
                // For subsequent segments, prepend an arrow.
                $formatted = ucwords(str_replace('_', ' ', $part));
                $result .= $i === 0 ? $formatted : " -> " . $formatted;
            }
        }
        return $result;
    }
}
