<?php

namespace App\Services;

use App\Facades\ZohoFacade;
use App\Services\ZohoEUProvider;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Asad\OAuth2\Client\Provider\Zoho;
use Illuminate\Support\Facades\Cache;
use App\Exceptions\ZohoAuthRequiredException;
use League\OAuth2\Client\Token\AccessTokenInterface;

class ZohoService
{

    protected ZohoEUProvider $provider;

    public function __construct()
    {
        $this->provider = new ZohoEUProvider([
            'clientId'     => config('services.zoho.client_id'),
            'clientSecret' => config('services.zoho.client_secret'),
            'redirectUri'  => route('zoho.callback', absolute: true),
        ]);
    }

    protected function getValidAccessToken(): ?AccessTokenInterface
    {
        $user = Auth::user();
        $token = $this->getAccessTokenFromUser($user);

        if (!$token || !$token->getRefreshToken()) {
            // Token is unusable — either missing or not refreshable
            return null;
        }

        if ($token->hasExpired()) {
            $newToken = ZohoFacade::getAccessToken('refresh_token', [
                'refresh_token' => $token->getRefreshToken(),
            ]);

            $user->zoho_access_token = serialize($newToken);
            $user->save();

            return $newToken;
        }

        return $token;
    }

    public function isConnected(): bool
    {
        return $this->getValidAccessToken() !== null;
    }

    public function getAccessToken(string $grant = 'authorization_code', array $params = []): AccessTokenInterface
    {
        return $this->provider->getAccessToken($grant, $params);
    }

    protected function getAccessTokenFromUser($user): ?AccessTokenInterface
    {
        if (!$user || !$user->zoho_access_token) {
            return null;
        }

        return unserialize($user->zoho_access_token);
    }

    public function createInvoice(array $payload, string $orgId): ?array
    {
        $token = $this->getValidAccessToken();
        if (!$token) {
            Log::warning('Zoho invoice creation aborted: no valid access token.');
            return null;             // Livewire can show "Connect to Zoho" button
        }

        $orgId = $orgId ?? env('ZOHO_ORG_ID');

        $url = "https://www.zohoapis.eu/books/v3/invoices?organization_id={$orgId}";
        // Ensure Carbon date is properly formatted
        if (!empty($payload['date']) && $payload['date'] instanceof \Carbon\Carbon) {
            $payload['date'] = $payload['date']->toDateString(); // yyyy-mm-dd
        }

        // Log the payload for debugging
        Log::info('Zoho API – creating invoice', [
            'customer_id' => $payload['customer_id'] ?? null,
            'template_id' => $payload['template_id'] ?? null,
            'reference_number' => $payload['reference_number'] ?? null,
            'line_items_count' => count($payload['line_items'] ?? []),
            'first_line_item' => !empty($payload['line_items']) ? [
                'name' => $payload['line_items'][0]['name'] ?? null,
                'account_id' => $payload['line_items'][0]['account_id'] ?? null,
                'tax_id' => $payload['line_items'][0]['tax_id'] ?? null,
                'tags_count' => count($payload['line_items'][0]['tags'] ?? []),
            ] : null,
        ]);

        $response = Http::withToken($token->getToken())
            ->withHeaders(['Content-Type' => 'application/json'])
            ->post($url, $payload);

        if (!$response->successful()) {
            $errorBody = $response->json();
            $errorMessage = $errorBody['message'] ?? $response->body();
            
            Log::error('Zoho API – invoice creation failed', [
                'status' => $response->status(),
                'body'   => $response->body(),
                'message' => $errorMessage,
                'payload_summary' => [
                    'customer_id' => $payload['customer_id'] ?? null,
                    'line_items' => collect($payload['line_items'] ?? [])->map(fn($li) => [
                        'account_id' => $li['account_id'] ?? null,
                        'tax_id' => $li['tax_id'] ?? null,
                    ])->toArray(),
                ],
            ]);
            
            throw new \Exception('Zoho API Error: ' . $errorMessage);
        }

        return $response->json('invoice');
    }

    /**
     * Create a contact (customer) in Zoho Books
     * 
     * @param array $payload Contact data
     * @param string $orgId Organization ID
     * @return array|null Created contact data or null if failed
     */
    public function createContact(array $payload, string $orgId): ?array
    {
        $token = $this->getValidAccessToken();
        if (!$token) {
            Log::warning('Zoho contact creation aborted: no valid access token.');
            return null;
        }

        $url = "https://www.zohoapis.eu/books/v3/contacts?organization_id={$orgId}";

        $response = Http::withToken($token->getToken())
            ->withHeaders(['Content-Type' => 'application/json'])
            ->post($url, $payload);

        if (!$response->successful()) {
            $errorBody = $response->json();
            $errorMessage = $errorBody['message'] ?? $response->body();
            
            Log::error('Zoho API – contact creation failed', [
                'status' => $response->status(),
                'body'   => $response->body(),
                'message' => $errorMessage,
            ]);
            
            throw new \Exception('Zoho API Error: ' . $errorMessage);
        }

        return $response->json('contact');
    }

    public function getAuthUrl(): string
    {
        return $this->provider->getAuthorizationUrl([
            'scope'           => [
                'ZohoBooks.invoices.CREATE',
                'ZohoBooks.invoices.READ',
                'ZohoBooks.contacts.CREATE',
                'ZohoBooks.contacts.READ',
                'ZohoBooks.accountants.READ',
                'ZohoBooks.settings.READ',
                'ZohoBooks.vendorpayments.READ',
            ],
            'access_type'     => 'offline',
            'prompt'          => 'consent',
            'approval_prompt' => null,
        ]);
    }

    public function getState(): string
    {
        return $this->provider->getState();
    }

    public function getProvider(): BaseZoho
    {
        return $this->provider;
    }

    public function clearCache(): void
    {
        Cache::tags('zoho')->flush();
    }

    /* ---------------------------------------------------------------------
    |  ORGANISATIONS  ( GET /organizations )
    |--------------------------------------------------------------------*/


    public function getOrganisations(int $cacheTtl = 604_800): Collection
    {
        /** tag-based cache so you can wipe everything with Cache::tags('zoho')->flush() */
        return Cache::tags(['zoho'])->remember(
            key: 'zoho-orgs',
            ttl: $cacheTtl,                       // default = 7 days
            callback: function (): Collection {
                $token = $this->getValidAccessToken();
                if (!$token) {
                    return collect();             // Livewire can show “Connect to Zoho” button
                }

                $url   = 'https://www.zohoapis.eu/books/v3/organizations';
                $resp  = Http::withToken($token->getToken())->get($url);

                if (!$resp->successful()) {
                    Log::error('Zoho API – organisations fetch failed', [
                        'status' => $resp->status(),
                        'body'   => $resp->body(),
                    ]);
                    return collect();
                }

                return collect($resp->json('organizations') ?? []);
            }
        );
    }

    public function getCustomers(string $orgId, int $perPage = 200): Collection
    {
        /** ──────────────────────────────────────────────────────────────
         *  Cache for 7 days (604 800 s) under the “zoho” tag so you can
         *  do `Cache::tags('zoho')->flush()` whenever you want to clear
         *  every Zoho-related result in one go.
         * ──────────────────────────────────────────────────────────────*/
        return Cache::tags(['zoho'])->remember(
            key: "zoho-customers-{$orgId}-{$perPage}",
            ttl: 604_800,
            callback: function () use ($orgId, $perPage): Collection {
                $token = $this->getValidAccessToken();
                if (!$token) {
                    return collect();      // livewire will show “Connect” button
                }

                $all  = collect();
                $page = 1;

                while (true) {
                    $url = sprintf(
                        'https://www.zohoapis.eu/books/v3/contacts?organization_id=%s&contact_type=customer&page=%d&per_page=%d',
                        $orgId,
                        $page,
                        $perPage
                    );

                    $resp = Http::withToken($token->getToken())->get($url);

                    if (!$resp->successful()) {
                        Log::error('Zoho API – customers fetch failed', [
                            'status' => $resp->status(),
                            'body'   => $resp->body(),
                        ]);
                        break;
                    }

                    $customers = collect($resp->json('contacts'))
                        ->where('contact_type', 'customer');

                    $all = $all->merge($customers);

                    if ($customers->count() < $perPage) {
                        break;          // no more pages
                    }
                    $page++;
                }

                return $all;
            }
        );
    }

    /* ---------------------------------------------------------------------
    |  INVOICE CUSTOM FIELDS
    |--------------------------------------------------------------------*/

    /** 31-day cache (2 678 400 s) of every reporting tag */
    public function getCustomFields(string $orgId): Collection
    {
        return Cache::tags(['zoho'])->remember(
            "zoho-custom-fields-{$orgId}",
            2_678_400,
            function () use ($orgId): Collection {
                $token = $this->getValidAccessToken();
                if (!$token) {
                    return collect();
                }

                $url = "https://www.zohoapis.eu/books/v3/settings/customfields?organization_id={$orgId}";

                $resp = Http::withToken($token->getToken())->get($url);

                if (!$resp->successful()) {
                    Log::error('Zoho API – custom-fields failed', [
                        'status' => $resp->status(),
                        'body'   => $resp->body(),
                    ]);
                    return collect();
                }
                // dd($resp->json('customfields'));
                return collect($resp->json('customfields'));
            }
        );
    }

    /* ---------------------------------------------------------------------
    |  REPORTING TAGS
    |--------------------------------------------------------------------*/

    /** 31-day cache (2 678 400 s) of every reporting tag */
    public function getReportingTags(string $orgId): Collection
    {
        return Cache::tags(['zoho'])->remember(
            "zoho-reporting-tags-{$orgId}",
            2_678_400,
            function () use ($orgId): Collection {
                $token = $this->getValidAccessToken();
                if (!$token) {
                    Log::warning('getReportingTags: No valid access token');
                    return collect();
                }

                $url = "https://www.zohoapis.eu/books/v3/settings/tags?organization_id={$orgId}";

                $resp = Http::withToken($token->getToken())->get($url);
                if (!$resp->successful()) {
                    Log::error('Zoho API – reporting-tags failed', [
                        'status' => $resp->status(),
                        'body'   => $resp->body(),
                    ]);
                    return collect();
                }

                $tags = collect($resp->json('reporting_tags'));
                Log::debug('Zoho API – reporting-tags response', [
                    'status' => $resp->status(),
                    'tags_count' => $tags->count(),
                    'response_keys' => array_keys($resp->json() ?? []),
                    'sample_response' => substr($resp->body(), 0, 500),
                ]);

                return $tags;
            }
        );
    }

    /* ---------------------------------------------------------------------
    |  REPORTING-TAG OPTIONS  ( GET /settings/tags/{tagId} )
    |--------------------------------------------------------------------*/

    public function getReportingTagOptions(string $orgId, string $tagId): Collection
    {
        /** 31-day cache, tagged "zoho" so it can be wiped with Cache::tags('zoho')->flush() */
        return Cache::tags(['zoho'])->remember(
            key: "zoho-tagopts-2-{$orgId}-{$tagId}",
            ttl: 2_678_400,         // 31 days
            callback: function () use ($orgId, $tagId): Collection {
                $token = $this->getValidAccessToken();
                if (!$token) {
                    return collect();        // caller can decide to show “Connect” button
                }

                $url  = "https://www.zohoapis.eu/books/v3/settings/tags/{$tagId}?organization_id={$orgId}";
                $resp = Http::withToken($token->getToken())->get($url);

                if (!$resp->successful()) {
                    Log::error('Zoho API – tag options fetch failed', [
                        'status' => $resp->status(),
                        'body'   => $resp->body(),
                    ]);
                    return collect();
                }

                return collect($resp->json('reporting_tag.tag_options') ?? []);
            }
        );
    }

    /* ---------------------------------------------------------------------
    |  CHART-OF-ACCOUNTS   ( GET /chartofaccounts )
    |--------------------------------------------------------------------*/

    public function getChartOfAccounts(
        string $orgId,
        array  $query   = [],   // e.g. ['filter_by' => 'AccountType.Active']
        int    $perPage = 200   // Zoho max page size
    ): Collection {

        /** 31-day cache keyed by org + filter signature */
        $cacheKey = 'zoho-coa-' . $orgId . '-' . md5(json_encode($query) . $perPage);

        return Cache::tags(['zoho'])->remember(
            key: $cacheKey,
            ttl: 2_678_400,                 // 31 days
            callback: function () use ($orgId, $query, $perPage): Collection {

                $token = $this->getValidAccessToken();
                if (!$token) {
                    return collect();       // caller decides what to do (e.g. show “Connect”)
                }

                $all  = collect();
                $page = 1;

                while (true) {
                    $q = array_merge(
                        $query,
                        ['organization_id' => $orgId, 'page' => $page, 'per_page' => $perPage]
                    );

                    $url  = 'https://www.zohoapis.eu/books/v3/chartofaccounts?' . http_build_query($q);
                    $resp = Http::withToken($token->getToken())->get($url);

                    if (!$resp->successful()) {
                        Log::error('Zoho API – chart-of-accounts fetch failed', [
                            'status' => $resp->status(),
                            'body'   => $resp->body(),
                        ]);
                        break;
                    }

                    $pageData = collect($resp->json('chartofaccounts') ?? []);
                    $all      = $all->merge($pageData);

                    if ($pageData->count() < $perPage) {
                        break;              // last page reached
                    }
                    $page++;
                }

                return $all->sortBy('account_code');                // Collection of accounts
            }
        );
    }

    /* ---------------------------------------------------------------------
    |  TAXES / VAT  (GET /settings/taxes)
    |--------------------------------------------------------------------*/

    public function getTaxes(string $orgId, int $perPage = 200): Collection
    {
        /** 31-day cache keyed by org-id */
        $cacheKey = "zoho-taxes-{$orgId}-{$perPage}";

        return Cache::tags(['zoho'])->remember(
            $cacheKey,
            2_678_400,                                // 31 days
            function () use ($orgId, $perPage): Collection {

                $token = $this->getValidAccessToken();
                if (!$token) {
                    return collect();                 // caller can prompt for OAuth
                }

                $all  = collect();
                $page = 1;

                while (true) {
                    $url = sprintf(
                        'https://www.zohoapis.eu/books/v3/settings/taxes?organization_id=%s&page=%d&per_page=%d',
                        $orgId,
                        $page,
                        $perPage
                    );

                    $resp = Http::withToken($token->getToken())->get($url);

                    if (!$resp->successful()) {
                        Log::error('Zoho API – taxes fetch failed', [
                            'status' => $resp->status(),
                            'body'   => $resp->body(),
                        ]);
                        break;
                    }

                    $pageTaxes = collect($resp->json('taxes') ?? []);
                    $all       = $all->merge($pageTaxes);

                    if ($pageTaxes->count() < $perPage) {
                        break;                        // last page
                    }
                    $page++;
                }

                return $all;                          // Collection of tax rows
            }
        );
    }

    /* ---------------------------------------------------------------------
    |  ADDRESSES   (/contacts/{id}/address)
    |--------------------------------------------------------------------*/
    public function getAddresses(string $orgId, string $contactId): Collection
    {
        $cacheKey = "zoho-addresses-{$orgId}-{$contactId}";

        return Cache::tags(['zoho'])->remember(
            $cacheKey,
            2_678_400, // 31 days
            function () use ($orgId, $contactId): Collection {
                $token = $this->getValidAccessToken();
                if (!$token) {
                    return collect(); // empty collection instead of null
                }

                $url = "https://www.zohoapis.eu/books/v3/contacts/{$contactId}/address?organization_id={$orgId}";
                $resp = Http::withToken($token->getToken())->get($url);

                if (!$resp->successful()) {
                    Log::error('Zoho API – addresses fetch failed', [
                        'status' => $resp->status(),
                        'body'   => $resp->body(),
                    ]);
                    return collect();
                }

                return collect($resp->json('addresses') ?? []);
            }
        );
    }


    /* ---------------------------------------------------------------------
    |  CHART OF ACCOUNTS   (account_code → account_id)
    |--------------------------------------------------------------------*/

    public function getAccountId(string $orgId, string $accountCode): ?string
    {
        $cacheKey = "zoho-account-{$orgId}-" . md5($accountCode);

        return Cache::tags(['zoho'])->remember(
            $cacheKey,
            2_678_400,
            function () use ($orgId, $accountCode): ?string {
                $token = $this->getValidAccessToken();
                if (!$token) {
                    return null;
                }

                $url  = "https://www.zohoapis.eu/books/v3/chartofaccounts?organization_id={$orgId}";
                $resp = Http::withToken($token->getToken())->get($url);

                if (!$resp->successful()) {
                    Log::error('Zoho API – accounts failed', [
                        'status' => $resp->status(),
                        'body'   => $resp->body(),
                    ]);
                    return null;
                }

                return collect($resp->json('chartofaccounts'))
                        ->firstWhere('account_code', $accountCode)['account_id'] ?? null;
            }
        );
    }

    /* ---------------------------------------------------------------------
    |  INVOICE TEMPLATES  ( GET /invoices/templates )
    |--------------------------------------------------------------------*/

    public function getInvoiceTemplates(string $orgId): Collection
    {
        /** 31-day cache, tagged “zoho” so you can flush everything with one call. */
        return Cache::tags(['zoho'])->remember(
            key: "zoho-templates-{$orgId}",
            ttl: 2_678_400,                // 31 days
            callback: function () use ($orgId): Collection {
                $token = $this->getValidAccessToken();
                if (!$token) {
                    return collect();       // caller shows “Connect to Zoho” button
                }

                $url  = "https://www.zohoapis.eu/books/v3/invoices/templates?organization_id={$orgId}";
                $resp = Http::withToken($token->getToken())->get($url);

                if (!$resp->successful()) {
                    Log::error('Zoho API – templates fetch failed', [
                        'status' => $resp->status(),
                        'body'   => $resp->body(),
                    ]);
                    return collect();
                }

                /** return full template records; filter/map in the component if needed */
                return collect($resp->json('templates') ?? []);
            }
        );
    }

    /* ---------------------------------------------------------------------
    |  TAX – “Standard Rate” (single call, cached)
    |--------------------------------------------------------------------*/

    public function getStandardRateTax(string $orgId): ?array
    {
        return Cache::tags(['zoho'])->remember(
            "zoho-tax-standard-{$orgId}",
            2_678_400,
            function () use ($orgId): ?array {
                $token = $this->getValidAccessToken();
                if (!$token) {
                    return null;
                }

                $url  = "https://www.zohoapis.eu/books/v3/settings/taxes?organization_id={$orgId}";
                $resp = Http::withToken($token->getToken())->get($url);

                if (!$resp->successful()) {
                    Log::error('Zoho API – taxes failed', [
                        'status' => $resp->status(),
                        'body'   => $resp->body(),
                    ]);
                    return null;
                }

                return collect($resp->json('taxes'))
                        ->firstWhere('tax_name', 'Standard Rate');
            }
        );
    }

    /* ---------------------------------------------------------------------
    |  VENDOR PAYMENTS  ( GET /vendorpayments )
    |--------------------------------------------------------------------*/

    /**
     * Get vendor payments within a date range
     * 
     * @param string $orgId Organization ID
     * @param string $dateStart Start date (Y-m-d format)
     * @param string $dateEnd End date (Y-m-d format)
     * @return Collection
     */
    public function getVendorPayments(string $orgId, string $dateStart, string $dateEnd): Collection
    {
        $token = $this->getValidAccessToken();
        if (!$token) {
            Log::warning('getVendorPayments: No valid access token');
            return collect();
        }

        $url = sprintf(
            'https://www.zohoapis.eu/books/v3/vendorpayments?organization_id=%s&date_start=%s&date_end=%s',
            $orgId,
            $dateStart,
            $dateEnd
        );

        $resp = Http::withToken($token->getToken())->get($url);

        if (!$resp->successful()) {
            Log::error('Zoho API – vendor payments fetch failed', [
                'status' => $resp->status(),
                'body'   => $resp->body(),
            ]);
            return collect();
        }

        return collect($resp->json('vendorpayments') ?? []);
    }

    /* ---------------------------------------------------------------------
    |  VENDOR / CONTACT DETAILS  ( GET /contacts/{id} )
    |--------------------------------------------------------------------*/

    /**
     * Get vendor details by contact ID
     * 
     * @param string $orgId Organization ID
     * @param string $contactId Contact/Vendor ID
     * @return array|null
     */
    public function getVendorById(string $orgId, string $contactId): ?array
    {
        $token = $this->getValidAccessToken();
        if (!$token) {
            Log::warning('getVendorById: No valid access token');
            return null;
        }

        $url = sprintf(
            'https://www.zohoapis.eu/books/v3/contacts/%s?organization_id=%s',
            $contactId,
            $orgId
        );

        $resp = Http::withToken($token->getToken())->get($url);

        if (!$resp->successful()) {
            Log::error('Zoho API – vendor fetch failed', [
                'status' => $resp->status(),
                'body'   => $resp->body(),
                'contact_id' => $contactId,
            ]);
            return null;
        }

        return $resp->json('contact');
    }
}
