<?php

namespace App\Services;

use Google\Cloud\AIPlatform\V1\PredictRequest;
use Google\Cloud\AIPlatform\V1\PredictionServiceClient;
use Google\Cloud\AIPlatform\V1\Content;
use Google\Cloud\AIPlatform\V1\Part;
use Google\Cloud\AIPlatform\V1\Blob;
use Google\Cloud\AIPlatform\V1\FileData;
use Google\Cloud\AIPlatform\V1\GenerateContentRequest;
use Google\Cloud\AIPlatform\V1\GenerationConfig;
use Google\Protobuf\Value;
use Google\Protobuf\Struct;
use Google\Protobuf\ListValue;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;

class VertexAiService
{
    protected string $projectId;
    protected string $location;
    protected ?string $apiKey;
    protected Client $httpClient;

    public function __construct()
    {
        $this->projectId = config('services.vertex_ai.project_id', 'your-project-id');
        $this->location = config('services.vertex_ai.location', 'us-central1');
        $this->apiKey = config('services.vertex_ai.api_key');
        
        if (empty($this->apiKey)) {
            throw new \RuntimeException('Vertex AI API key is not configured. Please set VERTEX_AI_API_KEY in your .env file.');
        }
        
        $this->httpClient = new Client([
            'base_uri' => "https://{$this->location}-aiplatform.googleapis.com",
            'timeout' => 300, // 5 minutes total timeout
            'connect_timeout' => 30, // 30 seconds to establish connection
            'read_timeout' => 300, // 5 minutes to read response
        ]);
    }

    /**
     * Extract structured data from a file using Gemini 2.5 Pro
     * 
     * @param string $filePath Path to the file
     * @param string $prompt The extraction prompt
     * @param string $mimeType MIME type of the file
     * @param int $maxRetries Maximum number of retries
     * @return array Parsed JSON response
     */
    public function extractFromFile(string $filePath, string $prompt, string $mimeType, int $maxRetries = 3): ?array
    {
        if (!file_exists($filePath)) {
            throw new \InvalidArgumentException("File not found: {$filePath}");
        }

        $fileContent = base64_encode(file_get_contents($filePath));
        
        $attempts = 0;
        $result = null;

        while (!$result && $attempts < $maxRetries) {
            try {
                $attempts++;

                $response = $this->generateContent($prompt, $fileContent, $mimeType);
                
                if (!$response) {
                    continue;
                }

                // Clean and parse JSON response
                $cleanedResponse = $this->cleanJsonResponse($response);
                $result = json_decode($cleanedResponse, true);

                if (json_last_error() !== JSON_ERROR_NONE) {
                    $result = null;
                    continue;
                }

            } catch (\Exception $e) {
                Log::error("Vertex AI - Extraction error", [
                    'error' => $e->getMessage()
                ]);
                
                if ($attempts >= $maxRetries) {
                    throw $e;
                }
            }
        }

        return $result;
    }

    /**
     * Generate text-only content using Vertex AI Gemini API via REST
     * 
     * @param string $prompt The text prompt
     * @param string $model The model to use (default: gemini-2.5-flash-lite)
     * @param int $maxRetries Maximum number of retries
     * @return array|null Parsed JSON response
     */
    public function generateText(string $prompt, string $model = 'gemini-2.5-flash-lite', int $maxRetries = 3): ?array
    {
        $attempts = 0;
        $result = null;

        while (!$result && $attempts < $maxRetries) {
            try {
                $attempts++;

                $response = $this->generateTextContent($prompt, $model);
                
                if (!$response) {
                    continue;
                }

                // Clean and parse JSON response
                $cleanedResponse = $this->cleanJsonResponse($response);
                $result = json_decode($cleanedResponse, true);

                if (json_last_error() !== JSON_ERROR_NONE) {
                    $result = null;
                    continue;
                }

            } catch (\Exception $e) {
                Log::error("Vertex AI - Text generation error", [
                    'error' => $e->getMessage()
                ]);
                
                if ($attempts >= $maxRetries) {
                    throw $e;
                }
            }
        }

        return $result;
    }

    /**
     * Generate text-only content using Vertex AI Gemini API via REST (internal)
     */
    protected function generateTextContent(string $prompt, string $model = 'gemini-2.5-flash-lite'): ?string
    {
        $endpoint = "/v1/projects/{$this->projectId}/locations/{$this->location}/publishers/google/models/{$model}:generateContent";

        $payload = [
            'contents' => [
                [
                    'role' => 'user',
                    'parts' => [
                        [
                            'text' => $prompt
                        ]
                    ]
                ]
            ],
            'generation_config' => [
                'temperature' => 0.1,
                'max_output_tokens' => 8192,
                'response_mime_type' => 'application/json'
            ]
        ];

        try {
            $response = $this->httpClient->post($endpoint, [
                'headers' => [
                    'x-goog-api-key' => $this->apiKey,
                    'Content-Type' => 'application/json',
                ],
                'json' => $payload,
            ]);
            $body = json_decode($response->getBody()->getContents(), true);
            
            // Extract text from Vertex AI response structure
            if (isset($body['candidates'][0]['content']['parts'][0]['text'])) {
                $text = $body['candidates'][0]['content']['parts'][0]['text'];
                return $text;
            }

            return null;

        } catch (\GuzzleHttp\Exception\RequestException $e) {
            Log::error('Vertex AI - HTTP request failed', [
                'error' => $e->getMessage(),
                'status_code' => $e->hasResponse() ? $e->getResponse()->getStatusCode() : null
            ]);
            throw $e;
        } catch (\Exception $e) {
            Log::error('Vertex AI - API call failed', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Extract data from multiple inline files using Vertex AI
     * 
     * @param string $prompt The extraction prompt
     * @param array $files Array of [base64Content, mimeType] pairs
     * @param int $maxRetries Maximum number of retries
     * @return array|null Parsed JSON response
     */
    public function extractFromMultipleFiles(string $prompt, array $files, int $maxRetries = 3): ?array
    {
        $attempts = 0;
        $result = null;

        while (!$result && $attempts < $maxRetries) {
            try {
                $attempts++;

                $response = $this->generateContentMultiFile($prompt, $files);
                
                if (!$response) {
                    continue;
                }

                // Clean and parse JSON response
                $cleanedResponse = $this->cleanJsonResponse($response);
                $result = json_decode($cleanedResponse, true);

                if (json_last_error() !== JSON_ERROR_NONE) {
                    $result = null;
                    continue;
                }

            } catch (\Exception $e) {
                Log::error("Vertex AI - Multi-file extraction error", [
                    'error' => $e->getMessage()
                ]);
                
                if ($attempts >= $maxRetries) {
                    throw $e;
                }
            }
        }

        return $result;
    }

    /**
     * Generate content from multiple inline files using Vertex AI Gemini API via REST
     */
    protected function generateContentMultiFile(string $prompt, array $files): ?string
    {
        $model = config('services.vertex_ai.primary_model', 'gemini-2.5-pro');
        $endpoint = "/v1/projects/{$this->projectId}/locations/{$this->location}/publishers/google/models/{$model}:generateContent";

        // Build parts array with prompt and all files
        $parts = [
            ['text' => $prompt]
        ];

        foreach ($files as $file) {
            $parts[] = [
                'inline_data' => [
                    'mime_type' => $file['mime_type'],
                    'data' => $file['data']
                ]
            ];
        }

        $payload = [
            'contents' => [
                [
                    'role' => 'user',
                    'parts' => $parts
                ]
            ],
            'generation_config' => [
                'temperature' => 0.1,
                'max_output_tokens' => 65536,
                'response_mime_type' => 'application/json'
            ]
        ];

        try {
            $response = $this->httpClient->post($endpoint, [
                'headers' => [
                    'x-goog-api-key' => $this->apiKey,
                    'Content-Type' => 'application/json',
                ],
                'json' => $payload,
            ]);

            $body = json_decode($response->getBody()->getContents(), true);
            
            // Extract text from Vertex AI response structure
            if (isset($body['candidates'][0]['content']['parts'][0]['text'])) {
                return $body['candidates'][0]['content']['parts'][0]['text'];
            }

            return null;

        } catch (\GuzzleHttp\Exception\RequestException $e) {
            Log::error('Vertex AI - HTTP request failed', [
                'error' => $e->getMessage(),
                'status_code' => $e->hasResponse() ? $e->getResponse()->getStatusCode() : null
            ]);
            throw $e;
        } catch (\Exception $e) {
            Log::error('Vertex AI - API call failed', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Generate content using Vertex AI Gemini API via REST
     */
    protected function generateContent(string $prompt, string $fileContentBase64, string $mimeType): ?string
    {
        $model = config('services.vertex_ai.primary_model', 'gemini-3.0-pro');
        $endpoint = "/v1/projects/{$this->projectId}/locations/{$this->location}/publishers/google/models/{$model}:generateContent";

        $payload = [
            'contents' => [
                [
                    'role' => 'user',
                    'parts' => [
                        [
                            'text' => $prompt
                        ],
                        [
                            'inline_data' => [
                                'mime_type' => $mimeType,
                                'data' => $fileContentBase64
                            ]
                        ]
                    ]
                ]
            ],
            'generation_config' => [
                'temperature' => 0.1,
                'max_output_tokens' => 65536, // Increased from 8192 to handle larger files
                'response_mime_type' => 'application/json'
            ]
        ];

        try {
            $response = $this->httpClient->post($endpoint, [
                'headers' => [
                    'x-goog-api-key' => $this->apiKey,
                    'Content-Type' => 'application/json',
                ],
                'json' => $payload,
            ]);

            $body = json_decode($response->getBody()->getContents(), true);
            
            // Extract text from Vertex AI response structure
            if (isset($body['candidates'][0]['content']['parts'][0]['text'])) {
                $text = $body['candidates'][0]['content']['parts'][0]['text'];
                return $text;
            }

            return null;

        } catch (\GuzzleHttp\Exception\RequestException $e) {
            Log::error('Vertex AI - HTTP request failed', [
                'error' => $e->getMessage(),
                'status_code' => $e->hasResponse() ? $e->getResponse()->getStatusCode() : null
            ]);
            throw $e;
        } catch (\Exception $e) {
            Log::error('Vertex AI - API call failed', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Clean JSON response from markdown code blocks and extra formatting
     */
    protected function cleanJsonResponse(string $response): string
    {
        // Remove markdown code blocks
        $response = preg_replace('/```json\s*/i', '', $response);
        $response = preg_replace('/```\s*$/', '', $response);
        $response = str_replace(['```json', '```'], '', $response);
        
        return trim($response);
    }

    /**
     * Get MIME type for common file extensions
     */
    public static function getMimeType(string $extension): string
    {
        $mimeTypes = [
            'pdf' => 'application/pdf',
            'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'xls' => 'application/vnd.ms-excel',
            'csv' => 'text/csv',
            'txt' => 'text/plain',
            'png' => 'image/png',
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
        ];

        $extension = strtolower($extension);
        return $mimeTypes[$extension] ?? 'application/octet-stream';
    }
}

