<?php

namespace App\Services;

use App\Models\Price;
use Illuminate\Database\Eloquent\Builder;

class PriceFinder
{
    /**
     * Build a subquery for the best matching price details.
     *
     * Returns a JSON object containing 'quote', 'cmt' and 'cost'.
     * Only the style version column is required. The other columns are optional.
     *
     * @param  string      $styleVersionColumn  Outer query column for style_versions_id.
     * @param  string|null $colourwaysColumn    Outer query column for colourways_id.
     * @param  string|null $colourTypeColumn    Outer query column for colour_type.
     * @param  string|null $phaseColumn         Outer query column for phase_id.
     * @param  string|null $sizesColumn         Outer query column for sizes_id.
     * @return Builder
     */
    public static function priceSubQuery(
        string $styleVersionColumn,
        ?string $colourwaysColumn = null,
        ?string $colourTypeColumn = null,
        ?string $phaseColumn = null,
        ?string $sizesColumn = null
    ): Builder {
        // Select a JSON object containing the three desired fields.
        $query = Price::selectRaw("JSON_OBJECT(
            'id', prices.id,
            'model', prices.model,
            'yarn_trans', prices.yarn_trans,
            'yarn_trans_currency', prices.yarn_trans_currency,
            'cmt', prices.cmt,
            'gmt_trans', prices.gmt_trans,
            'gmt_trans_currency', prices.gmt_trans_currency,
            'aoc', prices.aoc,
            'weight', prices.weight,
            'yarn', prices.yarn,
            'yarn_currency', prices.yarn_currency,
            'accessory', prices.accessory,
            'accessory_currency', prices.accessory_currency,
            'embroidery', prices.embroidery,
            'embroidery_currency', prices.embroidery_currency,
            'costed_weight', prices.costed_weight,
            'min_qty', prices.min_qty,
            'label', prices.label,
            'label_currency', prices.label_currency,
            'yarn_value', prices.yarn_value,
            'yarn_value_currency', prices.yarn_value_currency,
            'testing', prices.testing,
            'testing_courier', prices.testing_courier,
            'cmt_status', prices.cmt_status,
            'quote_status', prices.quote_status,
            'valid_until', prices.valid_until,
            'quote', prices.quote,
            'discount', prices.discount,
            'duty', prices.duty,
            'yarnkg', prices.yarnkg,
            'yarnkg_currency', prices.yarnkg_currency,
            'fact_currency', f.currency,
            'cust_currency', c.currency,
            'seasons_id', s.seasons_id,
            'customer_customs', c.customs,
            'factory_customs', f.customs,
            'cust_terms', (SELECT
                    JSON_ARRAYAGG(
                        JSON_OBJECT(
                            'percentage', percentage,
                            'days', days
                        )
                    )
                FROM customer_payment_terms
                WHERE customer_payment_terms.customer_id = customers.id
            ),
            -- yarn_cost_kg_eur removed - will be computed separately when needed
        ) as price_data")

        ->join('style_versions as sv', 'sv.id', '=', 'prices.style_versions_id')
        ->join('suppliers as f', 'f.id', '=', 'sv.factory_id')
        ->join('styles as s', 's.id', '=', 'sv.styles_id')
        ->join('customers as c', 'c.id', '=', 's.customers_id')
        ->whereColumn('prices.style_versions_id', $styleVersionColumn)
        ->whereNull('prices.deleted_at');

        if ($colourwaysColumn !== null) {
            $query->where(function (Builder $query) use ($colourwaysColumn) {
                $query->whereColumn('colourways_id', $colourwaysColumn)
                      ->orWhereNull('colourways_id');
            });
        }

        if ($colourTypeColumn !== null) {
            $query->where(function (Builder $query) use ($colourTypeColumn) {
                $query->whereColumn('colour_type', $colourTypeColumn)
                      ->orWhereNull('colour_type');
            });
        }

        if ($phaseColumn !== null) {
            $query->where(function (Builder $query) use ($phaseColumn) {
                $query->whereColumn('phase_id', $phaseColumn)
                      ->orWhereNull('phase_id');
            });
        }

        if ($sizesColumn !== null) {
            $query->where(function (Builder $query) use ($sizesColumn) {
                $query->whereColumn('sizes_id', $sizesColumn)
                      ->orWhereRaw('sizes_id = 1'); // fallback to default size
            })
            ->orderByRaw("CASE WHEN sizes_id = {$sizesColumn} THEN 0 ELSE 1 END ASC");
        }

        $query->orderByRaw("CASE WHEN quote_status = 'confirmed' THEN 1 ELSE 0 END DESC")
              ->orderByRaw("colourways_id IS NOT NULL DESC")
              ->orderByRaw("colour_type IS NOT NULL DESC")
            //   ->orderByRaw("CASE
            //                     WHEN sizes_id = {$sizesColumn} THEN 0 -- Prioritize the specific size
            //                     ELSE 1 -- Fallback to the default size
            //                 END ASC")
              ->orderByDesc('min_qty')
              ->limit(1);

        return $query;
    }
}
