<?php

namespace App\Services\PackingListImports;

use App\Models\Sizes;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use App\Models\CustomerOrderLines;
use Illuminate\Support\Facades\DB;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

class PackingListImportService
{
    /**
     * Validate standardized packing list data per shipment line
     */
    public function validateImportData(array $data): array
    {
        $errors = [];

        if (!isset($data['packing_lists']) || !is_array($data['packing_lists'])) {
            $errors[] = 'Invalid data structure: packing_lists array is required';
            return $errors;
        }

        $hasUnmatched = false;
        foreach ($data['packing_lists'] as $plIndex => $packingList) {
            $prefix = "PackingList {$plIndex}: ";
            
            if (empty($packingList['customer_order_line_id'])) {
                $hasUnmatched = true;
            }
            
            if (!isset($packingList['sizes']) || !is_array($packingList['sizes']) || empty($packingList['sizes'])) {
                $errors[] = $prefix . 'At least one size quantity is required';
            }
        }

        // Add a warning if some items couldn't be matched
        if ($hasUnmatched) {
            $errors[] = 'Warning: Some packing list items could not be matched to existing order lines. Please ensure the order has been imported first, or the style/color codes match exactly.';
        }

        return $errors;
    }

    /**
     * Apply packing list data to shipment lines and sizes, mark as shipped
     */
    public function processImport(array $standardizedData, bool $markAsShipped = true): array
    {
        $results = [
            'success' => false,
            'shipment_lines_updated' => 0,
            'sizes_updated' => 0,
            'errors' => [],
        ];

        try {
            DB::beginTransaction();
            
            // Extract factory invoice number from metadata if available
            $factoryInvoiceNo = $standardizedData['metadata']['invoice_data']['invoice_no'] 
                ?? $standardizedData['metadata']['invoice_data']['invoice_number'] 
                ?? null;

            foreach ($standardizedData['packing_lists'] as $packingList) {
                if (empty($packingList['customer_order_line_id'])) {
                    $results['errors'][] = "Skipped: No order line ID for style {$packingList['_style']} color {$packingList['_color']}";
                    continue;
                }

                // The customer_order_line_id is actually the shipment_line_id from the extractor
                $shipmentLine = ShipmentLine::find((int) $packingList['customer_order_line_id']);
                
                if (!$shipmentLine) {
                    $results['errors'][] = "Shipment line {$packingList['customer_order_line_id']} not found";
                    continue;
                }

                // Update exfty if provided
                if (!empty($packingList['exfty'])) {
                    $shipmentLine->exfty = $packingList['exfty'];
                }
                
                // Update factory invoice if available
                if (!empty($factoryInvoiceNo)) {
                    $shipmentLine->factory_invoice = $factoryInvoiceNo;
                }

                // Mark as shipped/complete only if requested
                if ($markAsShipped) {
                    $shipmentLine->shipped = 1;
                }
                $shipmentLine->save();

                // Get existing shipment line sizes
                $existingSizes = ShipmentLineSizes::where('shipment_line_id', $shipmentLine->id)
                    ->get()
                    ->keyBy('sizes_id');

                // Update or create sizes with shipped quantities
                $sizesWithQuantities = [];
                foreach ($packingList['sizes'] as $row) {
                    $size = Sizes::firstOrCreate(['name' => $row['size']]);
                    $qty = (int) ($row['qty'] ?? 0);
                    
                    if (isset($existingSizes[$size->id])) {
                        // Update existing size - set shipped_qty
                        $sizeRecord = $existingSizes[$size->id];
                        $sizeRecord->shipped_qty = $qty;
                        $sizeRecord->save();
                        $results['sizes_updated']++;
                    } else {
                        // Create new size record
                        ShipmentLineSizes::create([
                            'shipment_line_id' => $shipmentLine->id,
                            'sizes_id' => $size->id,
                            'qty' => $qty,
                            'shipped_qty' => $qty,
                        ]);
                        $results['sizes_updated']++;
                    }
                }

                $results['shipment_lines_updated']++;
            }

            DB::commit();
            $results['success'] = true;
        } catch (\Exception $e) {
            DB::rollBack();
            $results['errors'][] = $e->getMessage();
        }

        return $results;
    }
}


