<?php

namespace App\Services\PackingListImports;

use App\Services\PackingListImports\Extractors\BodenPackingListExtractor;
use App\Services\PackingListImports\Extractors\ChintiPackingListExtractor;
use App\Services\PackingListImports\Extractors\MrPorterPackingListExtractor;
use App\Services\PackingListImports\Extractors\PurePackingListExtractor;
use App\Services\PackingListImports\Extractors\CharlesTyrwhittPackingListExtractor;

class PackingListExtractorFactory
{
    /**
     * Available extractors mapped by customer name
     */
    private static array $extractors = [
        'boden' => BodenPackingListExtractor::class,
        'chinti' => ChintiPackingListExtractor::class,
        'mr porter' => MrPorterPackingListExtractor::class,
        'pure' => PurePackingListExtractor::class,
        'pure collection' => PurePackingListExtractor::class,
        'charles tyrwhitt' => CharlesTyrwhittPackingListExtractor::class,
    ];

    /**
     * Create extractor instance for given customer
     */
    public static function make(string $customerName): BaseExtractor
    {
        $customerName = strtolower(trim($customerName));

        if (!isset(self::$extractors[$customerName])) {
            throw new \InvalidArgumentException("No packing list extractor found for customer: {$customerName}");
        }

        $extractorClass = self::$extractors[$customerName];

        if (!class_exists($extractorClass)) {
            throw new \RuntimeException("Packing list extractor class does not exist: {$extractorClass}");
        }

        return new $extractorClass();
    }

    /**
     * Get list of available customers
     */
    public static function getAvailableCustomers(): array
    {
        $customers = [];
        $seenNames = [];

        foreach (self::$extractors as $key => $class) {
            if (!class_exists($class)) continue;

            $instance = new $class();
            $name = $instance->getCustomerName();

            if (in_array($name, $seenNames, true)) continue; // avoid duplicate labels

            $customers[$key] = $name;
            $seenNames[] = $name;
        }

        return $customers;
    }

    /**
     * Register a new customer extractor
     */
    public static function register(string $customerKey, string $extractorClass): void
    {
        if (!is_subclass_of($extractorClass, BaseExtractor::class)) {
            throw new \InvalidArgumentException("Extractor class must extend BaseExtractor");
        }

        self::$extractors[$customerKey] = $extractorClass;
    }

    /**
     * Check if customer extractor exists
     */
    public static function hasExtractor(string $customerName): bool
    {
        $customerName = strtolower(trim($customerName));
        return isset(self::$extractors[$customerName]);
    }
}


