<?php

namespace App\Services\PackingListImports\Extractors;

use App\Models\Customer;
use App\Models\CustomerOrders;
use App\Models\CustomerOrderLines;
use App\Services\PackingListImports\BaseExtractor;
use App\Services\CommissionImports\TabulaService;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Spatie\PdfToText\Pdf;
use Illuminate\Support\Facades\Log;
use Gemini;
use Gemini\Data\Blob;
use Gemini\Enums\MimeType;
use OpenAI;

class BodenPackingListExtractor extends BaseExtractor
{
    private TabulaService $tabulaService;

    public function __construct()
    {
        $this->tabulaService = new TabulaService();
    }
    public function extractData(TemporaryUploadedFile $file): array
    {
        $extension = strtolower($file->getClientOriginalExtension());

        if ($extension === 'pdf') {
            return $this->extractFromPdf($file);
        }

        throw new \InvalidArgumentException("Unsupported file type for Boden packing lists: {$extension}");
    }

    public function getCustomerName(): string
    {
        return 'Boden';
    }

    public function getCustomerId(): int
    {
        return (int) Customer::where('name', 'Boden')->value('id');
    }

    public function validateCustomerData(array $data): array
    {
        $errors = [];
        $warnings = [];

        foreach ($data['packing_lists'] ?? [] as $index => $pl) {
            if (empty($pl['customer_order_line_id'])) {
                // This is a warning, not an error - user might import order later
                $warnings[] = "Style {$pl['_style']} Color {$pl['_color']}: Not matched to existing order line (PO: {$pl['_po_number']})";
            }
            
            if (empty($pl['sizes'])) {
                $errors[] = "Packing list {$index}: No size quantities found";
            }
        }

        // Add warnings as info messages, not blocking errors
        if (!empty($warnings)) {
            $errors[] = "INFO: " . count($warnings) . " item(s) could not be matched. Import the order first, then re-upload this packing list. Details: " . implode('; ', array_slice($warnings, 0, 3));
        }

        return $errors;
    }

    /**
     * Extract packing list data from PDF using Tabula for tables and pdftotext for metadata
     */
    private function extractFromPdf(TemporaryUploadedFile $file): array
    {
        // Extract full text for metadata (invoice numbers, dates, etc.)
        $fullText = Pdf::getText($file->getRealPath());
        
        // Also get layout-preserved text for better PO extraction
        $layoutText = $this->pdfToTextWithLayout($file->getRealPath());
        
        // Extract invoice metadata from text
        $invoiceData = $this->extractInvoiceData($fullText, $layoutText);

        // Use Tabula for deterministic table extraction
        $packingListItems = $this->extractWithTabulaDeterministic($file->getRealPath(), $invoiceData);

        return [
            'packing_lists' => $packingListItems,
            'metadata' => [
                'filename' => $file->getClientOriginalName(),
                'extraction_method' => 'boden_packing_list_tabula',
                'invoice_data' => $invoiceData,
            ],
        ];
    }

    /**
     * Convert PDF to text with layout preservation
     */
    private function pdfToTextWithLayout(string $pdfPath): string
    {
        $tmp = tempnam(sys_get_temp_dir(), 'pdfTxt');
        exec('pdftotext -layout ' . escapeshellarg($pdfPath) . ' ' . escapeshellarg($tmp));
        $text = file_get_contents($tmp);
        unlink($tmp);
        return $text;
    }

    /**
     * Extract invoice metadata (PO, invoice number, date, etc.)
     */
    private function extractInvoiceData(string $text, string $layoutText = ''): array
    {
        $data = [
            'invoice_no' => null,
            'po_number' => null,
            'invoice_date' => null,
            'exfty_date' => null,
            'net_weight' => null,
            'gross_weight' => null,
        ];

        // Extract Invoice Number
        if (preg_match('/Invoice\s+#?[:\.]?\s*([A-Z0-9\-]+)/i', $text, $matches)) {
            $data['invoice_no'] = trim($matches[1]);
        } elseif (preg_match('/Invoice\s+no[:\.]?\s*([A-Z0-9\-]+)/i', $text, $matches)) {
            $data['invoice_no'] = trim($matches[1]);
        }

        // Extract PO Number - try layout text first (better formatting)
        $poPatterns = [
            '/PO[：:\s]+(\d{6,10})/i',  // Matches "PO：66359501" or "PO 66359501"
            '/PO\s+No\.?\s*#?\s*(\d{6,10})/i',
            '/Purchase\s+Order[：:\.]?\s*(\d{6,10})/i',
        ];
        
        // Try layout text first
        if (!empty($layoutText)) {
            foreach ($poPatterns as $pattern) {
                if (preg_match($pattern, $layoutText, $matches)) {
                    $data['po_number'] = trim($matches[1]);
                    break;
                }
            }
        }
        
        // Fall back to regular text if not found
        if (empty($data['po_number'])) {
            foreach ($poPatterns as $pattern) {
                if (preg_match($pattern, $text, $matches)) {
                    $data['po_number'] = trim($matches[1]);
                    break;
                }
            }
        }

        // Extract Invoice Date
        if (preg_match('/Invoice\s+[Dd]ate[:\.]?\s*(\d{4}[\/\-]\d{1,2}[\/\-]\d{1,2})/i', $text, $matches)) {
            $data['invoice_date'] = $this->parseDate($matches[1]);
        }

        // Extract Delivery Date as exfty
        if (preg_match('/Delivery\s+Date[:\.]?\s*(\d{4}[\/\-]\d{1,2}[\/\-]\d{1,2})/i', $text, $matches)) {
            $data['exfty_date'] = $this->parseDate($matches[1]);
        }

        // Extract weights
        if (preg_match('/NET\s+WEIGHT[:\.]?\s*([\d.]+)\s*KGS?/i', $text, $matches)) {
            $data['net_weight'] = $matches[1];
        }
        
        if (preg_match('/GROSS\s+WEIGHT[:\.]?\s*([\d.]+)\s*KGS?/i', $text, $matches)) {
            $data['gross_weight'] = $matches[1];
        }

        // Extract unit price from invoice (using layout text for better parsing)
        // Format: Item # Product ID Description Qty Unit Price Line Total
        //         1      K1189     LADIES'...     690  51.00   35,190.00
        if (preg_match('/^\s*\d+\s+\w+.*?\s+(\d+)\s+([\d.,]+)\s+([\d.,]+)\s*$/m', $layoutText, $matches)) {
            $data['unit_price'] = (float) str_replace(',', '', $matches[2]);
        } elseif (preg_match('/Unit\s+Price[:\.]?\s*[\$£€]?\s*([\d.,]+)/i', $text, $matches)) {
            $data['unit_price'] = (float) str_replace(',', '', $matches[1]);
        }

        return $data;
    }

    /**
     * Extract packing list using Tabula (deterministic, accurate)
     */
    private function extractWithTabulaDeterministic(string $pdfPath, array $invoiceData): array
    {
        try {
            // Extract table from page 2 using lattice mode (for bordered tables)
            $rows = $this->tabulaService->extractTables($pdfPath, [2], 'lattice');
            
            if (empty($rows)) {
                $rows = $this->tabulaService->extractTables($pdfPath, [2], 'stream');
            }
            
            if (empty($rows)) {
                return [];
            }
            
            // TabulaService returns a flat array of rows, so we can use it directly
            // Parse the table
            $cartons = $this->parseTabulaTable($rows);
            
            // Normalize and collate
            return $this->normalizeAndCollate($cartons, $invoiceData);
            
        } catch (\Exception $e) {
            Log::error('Boden Tabula - Extraction failed', [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }
    
    /**
     * Parse Tabula table into carton data
     */
    private function parseTabulaTable(array $table): array
    {
        $cartons = [];
        
        // Find header row and column indices
        $headerRowIndex = null;
        $columnMap = [];
        
        foreach ($table as $index => $row) {
            if (!is_array($row)) {
                continue;
            }
            
            $row = array_map('trim', $row);
            $rowText = implode(' ', $row);
            
            // Look for header row
            if (stripos($rowText, 'XS') !== false && stripos($rowText, 'XL') !== false) {
                $headerRowIndex = $index;
                
                // Map column indices
                foreach ($row as $colIndex => $cellValue) {
                    $cellValue = strtoupper(trim($cellValue));
                    
                    // Map known columns - with flexible matching
                    
                    // Carton number column
                    if (stripos($cellValue, 'CTN') !== false || 
                        stripos($cellValue, 'CARTON') !== false ||
                        $cellValue === 'NO' && $colIndex === 0) {
                        $columnMap['ctn'] = $colIndex;
                    }
                    
                    // Style/Code column 
                    elseif (stripos($cellValue, 'CODICE') !== false || 
                            stripos($cellValue, 'STYLE') !== false ||
                            stripos($cellValue, 'CODE') !== false ||
                            stripos($cellValue, 'ITEM') !== false) {
                        $columnMap['style'] = $colIndex;
                    }
                    
                    // Color column
                    elseif (stripos($cellValue, 'COLOR') !== false ||
                            stripos($cellValue, 'COLOUR') !== false ||
                            stripos($cellValue, 'CLR') !== false) {
                        $columnMap['color'] = $colIndex;
                    }
                    
                    // Size columns
                    elseif ($cellValue === 'XS') {
                        $columnMap['xs'] = $colIndex;
                    } elseif ($cellValue === 'S') {
                        $columnMap['s'] = $colIndex;
                    } elseif ($cellValue === 'M') {
                        $columnMap['m'] = $colIndex;
                    } elseif ($cellValue === 'L') {
                        $columnMap['l'] = $colIndex;
                    } elseif ($cellValue === 'XL') {
                        $columnMap['xl'] = $colIndex;
                    }
                }
                
                // If we're missing critical columns, try to infer them from standard positions
                // Typical Boden format: CTN (0), Destination (1), PO (2), CODICE/Style (3), COLOR Code (4), XS (5), S (6), M (7), L (8), XL (9)
                if (!isset($columnMap['ctn']) && count($row) > 0) {
                    $columnMap['ctn'] = 0;
                }
                if (!isset($columnMap['style']) && count($row) > 3) {
                    // Style is typically at position 3 or 4
                    $columnMap['style'] = 3;
                }
                if (!isset($columnMap['color']) && count($row) > 4) {
                    // Color is typically at position 4 or 5
                    $columnMap['color'] = 4;
                }
                
                break;
            }
        }
        
        if ($headerRowIndex === null || empty($columnMap)) {
            return [];
        }
        
        // Verify we have minimum required columns
        if (!isset($columnMap['ctn'])) {
            $columnMap['ctn'] = 0;
        }
        if (!isset($columnMap['style'])) {
            $columnMap['style'] = 3;
        }
        if (!isset($columnMap['color'])) {
            $columnMap['color'] = 4;
        }
        
        // Extract data rows
        for ($i = $headerRowIndex + 1; $i < count($table); $i++) {
            $row = array_map('trim', $table[$i]);
            
            // Check if this is a valid data row (starts with carton number)
            $ctnValue = $row[$columnMap['ctn'] ?? 0] ?? '';
            
            if (!is_numeric($ctnValue) || empty($ctnValue)) {
                continue;
            }
            
            // Extract carton data
            $carton = [
                'ctn' => (int) $ctnValue,
                'style' => $row[$columnMap['style'] ?? 999] ?? null,
                'color' => $row[$columnMap['color'] ?? 999] ?? null,
                'xs' => null,
                's' => null,
                'm' => null,
                'l' => null,
                'xl' => null,
            ];
            
            // Extract sizes
            foreach (['xs', 's', 'm', 'l', 'xl'] as $size) {
                if (isset($columnMap[$size])) {
                    $value = $row[$columnMap[$size]] ?? '';
                    $value = trim($value);
                    
                    // Clean and parse the value
                    if (is_numeric($value) && $value > 0) {
                        $carton[$size] = (int) $value;
                    }
                }
            }
            
            Log::debug("Boden Tabula - Extracted carton data", [
                'carton' => $carton,
                'has_style' => !empty($carton['style']),
                'has_color' => !empty($carton['color']),
                'has_sizes' => ($carton['xs'] || $carton['s'] || $carton['m'] || $carton['l'] || $carton['xl'])
            ]);
            
            // Only add if we have valid style, color, and at least one size
            if (!empty($carton['style']) && !empty($carton['color']) && 
                ($carton['xs'] || $carton['s'] || $carton['m'] || $carton['l'] || $carton['xl'])) {
                
                $cartons[] = $carton;
                
                Log::info("Boden Tabula - Carton {$carton['ctn']}", [
                    'style' => $carton['style'],
                    'color' => $carton['color'],
                    'sizes' => array_filter([
                        'XS' => $carton['xs'],
                        'S' => $carton['s'],
                        'M' => $carton['m'],
                        'L' => $carton['l'],
                        'XL' => $carton['xl']
                    ])
                ]);
            } else {
                Log::warning("Boden Tabula - Skipping carton (missing required data)", [
                    'ctn' => $carton['ctn'],
                    'style' => $carton['style'],
                    'color' => $carton['color'],
                    'sizes' => array_filter([
                        'XS' => $carton['xs'],
                        'S' => $carton['s'],
                        'M' => $carton['m'],
                        'L' => $carton['l'],
                        'XL' => $carton['xl']
                    ]),
                    'row' => $row
                ]);
            }
        }
        
        return $cartons;
    }

    /**
     * Extract packing list using GPT-5 Vision (fallback)
     */
    private function extractWithGPT(string $pdfPath, array $invoiceData): array
    {
        $apiKey = getenv('OPENAI_API_KEY');
        
        // Convert PDF page 2 to image for GPT Vision
        $imagePath = $this->convertPdfPageToImage($pdfPath, 2);
        if (!$imagePath) {
            Log::error('Boden GPT - Failed to convert PDF to image, falling back to Gemini');
            return $this->extractPackingListItemsWithAI($pdfPath, $invoiceData);
        }
        
        try {
            $imageData = base64_encode(file_get_contents($imagePath));
            
            $prompt = "Extract ALL rows from the packing list table in this image.\n\n" .
                     "Return ONLY a JSON array with this EXACT structure:\n" .
                     "[{\"ctn\": 1, \"style\": \"K1156\", \"color\": \"NEU\", \"xs\": 14, \"s\": 6, \"m\": 20, \"l\": null, \"xl\": null}, ...]\n\n" .
                     "CRITICAL RULES:\n" .
                     "1. Extract EVERY single row from the table - typically 5-20 rows\n" .
                     "2. ctn = carton number (first column)\n" .
                     "3. style = CODICE column value (e.g. K1156, K1189)\n" .
                     "4. color = COLOR Code column value (e.g. NEU, PRP, DGY)\n" .
                     "5. xs, s, m, l, xl = EXACT numbers from those size columns\n" .
                     "6. Use null (not 0) for empty cells or dashes\n" .
                     "7. Do NOT skip any rows\n" .
                     "8. Return ONLY the JSON array, no markdown, no text, no explanation";
            
            $client = OpenAI::client($apiKey);
            
            $response = $client->chat()->create([
                'model' => 'gpt-5',
                'max_tokens' => 4096,
                'messages' => [
                    [
                        'role' => 'user',
                        'content' => [
                            [
                                'type' => 'text',
                                'text' => $prompt,
                            ],
                            [
                                'type' => 'image_url',
                                'image_url' => [
                                    'url' => 'data:image/png;base64,' . $imageData,
                                ],
                            ],
                        ],
                    ],
                ],
            ]);
            
            $responseText = $response->choices[0]->message->content ?? '';
            $responseText = str_replace(['```json', '```'], '', $responseText);
            $responseText = trim($responseText);
            
            Log::info('Boden GPT - Response received', [
                'response_length' => strlen($responseText),
                'response_preview' => substr($responseText, 0, 500)
            ]);
            
            $cartons = json_decode($responseText, true);
            
            if (json_last_error() !== JSON_ERROR_NONE || empty($cartons)) {
                Log::error('Boden GPT - Failed to parse response', [
                    'error' => json_last_error_msg(),
                    'response' => substr($responseText, 0, 500)
                ]);
                unlink($imagePath);
                return $this->extractPackingListItemsWithAI($pdfPath, $invoiceData);
            }
            
            Log::info('Boden GPT - Successfully extracted', [
                'cartons' => count($cartons),
                'first_carton' => $cartons[0] ?? null,
                'last_carton' => end($cartons) ?: null
            ]);
            
            unlink($imagePath);
            
            // Normalize and collate
            return $this->normalizeAndCollate($cartons, $invoiceData);
            
        } catch (\Exception $e) {
            Log::error('Boden GPT - Extraction failed', ['error' => $e->getMessage()]);
            if (isset($imagePath) && file_exists($imagePath)) unlink($imagePath);
            return $this->extractPackingListItemsWithAI($pdfPath, $invoiceData);
        }
    }
    
    /**
     * Convert PDF page to PNG image for vision AI
     */
    private function convertPdfPageToImage(string $pdfPath, int $page): ?string
    {
        try {
            $outputPath = sys_get_temp_dir() . '/boden_pl_' . uniqid() . '.png';
            $command = sprintf(
                'pdftoppm -png -f %d -l %d -r 150 %s %s',
                $page,
                $page,
                escapeshellarg($pdfPath),
                escapeshellarg(str_replace('.png', '', $outputPath))
            );
            
            exec($command, $output, $returnCode);
            
            // pdftoppm adds -1 suffix
            $actualPath = str_replace('.png', '-1.png', $outputPath);
            if (file_exists($actualPath)) {
                return $actualPath;
            }
            
            Log::error('Boden - PDF to image conversion failed', [
                'command' => $command,
                'return_code' => $returnCode
            ]);
            return null;
            
        } catch (\Exception $e) {
            Log::error('Boden - PDF to image error', ['error' => $e->getMessage()]);
            return null;
        }
    }
    
    /**
     * Normalize cartons and collate by style/color
     */
    private function normalizeAndCollate(array $cartons, array $invoiceData): array
    {
        // Normalize
        $normalized = [];
        foreach ($cartons as $carton) {
            $sizes = [];
            foreach (['xs' => 'XS', 's' => 'S', 'm' => 'M', 'l' => 'L', 'xl' => 'XL'] as $key => $sizeName) {
                $qty = $carton[$key] ?? null;
                if ($qty !== null && $qty > 0) {
                    $sizes[$sizeName] = (int) $qty;
                }
            }
            
            if (!empty($sizes)) {
                $normalized[] = [
                    'carton_number' => $carton['ctn'] ?? 0,
                    'style' => strtoupper($carton['style'] ?? 'UNKNOWN'),
                    'color' => strtoupper($carton['color'] ?? 'UNKNOWN'),
                    'sizes' => $sizes,
                ];
            }
        }
        
        // Collate
        $collated = $this->collateQuantitiesByStyleAndColor($normalized);
        
        // Match to order lines
        return $this->matchToOrderLines($collated, $invoiceData);
    }

    /**
     * Extract packing list items using AI (Gemini)
     */
    private function extractPackingListItemsWithAI(string $pdfPath, array $invoiceData): array
    {
        $apiKey = getenv('GEMINI_API_KEY');
        if (!$apiKey) {
            Log::error('GEMINI_API_KEY not set');
            return [];
        }
        
        $client = Gemini::client($apiKey);
        
        // Read PDF as binary for AI
        $pdfContent = file_get_contents($pdfPath);
        $encoded = base64_encode($pdfContent);
        
        $prompt = '
You are a data extraction agent. Your ONLY job is to read the packing list table and return the raw data.

TASK:
Find the packing list table (page 2). Extract EVERY row exactly as shown.

TABLE STRUCTURE:
CTN | Destination | PO | CODICE | COLOR Code | XS | S | M | L | XL | Total | ...

WHAT TO EXTRACT (for each row):
1. CTN number (carton number)
2. CODICE (style code like K1156, K1189)
3. COLOR Code (like NEU, PRP, DGY)
4. XS quantity (number under XS column, or null if empty/dash)
5. S quantity (number under S column, or null if empty/dash)
6. M quantity (number under M column, or null if empty/dash)
7. L quantity (number under L column, or null if empty/dash)
8. XL quantity (number under XL column, or null if empty/dash)

RULES:
- Extract EVERY row in the table
- Read EXACTLY what is in each cell
- Use null for empty cells, dashes, or zeros
- Do NOT calculate or sum anything
- Do NOT skip any rows
- Return ONLY valid JSON

EXAMPLE:
Table row: "1  UK  66303802  K1156  NEU  14  6  20  -  -"
Output: {"ctn": 1, "style": "K1156", "color": "NEU", "xs": 14, "s": 6, "m": 20, "l": null, "xl": null}

Table row: "5  UK  66303802  K1156  PRP  11  28  34  17  10"
Output: {"ctn": 5, "style": "K1156", "color": "PRP", "xs": 11, "s": 28, "m": 34, "l": 17, "xl": 10}

OUTPUT (JSON array):
[
  {"ctn": 1, "style": "K1156", "color": "NEU", "xs": 14, "s": 6, "m": 20, "l": null, "xl": null},
  {"ctn": 2, "style": "K1156", "color": "NEU", "xs": null, "s": 14, "m": 20, "l": null, "xl": null},
  ...
]

Extract ALL rows now:
';
        
        $parts = [
            $prompt,
            new Blob(
                mimeType: MimeType::APPLICATION_PDF,
                data: $encoded
            ),
        ];
        
        $attempts = 0;
        $cartons = null;
        
        while (!$cartons && $attempts < 3) {
            try {
                // Use Gemini Flash from config
                $result = $client->generativeModel(config('gemini.models.flash'))->generateContent($parts);
                $responseText = $result->text();
                
                Log::info("Boden AI - Attempt {$attempts} raw response", [
                    'response_length' => strlen($responseText),
                    'response_preview' => substr($responseText, 0, 1000)
                ]);
                
                // Clean up the response
                $responseText = str_replace(['```json', '```'], '', $responseText);
                $responseText = trim($responseText);
                
                $cartons = json_decode($responseText, true);
                
                if (json_last_error() !== JSON_ERROR_NONE) {
                    Log::warning('Boden AI - JSON decode error', [
                        'error' => json_last_error_msg(),
                        'response' => substr($responseText, 0, 500)
                    ]);
                    $cartons = null;
                } else {
                    Log::info('Boden AI - Successfully decoded JSON', [
                        'cartons_count' => count($cartons),
                        'first_3_cartons' => array_slice($cartons, 0, 3)
                    ]);
                }
                
            } catch (\Exception $e) {
                Log::error('Boden AI - Extraction error', ['error' => $e->getMessage()]);
            }
            
            $attempts++;
        }
        
        if (!$cartons) {
            Log::error('Boden AI - Failed to extract cartons after 3 attempts');
            return [];
        }
        
        Log::info('Boden AI - Extracted cartons', [
            'count' => count($cartons),
            'sample' => $cartons[0] ?? null
        ]);
        
        // Normalize carton structure - convert flat structure to nested
        $normalizedCartons = [];
        foreach ($cartons as $carton) {
            $sizes = [];
            
            // Map from flat structure (xs, s, m, l, xl) to sizes array
            $sizeMap = [
                'xs' => 'XS',
                's' => 'S', 
                'm' => 'M',
                'l' => 'L',
                'xl' => 'XL'
            ];
            
            foreach ($sizeMap as $key => $sizeName) {
                $qty = $carton[$key] ?? $carton[strtoupper($key)] ?? null;
                if ($qty !== null && $qty > 0) {
                    $sizes[$sizeName] = (int) $qty;
                }
            }
            
            if (empty($sizes)) {
                Log::warning("Boden AI - Carton has no sizes", ['carton' => $carton]);
                continue;
            }
            
            $normalizedCartons[] = [
                'carton_number' => $carton['ctn'] ?? $carton['carton_number'] ?? 0,
                'style' => strtoupper($carton['style'] ?? 'UNKNOWN'),
                'color' => strtoupper($carton['color'] ?? 'UNKNOWN'),
                'sizes' => $sizes,
            ];
            
            Log::info("Boden AI - Normalized carton {$carton['ctn']}", [
                'style' => $carton['style'],
                'color' => $carton['color'],
                'sizes' => $sizes
            ]);
        }
        
        // Collate by style and color
        $collatedItems = $this->collateQuantitiesByStyleAndColor($normalizedCartons);
        
        Log::info('Boden AI - Collated items', [
            'count' => count($collatedItems),
            'sample' => $collatedItems[0] ?? null
        ]);
        
        // Match to existing order lines in database
        return $this->matchToOrderLines($collatedItems, $invoiceData);
    }
    
    /**
     * Extract packing list items using Tabula (DEPRECATED - use AI instead)
     */
    private function extractPackingListItemsWithTabula(string $pdfPath, array $invoiceData): array
    {
        // Extract tables from PDF (packing list is usually on page 2)
        $mode = 'lattice'; // Try lattice mode first for structured tables
        $pages = [2]; // Packing list is typically on page 2
        
        try {
            $rows = $this->tabulaService->extractTables($pdfPath, $pages, $mode);
        } catch (\Exception $e) {
            Log::warning('Boden Packing List - Tabula extraction failed', ['error' => $e->getMessage()]);
            return [];
        }
        
        Log::info('Boden Packing List - Tabula extracted rows', [
            'row_count' => count($rows),
            'sample_rows' => array_slice($rows, 0, 3),
        ]);

        // Extract carton data from Tabula rows
        $cartonData = $this->parseCartonDataFromTabulaRows($rows);
        
        Log::info('Boden Packing List - Cartons parsed', [
            'carton_count' => count($cartonData),
        ]);

        if (empty($cartonData)) {
            Log::warning('Boden Packing List - No carton data found');
            return [];
        }

        // Collate quantities by style and color
        $collatedItems = $this->collateQuantitiesByStyleAndColor($cartonData);
        
        Log::info('Boden Packing List - Items collated', [
            'item_count' => count($collatedItems),
        ]);

        // Match to existing order lines in database
        $packingListItems = $this->matchToOrderLines($collatedItems, $invoiceData);

        return $packingListItems;
    }

    /**
     * Parse carton data from Tabula extracted rows
     */
    private function parseCartonDataFromTabulaRows(array $rows): array
    {
        $cartons = [];
        $headerFound = false;
        $sizeColumnIndices = [];

        foreach ($rows as $rowIndex => $row) {
            // Look for the header row with size columns
            if (!$headerFound) {
                // Check if this row contains size headers
                $rowText = implode(' ', $row);
                if (preg_match('/XS\s+S\s+M\s+L\s+XL/i', $rowText)) {
                    $headerFound = true;
                    // Find which columns contain each size
                    foreach ($row as $colIndex => $cell) {
                        $cell = trim(strtoupper($cell));
                        if (in_array($cell, ['XS', 'S', 'M', 'L', 'XL'])) {
                            $sizeColumnIndices[$cell] = $colIndex;
                        }
                    }
                    Log::info('Boden Packing List - Found size columns', ['indices' => $sizeColumnIndices]);
                }
                continue;
            }

            // Parse data rows
            // Expected format: [carton_no, destination, po_number, style, color, xs_qty, s_qty, m_qty, l_qty, xl_qty, total]
            if (empty($row[0]) || !is_numeric($row[0])) {
                continue; // Skip rows that don't start with a carton number
            }

            $cartonNo = (int) $row[0];
            
            // Find style and color in the row
            $style = null;
            $color = null;
            $poNumber = null;
            
            foreach ($row as $colIndex => $cell) {
                $cell = trim($cell);
                
                // Look for style code (e.g., K1189)
                if (preg_match('/^[A-Z]+\d+$/', $cell)) {
                    $style = $cell;
                }
                
                // Look for color code (3-4 letter codes like DGY, PUR)
                if (preg_match('/^[A-Z]{2,4}$/', $cell) && $cell !== 'UK' && $cell !== 'US' && !in_array($cell, ['XS', 'S', 'M', 'L', 'XL'])) {
                    $color = $cell;
                }
                
                // Look for PO number (8 digits)
                if (preg_match('/^\d{8}$/', $cell)) {
                    $poNumber = $cell;
                }
            }

            if (!$style || !$color) {
                continue; // Skip incomplete rows
            }

            // Extract quantities for each size using the column indices
            $sizes = [];
            foreach ($sizeColumnIndices as $sizeName => $colIndex) {
                if (isset($row[$colIndex])) {
                    $cellValue = trim($row[$colIndex]);
                    // Make sure we're reading a numeric value and not empty
                    if (is_numeric($cellValue)) {
                        $qty = (int) $cellValue;
                        if ($qty > 0) {
                            $sizes[$sizeName] = $qty;
                        }
                    }
                }
            }
            
            // Log the raw row data for debugging
            Log::info("Boden Packing List - Raw row $cartonNo", [
                'row' => $row,
                'style' => $style,
                'color' => $color,
                'extracted_sizes' => $sizes,
                'size_column_indices' => $sizeColumnIndices
            ]);

            if (!empty($sizes)) {
                $cartons[] = [
                    'carton_number' => $cartonNo,
                    'po_number' => $poNumber,
                    'style' => $style,
                    'color' => $color,
                    'sizes' => $sizes,
                ];

                Log::info("Boden Packing List - Parsed carton $cartonNo: $style $color", ['sizes' => $sizes]);
            }
        }

        return $cartons;
    }


    /**
     * Collate quantities by style and color across all cartons
     */
    private function collateQuantitiesByStyleAndColor(array $cartons): array
    {
        $collated = [];

        foreach ($cartons as $carton) {
            $style = $carton['style'] ?? 'UNKNOWN';
            $color = $carton['color'] ?? 'UNKNOWN';
            $key = $style . '|' . $color;

            if (!isset($collated[$key])) {
                $collated[$key] = [
                    'style' => $style,
                    'color' => $color,
                    'sizes' => [],
                    'total_cartons' => 0,
                ];
            }

            $collated[$key]['total_cartons']++;

            // Add up quantities for each size
            foreach ($carton['sizes'] as $size => $qty) {
                if (!isset($collated[$key]['sizes'][$size])) {
                    $collated[$key]['sizes'][$size] = 0;
                }
                $collated[$key]['sizes'][$size] += $qty;
            }
        }
        
        // Log the collated results for debugging
        foreach ($collated as $item) {
            $totalQty = array_sum($item['sizes']);
            Log::info("Boden Packing List - Collated: {$item['style']} {$item['color']}", [
                'sizes' => $item['sizes'],
                'total_cartons' => $item['total_cartons'],
                'total_qty' => $totalQty
            ]);
        }

        return array_values($collated);
    }

    /**
     * Match collated items to existing order lines in database
     */
    private function matchToOrderLines(array $collatedItems, array $invoiceData): array
    {
        $packingLists = [];
        $customerId = $this->getCustomerId();
        $poNumber = $invoiceData['po_number'] ?? null;

        if (!$poNumber) {
            Log::warning('Boden Packing List - No PO number found');
            return $this->createUnmatchedPackingLists($collatedItems, $invoiceData, $poNumber);
        }

        // Remove last 2 digits from PO number for matching (66359501 -> 663595)
        $poNumberForMatching = substr($poNumber, 0, -2);

        // Find the customer order by PO number (without last 2 digits)
        $order = CustomerOrders::where('customers_id', $customerId)
            ->where('customer_po', $poNumberForMatching)
            ->first();

        if (!$order) {
            Log::warning('Boden Packing List - Order not found', [
                'po_number' => $poNumber,
                'po_for_matching' => $poNumberForMatching,
                'customer_id' => $customerId,
            ]);
            
            return $this->createUnmatchedPackingLists($collatedItems, $invoiceData, $poNumber);
        }

        Log::info('Boden Packing List - Order found', ['order_id' => $order->id]);

        // Get all order lines for this order with their colourway data
        $includeCompleted = $this->options['include_completed_lines'] ?? false;
        
        if ($includeCompleted) {
            // Include completed lines but exclude invoiced ones
            // This allows generating invoices for manually completed shipments
            $orderLines = CustomerOrderLines::where('customer_orders_id', $order->id)
                ->whereHas('shipment_lines', function($query) {
                    $query->where(function($q) {
                              $q->where('invoiced', 0)
                                ->orWhereNull('invoiced');
                          });
                })
                ->with(['colourways.style_versions.styles'])
                ->get();
        } else {
            // Only match to lines that have shipment lines which are not complete and not invoiced
            // and don't have shipped quantities yet (shipped_qty = 0 or NULL on shipment_line_sizes)
            $orderLines = CustomerOrderLines::where('customer_orders_id', $order->id)
                ->whereHas('shipment_lines', function($query) {
                    $query->where(function($q) {
                              $q->where('complete', 0)
                                ->orWhereNull('complete');
                          })
                          ->where(function($q) {
                              $q->where('invoiced', 0)
                                ->orWhereNull('invoiced');
                          })
                          ->whereDoesntHave('shipment_line_sizes', function($sizeQuery) {
                              $sizeQuery->where('shipped_qty', '>', 0);
                          });
                })
                ->with(['colourways.style_versions.styles'])
                ->get();
        }

        // Match each collated item to an order line
        foreach ($collatedItems as $item) {
            $matchResult = $this->findMatchingOrderLine($orderLines, $item);

            // Convert sizes array to format expected by import service
            $sizes = [];
            $totalQty = 0;
            foreach ($item['sizes'] as $size => $qty) {
                $sizes[] = [
                    'size' => $size,
                    'qty' => $qty,
                ];
                $totalQty += $qty;
            }

            $packingList = [
                'customer_order_line_id' => $matchResult['order_line_id'],
                'exfty' => $invoiceData['exfty_date'] ?? $invoiceData['invoice_date'] ?? now()->toDateString(),
                'cartons' => $item['total_cartons'],
                'net_weight' => $invoiceData['net_weight'] ?? null,
                'gross_weight' => $invoiceData['gross_weight'] ?? null,
                'sizes' => $sizes,
                '_style' => $item['style'],
                '_color' => $item['color'],
                '_po_number' => $poNumber,
                '_total_qty' => $totalQty,
                '_match_status' => $matchResult['status'],
                '_possible_matches' => $matchResult['possible_matches'] ?? [],
                '_invoice_unit_price' => $invoiceData['unit_price'] ?? null,
            ];

            $packingLists[] = $packingList;
        }

        return $packingLists;
    }

    /**
     * Create packing lists for unmatched items
     */
    private function createUnmatchedPackingLists(array $collatedItems, array $invoiceData, ?string $poNumber): array
    {
        $packingLists = [];
        
        foreach ($collatedItems as $item) {
            $sizes = [];
            $totalQty = 0;
            foreach ($item['sizes'] as $size => $qty) {
                $sizes[] = ['size' => $size, 'qty' => $qty];
                $totalQty += $qty;
            }
            
            $packingLists[] = [
                'customer_order_line_id' => null,
                'exfty' => $invoiceData['exfty_date'] ?? $invoiceData['invoice_date'] ?? now()->toDateString(),
                'cartons' => $item['total_cartons'],
                'net_weight' => $invoiceData['net_weight'] ?? null,
                'gross_weight' => $invoiceData['gross_weight'] ?? null,
                'sizes' => $sizes,
                '_style' => $item['style'],
                '_color' => $item['color'],
                '_po_number' => $poNumber,
                '_total_qty' => $totalQty,
                '_match_status' => 'order_not_found',
                '_possible_matches' => [],
                '_invoice_unit_price' => $invoiceData['unit_price'] ?? null,
            ];
        }
        
        return $packingLists;
    }

    /**
     * Find matching order line for a packing list item
     * Returns: ['order_line_id' => int|null, 'status' => string, 'possible_matches' => array]
     */
    private function findMatchingOrderLine($orderLines, array $item): array
    {
        $style = strtoupper(trim($item['style']));
        $color = strtoupper(trim($item['color']));

        $styleMatches = [];
        $exactMatches = [];
        $partialMatches = [];

        foreach ($orderLines as $line) {
            if (!$line->colourways) continue;

            $styleVersion = $line->colourways->style_versions;
            if (!$styleVersion || !$styleVersion->styles) continue;

            // Match by style customer_ref
            $customerRef = strtoupper(trim($styleVersion->styles->customer_ref ?? ''));
            if ($customerRef !== $style) continue;

            // This line matches the style
            $colourwayName = strtoupper(trim($line->colourways->name ?? ''));
            
            // Try exact match first
            if ($colourwayName === $color) {
                $exactMatches[] = $line;
                continue;
            }
            
            // Try partial match
            if (strpos($colourwayName, $color) !== false || strpos($color, $colourwayName) !== false) {
                $partialMatches[] = $line;
                continue;
            }

            // Store shipment lines as style matches for potential user selection
            // Get all shipment lines for this order line
            $shipmentLines = $line->shipment_lines()
                ->where(function($q) {
                    $q->where('invoiced', 0)->orWhereNull('invoiced');
                })
                ->with('shipment_line_sizes')
                ->get();
                
            foreach ($shipmentLines as $shipmentLine) {
                $totalQty = $shipmentLine->shipment_line_sizes->sum('qty');
                $exfty = $shipmentLine->exfty ? $shipmentLine->exfty->format('d/m/Y') : 'N/A';
                
                $styleMatches[] = [
                    'order_line_id' => $shipmentLine->id, // This is actually shipment_line_id
                    'colourway_name' => $line->colourways->name,
                    'colourway_id' => $line->colourways->id,
                    'shipment_line_id' => $shipmentLine->id,
                    'line_display' => $line->colourways->name . ' (Line ' . $shipmentLine->id . ') - Qty: ' . $totalQty . ' - Ex-fty: ' . $exfty,
                ];
            }
        }

        // Check for exact matches
        if (!empty($exactMatches)) {
            // Get all shipment lines for exact matched order lines
            $allShipmentLines = [];
            foreach ($exactMatches as $line) {
                $shipmentLines = $line->shipment_lines()
                    ->where(function($q) {
                        $q->where('invoiced', 0)->orWhereNull('invoiced');
                    })
                    ->with('shipment_line_sizes')
                    ->get();
                    
                foreach ($shipmentLines as $shipmentLine) {
                    $totalQty = $shipmentLine->shipment_line_sizes->sum('qty');
                    $exfty = $shipmentLine->exfty ? $shipmentLine->exfty->format('d/m/Y') : 'N/A';
                    
                    $allShipmentLines[] = [
                        'order_line_id' => $shipmentLine->id, // This is actually shipment_line_id
                        'colourway_name' => $line->colourways->name,
                        'colourway_id' => $line->colourways->id,
                        'shipment_line_id' => $shipmentLine->id,
                        'line_display' => $line->colourways->name . ' (Line ' . $shipmentLine->id . ') - Qty: ' . $totalQty . ' - Ex-fty: ' . $exfty,
                    ];
                }
            }
            
            if (count($allShipmentLines) > 1) {
                Log::warning('Boden Packing List - Multiple shipment line matches found', [
                    'style' => $style,
                    'color' => $color,
                    'matches' => count($allShipmentLines),
                ]);
                
                // Return multiple shipment lines for user selection
                return [
                    'order_line_id' => null,
                    'status' => 'multiple_matches',
                    'possible_matches' => $allShipmentLines,
                ];
            }
            
            // Single shipment line match
            if (!empty($allShipmentLines)) {
                return [
                    'order_line_id' => $allShipmentLines[0]['shipment_line_id'],
                    'status' => 'matched',
                    'possible_matches' => [],
                ];
            }
        }
        
        // Check for partial matches
        if (!empty($partialMatches)) {
            // Get all shipment lines for partial matched order lines
            $allShipmentLines = [];
            foreach ($partialMatches as $line) {
                $shipmentLines = $line->shipment_lines()
                    ->where(function($q) {
                        $q->where('invoiced', 0)->orWhereNull('invoiced');
                    })
                    ->with('shipment_line_sizes')
                    ->get();
                    
                foreach ($shipmentLines as $shipmentLine) {
                    $totalQty = $shipmentLine->shipment_line_sizes->sum('qty');
                    $exfty = $shipmentLine->exfty ? $shipmentLine->exfty->format('d/m/Y') : 'N/A';
                    
                    $allShipmentLines[] = [
                        'order_line_id' => $shipmentLine->id, // This is actually shipment_line_id
                        'colourway_name' => $line->colourways->name,
                        'colourway_id' => $line->colourways->id,
                        'shipment_line_id' => $shipmentLine->id,
                        'line_display' => $line->colourways->name . ' (Line ' . $shipmentLine->id . ') - Qty: ' . $totalQty . ' - Ex-fty: ' . $exfty,
                    ];
                }
            }
            
            if (count($allShipmentLines) > 1) {
                Log::warning('Boden Packing List - Multiple partial shipment line matches found', [
                    'style' => $style,
                    'color' => $color,
                    'matches' => count($allShipmentLines),
                ]);
                
                // Return multiple shipment lines for user selection
                return [
                    'order_line_id' => null,
                    'status' => 'multiple_matches',
                    'possible_matches' => $allShipmentLines,
                ];
            }
            
            // Single shipment line match
            if (!empty($allShipmentLines)) {
                return [
                    'order_line_id' => $allShipmentLines[0]['shipment_line_id'],
                    'status' => 'matched',
                    'possible_matches' => [],
                ];
            }
        }

        // If no color match but style matches exist, return them for user selection
        if (!empty($styleMatches)) {
            Log::info('Boden Packing List - Style matched but color not found', [
                'style' => $style,
                'color' => $color,
                'possible_matches' => count($styleMatches),
            ]);

            return [
                'order_line_id' => null, // User needs to select
                'status' => 'style_matched_select_color',
                'possible_matches' => $styleMatches,
            ];
        }

        // No matches found
        Log::warning('Boden Packing List - No matching order line found', [
            'style' => $style,
            'color' => $color,
        ]);

        return [
            'order_line_id' => null,
            'status' => 'not_matched',
            'possible_matches' => [],
        ];
    }

    /**
     * Parse date from various formats
     */
    private function parseDate(string $dateString): ?string
    {
        $dateString = trim($dateString);
        
        if (empty($dateString)) {
            return null;
        }

        // Try common date formats
        $formats = [
            'Y-m-d',
            'Y/m/d',
            'd/m/Y',
            'm/d/Y',
            'd-m-Y',
            'm-d-Y',
            'd.m.Y',
        ];

        foreach ($formats as $format) {
            $date = \DateTime::createFromFormat($format, $dateString);
            if ($date) {
                return $date->format('Y-m-d');
            }
        }

        // Try strtotime as fallback
        $timestamp = strtotime($dateString);
        if ($timestamp !== false) {
            return date('Y-m-d', $timestamp);
        }

        return null;
    }
}
