<?php

namespace App\Services\PackingListImports;

use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

abstract class BaseExtractor
{
    protected array $options = [];
    
    /**
     * Set extractor options
     */
    public function setOptions(array $options): void
    {
        $this->options = $options;
    }
    
    /**
     * Extract data from uploaded file and return standardized format for packing lists
     */
    abstract public function extractData(TemporaryUploadedFile $file): array;

    /**
     * Get the customer name this extractor handles
     */
    abstract public function getCustomerName(): string;

    /**
     * Get the customer ID from database
     */
    abstract public function getCustomerId(): int;

    /**
     * Validate customer-specific data rules
     */
    abstract public function validateCustomerData(array $data): array;

    /**
     * Get supported file extensions for this customer
     */
    public function getSupportedExtensions(): array
    {
        return ['pdf', 'xlsx', 'xls'];
    }

    /**
     * Check if file extension is supported
     */
    public function isFileSupported(TemporaryUploadedFile $file): bool
    {
        $extension = strtolower($file->getClientOriginalExtension());
        return in_array($extension, $this->getSupportedExtensions());
    }
}


