<?php

namespace App\Services;

use App\Models\Design;
use Illuminate\Support\Facades\Log;

class DesignSimilarityService
{
    protected VertexAiService $vertexAi;

    public function __construct()
    {
        $this->vertexAi = new VertexAiService();
    }

    /**
     * Check if a description is similar to existing designs for a customer
     * 
     * @param string $description The new design description to check
     * @param int|null $customerId The customer to check designs for
     * @param int $limit Maximum number of existing designs to compare against
     * @return array Array with 'similar' (bool) and 'matches' (array of similar designs)
     */
    public function checkSimilarity(string $description, ?int $customerId = null, int $limit = 100): array
    {
        if (empty(trim($description)) || strlen(trim($description)) < 3) {
            return ['similar' => false, 'matches' => []];
        }

        try {
            // Get existing designs for this customer
            $existingDesigns = Design::query()
                ->when($customerId, fn($q) => $q->where('customers_id', $customerId))
                ->whereNotNull('description')
                ->where('description', '!=', '')
                ->select('id', 'description')
                ->orderBy('id', 'desc')
                ->limit($limit)
                ->get();

            if ($existingDesigns->isEmpty()) {
                return ['similar' => false, 'matches' => []];
            }

            // Format existing designs for the prompt
            $designsList = $existingDesigns->map(function ($design) {
                return "{$design->id}: {$design->description}";
            })->implode("\n");

            $prompt = <<<EOT
You are a duplicate detection system. Your job is to find CLEAR DUPLICATES only.

STRICT MATCHING RULES:
- Only flag if descriptions are nearly IDENTICAL (same words, minor variations)
- Match: "Jumper LS Rib Raglan" vs "Jumper Ls Rib Raglan" (same thing, different case)
- Match: "Cable Knit Sweater" vs "Cable Knit Sweater V2" (same base product)
- Match: "Rib Crew Neck" vs "Crew Neck Rib" (same words reordered)
- NO MATCH: "Jumper Rib" vs "Cardigan Rib" (different product types)
- NO MATCH: "Cable Knit" vs "Rib Knit" (different knit styles)
- NO MATCH: "Crew Neck Jumper" vs "V Neck Jumper" (different necklines)

DO NOT match products just because they share common fashion words like: jumper, cardigan, sweater, knit, rib, crew, neck, sleeve, etc.

NEW DESCRIPTION: "{$description}"

EXISTING DESIGNS:
{$designsList}

If there are CLEAR DUPLICATES (nearly identical descriptions), respond with JSON:
{"similar": true, "matches": [{"id": 123, "description": "...", "reason": "brief reason"}]}

Return just the numeric ID (e.g. 123, not "RT123").

If no clear duplicates found (which should be most cases), respond with:
{"similar": false, "matches": []}
EOT;

            $result = $this->vertexAi->generateText($prompt);
            
            if ($result && isset($result['similar'])) {
                return [
                    'similar' => (bool) $result['similar'],
                    'matches' => $result['matches'] ?? []
                ];
            }

            Log::warning('DesignSimilarityService: Unexpected response structure', [
                'result' => $result,
                'description' => $description
            ]);

            return ['similar' => false, 'matches' => []];

        } catch (\Exception $e) {
            Log::error('DesignSimilarityService error: ' . $e->getMessage(), [
                'description' => $description,
                'customer_id' => $customerId
            ]);
            
            return ['similar' => false, 'matches' => [], 'error' => $e->getMessage()];
        }
    }
}

