<?php

namespace App\Services;

use App\Models\CommodityCodes;
use App\Models\Styles;
use Illuminate\Support\Facades\Log;

class CommodityCodeSuggestionService
{
    protected VertexAiService $vertexAi;

    public function __construct()
    {
        $this->vertexAi = new VertexAiService();
    }

    /**
     * Suggest a commodity code for a style using AI
     */
    public function suggestForStyle(Styles $style): ?int
    {
        try {
            $styleData = $this->prepareStyleData($style);
            $commodityCodes = $this->getCommodityCodes();

            if (empty($commodityCodes)) {
                Log::warning('CommodityCodeSuggestion: No commodity codes in database');
                return null;
            }

            $suggestedId = $this->getSuggestion($styleData, $commodityCodes);

            if ($suggestedId) {
                // Verify the suggested ID exists
                $exists = CommodityCodes::where('id', $suggestedId)->exists();
                if ($exists) {
                    return $suggestedId;
                }
                Log::warning("CommodityCodeSuggestion: Suggested ID {$suggestedId} not found in database");
            }

            return null;
        } catch (\Exception $e) {
            Log::error('CommodityCodeSuggestion: Error suggesting commodity code', [
                'style_id' => $style->id,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }

    /**
     * Suggest and update a style's commodity code
     */
    public function suggestAndUpdate(Styles $style): bool
    {
        $suggestedId = $this->suggestForStyle($style);

        if ($suggestedId) {
            $style->commodity_codes_id = $suggestedId;
            $style->saveQuietly(); // Don't trigger events
            
            Log::info('CommodityCodeSuggestion: Updated commodity code', [
                'style_id' => $style->id,
                'commodity_code_id' => $suggestedId,
            ]);
            
            return true;
        }

        return false;
    }

    protected function prepareStyleData(Styles $style): array
    {
        // Load relationships if not already loaded
        $style->loadMissing([
            'designs',
            'customers',
            'departments',
            'seasons',
            'style_versions.colourways.colourway_yarns.yarn_colours.yarn.counts',
            'style_versions.colourways.colourway_yarns.yarn_colours.yarn.yarn_compositions.materials',
        ]);

        $yarns = [];
        $compositions = [];

        foreach ($style->style_versions as $version) {
            foreach ($version->colourways as $colourway) {
                foreach ($colourway->colourway_yarns as $colourwayYarn) {
                    $yarnColour = $colourwayYarn->yarn_colours;
                    if ($yarnColour && $yarnColour->yarn) {
                        $yarn = $yarnColour->yarn;
                        $count = $yarn->counts?->count ?? 'N/A';
                        $yarnDesc = "{$yarn->description} ({$count})";

                        if (!in_array($yarnDesc, $yarns)) {
                            $yarns[] = $yarnDesc;
                        }

                        foreach ($yarn->yarn_compositions ?? [] as $comp) {
                            $materialName = $comp->materials?->description ?? '';
                            $percentage = $comp->percentage ?? '';
                            $compDesc = trim("{$materialName} {$percentage}%");
                            if (!in_array($compDesc, $compositions) && $compDesc !== '%') {
                                $compositions[] = $compDesc;
                            }
                        }
                    }
                }
            }
        }

        // Fallback to design's yarn field if no colourway yarns found
        if (empty($yarns) && $style->designs?->yarn) {
            $yarns[] = $style->designs->yarn;
        }

        return [
            'style_id' => $style->id,
            'design_id' => $style->designs_id,
            'design_description' => $style->designs?->description ?? 'N/A',
            'customer_ref' => $style->customer_ref ?? 'N/A',
            'customer' => $style->customers?->name ?? 'N/A',
            'category' => $style->category ?? 'N/A',
            'department' => $style->departments?->description ?? 'N/A',
            'season' => $style->seasons?->description ?? 'N/A',
            'yarns' => array_unique($yarns),
            'compositions' => array_unique($compositions),
        ];
    }

    protected function getCommodityCodes(): array
    {
        return CommodityCodes::withoutGlobalScopes()
            ->select(['id', 'description', 'order', 'Gender', 'Type', 'Composition'])
            ->get()
            ->toArray();
    }

    protected function getSuggestion(array $styleData, array $commodityCodes): ?int
    {
        $prompt = $this->buildPrompt($styleData, $commodityCodes);

        $data = $this->vertexAi->generateText($prompt, 'gemini-2.5-flash-lite');

        if ($data && isset($data['commodity_code_id'])) {
            return (int) $data['commodity_code_id'];
        }

        return null;
    }

    protected function buildPrompt(array $styleData, array $commodityCodes): string
    {
        $commodityCodesJson = json_encode($commodityCodes, JSON_PRETTY_PRINT);
        $styleDataJson = json_encode($styleData, JSON_PRETTY_PRINT);

        return <<<PROMPT
You are an expert in textile and garment classification for customs purposes. Your task is to select the most appropriate commodity code for a garment based on its characteristics.

## Garment Information:
{$styleDataJson}

## Available Commodity Codes:
{$commodityCodesJson}

## Instructions:
1. Analyze the garment information including the design description, yarns used, composition, category (mens/ladies/accessories/childrens), and any other relevant details.
2. Match the garment to the most appropriate commodity code from the list provided.
3. Consider:
   - The type of garment (jumper, cardigan, coat, dress, etc.)
   - The material composition (wool, cotton, synthetic, etc.)
   - Gender category (mens, ladies, childrens)
   - Any specific characteristics mentioned

## Response Format:
Return ONLY a valid JSON object with no additional text:
{
    "commodity_code_id": <the ID of the best matching commodity code>
}
PROMPT;
    }
}

