<?php

namespace App\Services\CommissionImports;

use App\Models\Design;
use App\Models\Styles;
use App\Models\Colourways;
use App\Models\StyleVersions;
use App\Models\Suppliers;
use App\Models\Gauge;
use Illuminate\Support\Carbon;

class StyleMatchingService
{
    /**
     * Find existing style or create new one based on style number
     */
    public function findOrCreateStyle(
        string $styleNumber,
        int $customerId,
        int $seasonId,
        int $departmentId,
        array $styleData = []
    ): Styles {
        // Step 1: Check for existing style in current season
        $existingStyle = $this->findExistingStyle($styleNumber, $customerId, $seasonId, $departmentId);
        if ($existingStyle) {
            return $existingStyle;
        }

        // Step 2: Check for style in previous seasons
        $previousSeasonStyle = $this->findStyleInPreviousSeasons($styleNumber, $customerId, $departmentId);
        if ($previousSeasonStyle) {
            return $this->createStyleFromExisting($previousSeasonStyle, $seasonId, $styleData);
        }

        // Step 3: Create completely new style with new design
        return $this->createNewStyleWithDesign($styleNumber, $customerId, $seasonId, $departmentId, $styleData);
    }

    /**
     * Find existing style in current season
     */
    public function findExistingStyle(
        string $styleNumber,
        int $customerId,
        int $seasonId,
        int $departmentId
    ): ?Styles {
        return Styles::where('customer_ref', $styleNumber)
            ->where('customers_id', $customerId)
            ->where('seasons_id', $seasonId)
            ->where('departments_id', $departmentId)
            ->first();
    }

    /**
     * Find style in previous seasons
     */
    public function findStyleInPreviousSeasons(
        string $styleNumber,
        int $customerId,
        int $departmentId
    ): ?Styles {
        return Styles::where('customer_ref', $styleNumber)
            ->where('customers_id', $customerId)
            ->where('departments_id', $departmentId)
            ->orderBy('created_at', 'desc')
            ->first();
    }

    /**
     * Create style from existing style (new season)
     */
    public function createStyleFromExisting(
        Styles $existingStyle,
        int $newSeasonId,
        array $styleData = []
    ): Styles {
        $newStyle = Styles::create([
            'designs_id' => $existingStyle->designs_id,
            'customers_id' => $existingStyle->customers_id,
            'seasons_id' => $newSeasonId,
            'departments_id' => $existingStyle->departments_id,
            'customer_ref' => $existingStyle->customer_ref,
            'commodity_codes_id' => $existingStyle->commodity_codes_id,
            'category' => $existingStyle->category,
            'notes' => $styleData['notes'] ?? $existingStyle->notes,
        ]);

        // Clone style versions from existing style
        $this->cloneStyleVersions($existingStyle, $newStyle);

        return $newStyle;
    }

    /**
     * Create completely new style with new design
     */
    public function createNewStyleWithDesign(
        string $styleNumber,
        int $customerId,
        int $seasonId,
        int $departmentId,
        array $styleData = []
    ): Styles {
        // Create new design
        $design = $this->createDesign($customerId, $styleData);

        // Create new style
        $style = Styles::create([
            'designs_id' => $design->id,
            'customers_id' => $customerId,
            'seasons_id' => $seasonId,
            'departments_id' => $departmentId,
            'customer_ref' => $styleNumber,
            'commodity_codes_id' => $styleData['commodity_codes_id'] ?? null,
            'category' => $styleData['category'] ?? 'ladies',
            'notes' => $styleData['notes'] ?? '',
        ]);

        // Create default style version
        $this->createDefaultStyleVersion($style, $design);

        return $style;
    }

    /**
     * Find or create colourway for style
     */
    public function findOrCreateColourway(
        string $colourName,
        StyleVersions $styleVersion,
        array $colourwayData = []
    ): Colourways {
        // Look for existing colourway
        $existingColourway = Colourways::where('style_versions_id', $styleVersion->id)
            ->where('name', 'like', '%' . $colourName . '%')
            ->first();

        if ($existingColourway) {
            return $existingColourway;
        }

        // Create new colourway
        return Colourways::create([
            'style_versions_id' => $styleVersion->id,
            'name' => $colourName,
            'colour_type' => $colourwayData['colour_type'] ?? 'solid',
            'composition' => $colourwayData['composition'] ?? '',
            'customer_description' => $colourwayData['customer_description'] ?? '',
        ]);
    }

    /**
     * Create new design
     */
    private function createDesign(int $customerId, array $styleData): Design
    {
        $factoryId = null;
        if (!empty($styleData['factory_name'])) {
            $factory = Suppliers::where('name', $styleData['factory_name'])->first();
            $factoryId = $factory?->id;
        }
        // Default factory to Erdos (ID 5) when not provided
        if ($factoryId === null) {
            $factoryId = 5;
        }

        $gaugeId = null;
        if (!empty($styleData['gauge'])) {
            $gauge = Gauge::firstOrCreate(['gauge' => $styleData['gauge']]);
            $gaugeId = $gauge->id;
        }
        // Default gauge to ID 24 (ERDOS) when not provided
        if ($gaugeId === null) {
            $gaugeId = 24;
        }

        return Design::create([
            'factory_id' => $factoryId,
            'customers_id' => $customerId,
            'description' => $styleData['description'] ?? 'Imported Design',
            'gauge_id' => $gaugeId,
            'issued' => !empty($styleData['date_of_issue']) 
                ? Carbon::parse($styleData['date_of_issue'])->format('Y-m-d')
                : Carbon::now()->format('Y-m-d'),
            'yarn' => $styleData['composition'] ?? '',
        ]);
    }

    /**
     * Create default style version for new style
     */
    private function createDefaultStyleVersion(Styles $style, Design $design): StyleVersions
    {
        return StyleVersions::create([
            'styles_id' => $style->id,
            'name' => 'V1',
            'factory_id' => $design->factory_id,
            'gauge_id' => $design->gauge_id,
        ]);
    }

    /**
     * Clone style versions from existing style to new style
     */
    private function cloneStyleVersions(Styles $existingStyle, Styles $newStyle): void
    {
        foreach ($existingStyle->style_versions as $existingVersion) {
            StyleVersions::create([
                'styles_id' => $newStyle->id,
                'name' => $existingVersion->name,
                'factory_id' => $existingVersion->factory_id,
                'gauge_id' => $existingVersion->gauge_id,
            ]);
        }
    }

    /**
     * Get or create the first style version for a style
     */
    public function getFirstStyleVersion(Styles $style): StyleVersions
    {
        $styleVersion = $style->style_versions()->first();
        
        if (!$styleVersion) {
            $styleVersion = $this->createDefaultStyleVersion($style, $style->designs);
        }

        return $styleVersion;
    }
}


