<?php

namespace App\Services\CommissionImports;

use App\Models\CommissionImportBatch;
use App\Models\CommissionImportBatchFile;
use App\Models\CommissionOrder;
use App\Models\CommissionOrderLine;
use App\Models\CommissionOrderLineQuantity;
use App\Models\Customer;
use App\Models\Styles;
use App\Models\Colourways;
use App\Models\Sizes;
use App\Services\VertexAiService;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class MultiAgentImportService
{
    protected VertexAiService $vertexAi;
    protected StyleMatchingService $styleMatchingService;

    // Default extraction prompt - can be overridden per customer
    protected string $defaultExtractionPrompt = <<<'PROMPT'
Extract all order information from this document. Return a JSON object with this exact structure:

{
  "orders": [
    {
      "po_number": "string - the purchase order number",
      "order_date": "YYYY-MM-DD format if available",
      "lines": [
        {
          "style_ref": "string - customer style reference/code",
          "style_description": "string - product description",
          "colour": "string - colour name",
          "quantities": [
            {
              "size": "string - size name (XS, S, M, L, XL, etc.)",
              "qty": number
            }
          ],
          "unit_price": number or null
        }
      ]
    }
  ],
  "metadata": {
    "detected_season": "string if found",
    "currency": "string if found",
    "total_qty": number - sum of all quantities
  }
}

Important:
- Extract ALL lines from the document, do not skip any
- Match sizes exactly as they appear (e.g., "Small", "S", "SM" should be kept as-is)
- If a line spans multiple rows, combine them
- Include unit prices if visible
- Return valid JSON only, no markdown
PROMPT;

    public function __construct(VertexAiService $vertexAi, StyleMatchingService $styleMatchingService)
    {
        $this->vertexAi = $vertexAi;
        $this->styleMatchingService = $styleMatchingService;
    }

    /**
     * Process a batch through all three agents
     */
    public function processBatch(CommissionImportBatch $batch): void
    {
        Log::info("MultiAgent - Starting batch processing", ['batch_id' => $batch->id]);

        try {
            $batch->update([
                'status' => 'processing',
                'workflow_status' => 'pending',
                'started_at' => now(),
            ]);

            // Process each file in the batch
            $processedCount = 0;
            foreach ($batch->files as $file) {
                if ($file->status !== 'pending') {
                    $processedCount++;
                    continue;
                }

                try {
                    $this->processFile($batch, $file);
                    $processedCount++;
                    $batch->update(['processed_files' => $processedCount]);
                } catch (\Exception $e) {
                    Log::error("MultiAgent - File processing failed", [
                        'file_id' => $file->id,
                        'error' => $e->getMessage()
                    ]);
                    $file->markAsFailed($e->getMessage());
                    $processedCount++;
                    $batch->update(['processed_files' => $processedCount]);
                }
            }

            // Aggregate results and create skeleton order
            $this->createSkeletonOrder($batch);

            // Determine final batch status based on file results
            $files = $batch->files()->get();
            $totalFiles = $files->count();
            $failedFiles = $files->where('status', 'failed')->count();
            $successfulFiles = $totalFiles - $failedFiles;

            if ($failedFiles === 0) {
                // All files processed successfully
                $finalStatus = 'completed';
                $workflowStatus = 'verified';
            } elseif ($successfulFiles === 0) {
                // All files failed
                $finalStatus = 'failed';
                $workflowStatus = 'failed';
            } else {
                // Partial success - some files failed
                $finalStatus = 'completed_with_warnings';
                $workflowStatus = 'verified';
            }

            $batch->update([
                'status' => $finalStatus,
                'workflow_status' => $workflowStatus,
                'completed_at' => now(),
                'successful_files' => $successfulFiles,
                'failed_files' => $failedFiles,
            ]);

            Log::info("MultiAgent - Batch processing complete", [
                'batch_id' => $batch->id,
                'status' => $finalStatus,
                'successful_files' => $successfulFiles,
                'failed_files' => $failedFiles,
            ]);

        } catch (\Exception $e) {
            Log::error("MultiAgent - Batch processing failed", [
                'batch_id' => $batch->id,
                'error' => $e->getMessage()
            ]);

            $batch->update([
                'status' => 'failed',
                'workflow_status' => 'failed',
            ]);

            throw $e;
        }
    }

    /**
     * Process a single file through all agents
     */
    protected function processFile(CommissionImportBatch $batch, CommissionImportBatchFile $file): void
    {
        Log::info("MultiAgent - Processing file", [
            'file_id' => $file->id,
            'filename' => $file->original_filename
        ]);

        $file->update(['status' => 'processing']);

        // Agent 1: Extract data from file (with automatic retry if no lines found)
        $extractedData = $this->runExtractionAgent($batch, $file);
        
        if (!$extractedData) {
            throw new \RuntimeException("Agent 1 failed: No data extracted");
        }

        // Check if we got any lines - if not, retry once
        $totalLines = $this->countTotalLines($extractedData);
        if ($totalLines === 0) {
            Log::warning("MultiAgent - No lines extracted, retrying extraction", [
                'file_id' => $file->id,
                'attempt' => 1,
            ]);
            
            // Wait briefly before retry to avoid rate limiting
            sleep(2);
            
            $retryData = $this->runExtractionAgent($batch, $file);
            if ($retryData && $this->countTotalLines($retryData) > 0) {
                $extractedData = $retryData;
                Log::info("MultiAgent - Retry successful, lines found", [
                    'file_id' => $file->id,
                    'lines_found' => $this->countTotalLines($retryData),
                ]);
            } else {
                Log::warning("MultiAgent - Retry also returned no lines, proceeding with empty result", [
                    'file_id' => $file->id,
                ]);
            }
        }

        $file->update([
            'extracted_data' => $extractedData,
            'extracted_at' => now(),
        ]);

        // Agent 2: Match lines to styles/colourways
        $matchedLines = $this->runMatchingAgent($batch, $extractedData);
        
        $file->update([
            'matched_lines' => $matchedLines,
        ]);

        // Agent 3: Verify the data
        $verificationResult = $this->runVerificationAgent($extractedData, $matchedLines);
        
        $file->update([
            'verification_issues' => $verificationResult['issues'] ?? [],
            'status' => 'extracted',
            'total_orders' => count($extractedData['orders'] ?? []),
            'total_lines' => $this->countTotalLines($extractedData),
            'total_quantity' => $extractedData['metadata']['total_qty'] ?? 0,
        ]);

        Log::info("MultiAgent - File processed successfully", [
            'file_id' => $file->id,
            'lines_matched' => collect($matchedLines)->where('match_status', '!=', 'unmatched')->count(),
            'lines_unmatched' => collect($matchedLines)->where('match_status', 'unmatched')->count(),
        ]);
    }

    /**
     * Agent 1: Extract PO data from file using Gemini Pro
     */
    protected function runExtractionAgent(CommissionImportBatch $batch, CommissionImportBatchFile $file): ?array
    {
        Log::info("Agent 1 - Extraction starting", ['file_id' => $file->id]);

        $filePath = Storage::path($file->file_path);
        $extension = strtolower(pathinfo($file->original_filename, PATHINFO_EXTENSION));
        $mimeType = VertexAiService::getMimeType($extension);

        // Get customer-specific prompt or use default
        $prompt = $this->getExtractionPrompt($batch->customer);

        try {
            $result = $this->vertexAi->extractFromFile($filePath, $prompt, $mimeType);
            
            // Normalize the response structure - Gemini sometimes returns flat structure
            $result = $this->normalizeExtractionResult($result);
            
            Log::info("Agent 1 - Extraction complete", [
                'file_id' => $file->id,
                'orders_found' => count($result['orders'] ?? []),
            ]);

            return $result;

        } catch (\Exception $e) {
            Log::error("Agent 1 - Extraction failed", [
                'file_id' => $file->id,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Normalize extraction result to expected structure
     * Gemini sometimes returns flat structure instead of orders array
     */
    protected function normalizeExtractionResult(array $result): array
    {
        // If already has 'orders' array, return as-is
        if (isset($result['orders']) && is_array($result['orders'])) {
            return $result;
        }

        // If has 'lines' at root level, wrap in orders array
        if (isset($result['lines']) && is_array($result['lines'])) {
            $order = [
                'po_number' => $result['po_number'] ?? null,
                'order_date' => $result['order_date'] ?? null,
                'lines' => $result['lines'],
            ];

            return [
                'orders' => [$order],
                'metadata' => $result['metadata'] ?? [],
            ];
        }

        // Unknown structure, return with empty orders
        Log::warning("Unexpected extraction result structure", ['keys' => array_keys($result)]);
        return [
            'orders' => [],
            'metadata' => $result['metadata'] ?? [],
        ];
    }

    /**
     * Agent 2: Match each line to styles/colourways in database using Gemini Flash Lite
     */
    protected function runMatchingAgent(CommissionImportBatch $batch, array $extractedData): array
    {
        Log::info("Agent 2 - Matching starting", ['batch_id' => $batch->id]);

        $matchedLines = [];
        $customerId = $batch->customer_id;
        $seasonId = $batch->season_id;
        $departmentId = $batch->department_id;

        // Get all available styles for this customer/season
        $availableStyles = $this->getAvailableStylesForMatching($customerId, $seasonId, $departmentId);

        foreach ($extractedData['orders'] ?? [] as $orderIndex => $order) {
            foreach ($order['lines'] ?? [] as $lineIndex => $line) {
                $matchResult = $this->matchLineToStyle($line, $availableStyles, $customerId, $seasonId);
                
                $matchedLines[] = [
                    'order_index' => $orderIndex,
                    'line_index' => $lineIndex,
                    'po_number' => $order['po_number'] ?? null,
                    'imported_style_ref' => $line['style_ref'] ?? null,
                    'imported_description' => $line['style_description'] ?? null,
                    'imported_colour' => $line['colour'] ?? null,
                    'quantities' => $line['quantities'] ?? [],
                    'unit_price' => $line['unit_price'] ?? null,
                    'match_status' => $matchResult['status'],
                    'matched_colourway_id' => $matchResult['colourway_id'] ?? null,
                    'matched_style_id' => $matchResult['style_id'] ?? null,
                    'match_confidence' => $matchResult['confidence'] ?? 0,
                    'match_reason' => $matchResult['reason'] ?? null,
                ];
            }
        }

        Log::info("Agent 2 - Matching complete", [
            'batch_id' => $batch->id,
            'total_lines' => count($matchedLines),
            'matched' => collect($matchedLines)->where('match_status', '!=', 'unmatched')->count(),
        ]);

        return $matchedLines;
    }

    /**
     * Match a single line to a style/colourway
     */
    protected function matchLineToStyle(array $line, array $availableStyles, int $customerId, int $seasonId): array
    {
        $styleRef = $line['style_ref'] ?? '';
        $colour = $line['colour'] ?? '';
        $description = $line['style_description'] ?? '';

        // Try exact match first
        foreach ($availableStyles as $style) {
            // Match by customer ref
            if ($styleRef && strcasecmp($style['customer_ref'], $styleRef) === 0) {
                // Found style, now find colourway
                foreach ($style['colourways'] as $cw) {
                    if ($colour && $this->isColourMatch($cw['name'], $colour)) {
                        return [
                            'status' => 'ai_matched',
                            'colourway_id' => $cw['id'],
                            'style_id' => $style['id'],
                            'confidence' => 95,
                            'reason' => 'Exact style ref and colour match',
                        ];
                    }
                }
                
                // Style matched but no colourway - return partial match
                if (!empty($style['colourways'])) {
                    return [
                        'status' => 'ai_matched',
                        'colourway_id' => null,
                        'style_id' => $style['id'],
                        'confidence' => 70,
                        'reason' => "Style matched, colour '{$colour}' not found",
                    ];
                }
            }
        }

        // Try fuzzy match using AI for complex cases
        $aiMatch = $this->tryAiStyleMatch($line, $availableStyles);
        if ($aiMatch) {
            return $aiMatch;
        }

        // No match found
        return [
            'status' => 'unmatched',
            'colourway_id' => null,
            'style_id' => null,
            'confidence' => 0,
            'reason' => 'No matching style found in database',
        ];
    }

    /**
     * Use AI to attempt fuzzy matching
     */
    protected function tryAiStyleMatch(array $line, array $availableStyles): ?array
    {
        // Only use AI matching if we have styles to match against
        if (empty($availableStyles)) {
            return null;
        }

        // Limit styles for prompt size
        $stylesForPrompt = array_slice($availableStyles, 0, 50);

        $prompt = "Match this line to the best style from the database. Return JSON only.

Line to match:
- Style ref: " . ($line['style_ref'] ?? 'not provided') . "
- Description: " . ($line['style_description'] ?? 'not provided') . "
- Colour: " . ($line['colour'] ?? 'not provided') . "

Available styles:
" . json_encode($stylesForPrompt, JSON_PRETTY_PRINT) . "

Return format:
{
  \"matched\": true/false,
  \"style_id\": number or null,
  \"colourway_id\": number or null,
  \"confidence\": 0-100,
  \"reason\": \"explanation\"
}";

        try {
            $result = $this->vertexAi->generateText($prompt, 'gemini-2.5-flash-lite');
            
            if ($result && isset($result['matched']) && $result['matched']) {
                return [
                    'status' => 'ai_matched',
                    'colourway_id' => $result['colourway_id'] ?? null,
                    'style_id' => $result['style_id'] ?? null,
                    'confidence' => $result['confidence'] ?? 50,
                    'reason' => $result['reason'] ?? 'AI fuzzy match',
                ];
            }
        } catch (\Exception $e) {
            Log::warning("AI matching failed, continuing without", ['error' => $e->getMessage()]);
        }

        return null;
    }

    /**
     * Check if two colour names are a match
     */
    protected function isColourMatch(string $dbColour, string $importColour): bool
    {
        $db = strtolower(trim($dbColour));
        $import = strtolower(trim($importColour));

        // Exact match
        if ($db === $import) {
            return true;
        }

        // Contains match
        if (str_contains($db, $import) || str_contains($import, $db)) {
            return true;
        }

        // Levenshtein distance for typos
        if (levenshtein($db, $import) <= 2) {
            return true;
        }

        return false;
    }

    /**
     * Agent 3: Verify the extracted and matched data using Gemini Flash
     */
    protected function runVerificationAgent(array $extractedData, array $matchedLines): array
    {
        Log::info("Agent 3 - Verification starting");

        $issues = [];

        // Check for obvious data issues
        $totalQty = $extractedData['metadata']['total_qty'] ?? 0;
        $calculatedQty = 0;
        foreach ($matchedLines as $line) {
            foreach ($line['quantities'] ?? [] as $qty) {
                $calculatedQty += (int)($qty['qty'] ?? 0);
            }
        }

        if ($totalQty > 0 && abs($totalQty - $calculatedQty) > 0) {
            $issues[] = [
                'type' => 'quantity_mismatch',
                'severity' => 'warning',
                'message' => "Total quantity mismatch: extracted {$totalQty}, calculated {$calculatedQty}",
            ];
        }

        // Check for unmatched lines
        $unmatchedCount = collect($matchedLines)->where('match_status', 'unmatched')->count();
        if ($unmatchedCount > 0) {
            $issues[] = [
                'type' => 'unmatched_lines',
                'severity' => 'info',
                'message' => "{$unmatchedCount} lines could not be matched to existing styles",
            ];
        }

        // Check for low confidence matches
        $lowConfidenceCount = collect($matchedLines)
            ->where('match_status', 'ai_matched')
            ->where('match_confidence', '<', 80)
            ->count();
        
        if ($lowConfidenceCount > 0) {
            $issues[] = [
                'type' => 'low_confidence',
                'severity' => 'warning',
                'message' => "{$lowConfidenceCount} lines matched with low confidence - verify these",
            ];
        }

        Log::info("Agent 3 - Verification complete", [
            'issues_found' => count($issues),
        ]);

        return [
            'valid' => count(array_filter($issues, fn($i) => $i['severity'] === 'error')) === 0,
            'issues' => $issues,
        ];
    }

    /**
     * Create skeleton orders from batch results - one order per PO number
     */
    protected function createSkeletonOrder(CommissionImportBatch $batch): CommissionOrder
    {
        Log::info("Creating skeleton orders from batch", ['batch_id' => $batch->id]);

        // Group lines by PO number, tracking which files contributed to each PO
        $linesByPo = [];
        $filesByPo = [];
        
        foreach ($batch->files as $file) {
            if (!$file->matched_lines) continue;
            
            foreach ($file->matched_lines as $line) {
                $poNumber = $line['po_number'] ?? 'UNKNOWN';
                
                // Group the line under this PO
                if (!isset($linesByPo[$poNumber])) {
                    $linesByPo[$poNumber] = [];
                    $filesByPo[$poNumber] = [];
                }
                
                $linesByPo[$poNumber][] = $line;
                
                // Track which files contributed to this PO
                if (!in_array($file->id, $filesByPo[$poNumber])) {
                    $filesByPo[$poNumber][] = $file->id;
                }
            }
        }

        Log::info("Found POs in batch", [
            'batch_id' => $batch->id,
            'po_count' => count($linesByPo),
            'po_numbers' => array_keys($linesByPo),
        ]);

        $firstOrder = null;
        $orderIds = [];

        // Create one order per PO
        foreach ($linesByPo as $poNumber => $lines) {
            $order = $this->createOrderForPo($batch, $poNumber, $lines, $filesByPo[$poNumber]);
            $orderIds[] = $order->id;
            
            if ($firstOrder === null) {
                $firstOrder = $order;
            }
        }

        // Link batch to first order (for backwards compatibility)
        // Store all order IDs in batch metadata for reference
        $batch->update([
            'commission_order_id' => $firstOrder?->id,
        ]);

        Log::info("Skeleton orders created", [
            'batch_id' => $batch->id,
            'orders_created' => count($orderIds),
            'order_ids' => $orderIds,
        ]);

        return $firstOrder ?? $this->createEmptyOrder($batch);
    }

    /**
     * Create a single order for a specific PO number
     */
    protected function createOrderForPo(
        CommissionImportBatch $batch, 
        string $poNumber, 
        array $lines,
        array $fileIds
    ): CommissionOrder {
        // Get the source files for this PO
        $sourceFiles = $batch->files->whereIn('id', $fileIds);
        $sourceFileNames = $sourceFiles->pluck('original_filename')->implode(', ');

        // Create the commission order
        $order = CommissionOrder::create([
            'commission_import_batch_id' => $batch->id,
            'customers_id' => $batch->customer_id,
            'seasons_id' => $batch->season_id,
            'departments_id' => $batch->department_id,
            'status' => 'skeleton',
            'customer_po' => $poNumber !== 'UNKNOWN' ? $poNumber : $batch->batch_name,
            'order_date' => now(),
            'source_file' => $sourceFileNames ?: $batch->batch_name,
            'imported_by' => $batch->user_id,
            'imported_at' => now(),
        ]);

        Log::info("Creating order for PO", [
            'order_id' => $order->id,
            'po_number' => $poNumber,
            'line_count' => count($lines),
            'source_files' => $sourceFileNames,
        ]);

        // Create order lines with quantities
        foreach ($lines as $lineData) {
            $this->createOrderLine($order, $lineData);
        }

        // Attach the specific source files for this PO (don't fail if this errors)
        try {
            $this->attachSpecificFilesToOrder($order, $sourceFiles);
        } catch (\Exception $e) {
            Log::error("Failed to attach files to order, continuing anyway", [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
            ]);
        }

        return $order;
    }

    /**
     * Create an order line with its quantities
     */
    protected function createOrderLine(CommissionOrder $order, array $lineData): void
    {
        $line = CommissionOrderLine::create([
            'commission_orders_id' => $order->id,
            'colourways_id' => $lineData['matched_colourway_id'],
            'imported_style_ref' => $lineData['imported_style_ref'],
            'imported_style_description' => $lineData['imported_description'],
            'imported_colour' => $lineData['imported_colour'],
            'match_status' => $lineData['match_status'],
            'ai_suggested_colourway_id' => $lineData['matched_colourway_id'],
            'ai_match_confidence' => $lineData['match_confidence'],
            'imported_raw_data' => $lineData,
        ]);

        // Create quantity records
        $quantities = $lineData['quantities'] ?? [];
        $unitPrice = $lineData['unit_price'] ?? 0;
        
        foreach ($quantities as $qty) {
            $sizeName = (string) ($qty['size'] ?? '');
            $qtyValue = (int) ($qty['qty'] ?? 0);
            
            if (!$sizeName || $qtyValue <= 0) continue;
            
            try {
                $size = Sizes::firstOrCreate(
                    ['name' => $sizeName],
                    ['name' => $sizeName]
                );
                
                CommissionOrderLineQuantity::create([
                    'commission_order_lines_id' => $line->id,
                    'sizes_id' => $size->id,
                    'qty' => $qtyValue,
                    'price' => $unitPrice,
                ]);
            } catch (\Exception $e) {
                Log::error("Failed to create quantity", [
                    'line_id' => $line->id,
                    'size' => $sizeName,
                    'error' => $e->getMessage(),
                ]);
            }
        }
    }

    /**
     * Create an empty order when no POs were found
     */
    protected function createEmptyOrder(CommissionImportBatch $batch): CommissionOrder
    {
        return CommissionOrder::create([
            'commission_import_batch_id' => $batch->id,
            'customers_id' => $batch->customer_id,
            'seasons_id' => $batch->season_id,
            'departments_id' => $batch->department_id,
            'status' => 'skeleton',
            'customer_po' => $batch->batch_name,
            'order_date' => now(),
            'source_file' => $batch->batch_name,
            'imported_by' => $batch->user_id,
            'imported_at' => now(),
        ]);
    }

    /**
     * Attach specific files to an order
     */
    protected function attachSpecificFilesToOrder(CommissionOrder $order, $files): void
    {
        $permanentDir = "commission-orders/{$order->id}";
        $filePaths = [];

        // Ensure the directory exists
        Storage::disk('local')->makeDirectory($permanentDir);

        foreach ($files as $file) {
            if (!Storage::disk('local')->exists($file->file_path)) {
                Log::warning("File not found when attaching to order", [
                    'file_id' => $file->id,
                    'path' => $file->file_path,
                ]);
                continue;
            }

            // Copy file to permanent location
            $newPath = "{$permanentDir}/{$file->original_filename}";
            
            // Handle duplicate filenames
            $counter = 1;
            $baseName = pathinfo($file->original_filename, PATHINFO_FILENAME);
            $extension = pathinfo($file->original_filename, PATHINFO_EXTENSION);
            while (Storage::disk('local')->exists($newPath)) {
                $newPath = "{$permanentDir}/{$baseName}_{$counter}.{$extension}";
                $counter++;
            }

            Storage::disk('local')->copy($file->file_path, $newPath);
            $filePaths[] = $newPath;

            Log::info("File attached to order", [
                'order_id' => $order->id,
                'file' => $file->original_filename,
                'path' => $newPath,
            ]);
        }

        if (!empty($filePaths)) {
            $order->update([
                'source_file_path' => json_encode($filePaths),
            ]);
        }
    }

    /**
     * Get available styles for matching
     */
    protected function getAvailableStylesForMatching(?int $customerId, ?int $seasonId, ?int $departmentId): array
    {
        $query = Styles::with(['style_versions.colourways', 'designs'])
            ->whereHas('departments', fn($q) => $q->where('description', 'like', '%ERDOS%'));

        if ($customerId) {
            $query->where('customers_id', $customerId);
        }

        if ($seasonId) {
            $query->where('seasons_id', $seasonId);
        }

        return $query->get()->map(function ($style) {
            return [
                'id' => $style->id,
                'customer_ref' => $style->customer_ref,
                'design_id' => $style->designs?->id,
                'description' => $style->designs?->description,
                'colourways' => $style->style_versions->flatMap(function ($sv) {
                    return $sv->colourways->map(function ($cw) {
                        return [
                            'id' => $cw->id,
                            'name' => $cw->name,
                        ];
                    });
                })->toArray(),
            ];
        })->toArray();
    }

    /**
     * Get extraction prompt for customer
     * Default prompt is always used, customer-specific instructions are appended
     */
    protected function getExtractionPrompt(?Customer $customer): string
    {
        $basePrompt = config('services.commission.default_extraction_prompt', $this->defaultExtractionPrompt);

        // Append customer-specific instructions if they exist
        if ($customer && !empty($customer->commission_gemini_instructions)) {
            $basePrompt .= "\n\n--- CUSTOMER-SPECIFIC INSTRUCTIONS FOR " . strtoupper($customer->name) . " ---\n";
            $basePrompt .= $customer->commission_gemini_instructions;
        }

        return $basePrompt;
    }

    /**
     * Count total lines in extracted data
     */
    protected function countTotalLines(array $extractedData): int
    {
        $count = 0;
        foreach ($extractedData['orders'] ?? [] as $order) {
            $count += count($order['lines'] ?? []);
        }
        return $count;
    }
}

