<?php

namespace App\Services\CommissionImports\Extractors;

use App\Services\CommissionImports\BaseExtractor;
use App\Services\CommissionImports\TabulaService;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Log;
use Spatie\PdfToText\Pdf;

class PureExtractor extends BaseExtractor
{
    private TabulaService $tabulaService;

    public function __construct()
    {
        $this->tabulaService = new TabulaService();
    }

    public function getCustomerName(): string
    {
        return 'Pure Collection';
    }

    public function getCustomerId(): int
    {
        return 19; // Pure Collection customer ID from database
    }

    public function extractData(TemporaryUploadedFile $file): array
    {
        $extension = strtolower($file->getClientOriginalExtension());

        if ($extension === 'pdf') {
            return $this->extractFromPdf($file);
        } elseif (in_array($extension, ['xlsx', 'xls'])) {
            return $this->extractFromExcel($file);
        }

        throw new \InvalidArgumentException("Unsupported file type: {$extension}");
    }

    public function validateCustomerData(array $data): array
    {
        $errors = [];

        foreach ($data['orders'] ?? [] as $orderIndex => $order) {
            // Validate Pure-specific PO format (e.g., "PO Erdos 01.08.25.1")
            if (!empty($order['purchase_order_number'])) {
                if (!preg_match('/^PO\s+/i', $order['purchase_order_number'])) {
                    $errors[] = "Order {$orderIndex}: Pure PO numbers should start with 'PO'";
                }
            }

            // Validate Pure-specific style number format (e.g., LKZJ4)
            foreach ($order['lines'] ?? [] as $lineIndex => $line) {
                $styleNumber = $line['colourway']['style_number'] ?? '';
                if (!empty($styleNumber) && !preg_match('/^[A-Z]{4}\d{1,2}$/', $styleNumber)) {
                    $errors[] = "Order {$orderIndex}, Line {$lineIndex}: Pure style numbers should follow format 'LKZJ4'";
                }
            }
        }

        return $errors;
    }

    /**
     * Extract data from PDF using Tabula with Pure-specific parameters
     */
    private function extractFromPdf(TemporaryUploadedFile $file): array
    {
        // Extract text content from PDF first
        $fullText = Pdf::getText($file->getRealPath());

        // Get total number of pages and extract from all of them
        $pdfPath = $file->getRealPath();
        $pdfInfo = shell_exec("pdfinfo " . escapeshellarg($pdfPath) . " 2>&1");
        $totalPages = 1; // Default to 1 page

        if (preg_match('/Pages:\s+(\d+)/', $pdfInfo, $matches)) {
            $totalPages = (int)$matches[1];
        }

        // Build array of all page numbers [1, 2, 3, ...]
        $pages = range(1, $totalPages);
        $mode = 'stream'; // Pure documents work better with stream mode

        $rows = $this->tabulaService->extractTables($pdfPath, $pages, $mode);

        return $this->parsePdfTableData($rows, $fullText);
    }

    /**
     * Extract data from Excel with Pure-specific logic
     */
    private function extractFromExcel(TemporaryUploadedFile $file): array
    {
        $sheets = Excel::toArray(null, $file);
        $mainSheet = $sheets[0] ?? [];

        return $this->parseExcelData($mainSheet);
    }

    /**
     * Parse PDF table data into standardized format - Pure specific
     */
    private function parsePdfTableData(array $rows, string $fullText): array
    {
        if (empty($rows)) {
            throw new \RuntimeException('No table data found in PDF');
        }

        // Extract metadata from full text
        $season = $this->extractSeasonFromText($fullText);
        $poNumber = $this->extractPoNumber($fullText);
        $orderDate = $this->extractOrderDate($fullText);
        $exFactoryDate = $this->extractExFactoryDate($fullText);

        if (!$poNumber) {
            throw new \RuntimeException('Could not find purchase order number in PDF');
        }

        // Parse the table data - Pure format has a style header followed by color/size rows
        // All sizes for the same style/color should be grouped into ONE order line
        $orders = [];
        $currentStyleCode = '';
        $currentStyleName = '';
        $currentComposition = '';
        $styleColorLines = []; // Group lines by style+color

        foreach ($rows as $rowIndex => $row) {
            $firstColumn = trim($row[0] ?? '');

            // Skip empty rows
            if (empty($firstColumn)) {
                continue;
            }

            // Look for style code row (e.g., "LKZJ4")
            if (preg_match('/^[A-Z]{4}\d{1,2}$/', $firstColumn) && isset($row[1])) {
                // This is a style header row
                $currentStyleCode = $firstColumn;
                $currentStyleName = trim($row[1] ?? '');
                $currentComposition = trim($row[2] ?? '');
            } elseif (!empty($currentStyleCode) && $firstColumn !== 'Color' && count($row) >= 6) {
                // This is a color/size data row - collect it for grouping
                $colorName = trim($row[0] ?? '');
                $size = trim($row[1] ?? '');
                $itemNumber = trim($row[2] ?? '');
                $ean = trim($row[3] ?? '');
                $qty = (int)($row[4] ?? 0);
                $fob = trim($row[5] ?? '');

                if (!empty($colorName) && !empty($size) && $qty > 0 && !empty($fob)) {
                    $priceString = preg_replace('/[^0-9.]/', '', $fob);
                    $price = (float) $priceString;

                    if ($price > 0) {
                        $key = $currentStyleCode . '|' . strtoupper($colorName);

                        if (!isset($styleColorLines[$key])) {
                            $styleColorLines[$key] = [
                                'style_code' => $currentStyleCode,
                                'style_name' => $currentStyleName,
                                'color_name' => $colorName,
                                'quantities' => [],
                            ];
                        }

                        $styleColorLines[$key]['quantities'][] = [
                            'size' => $size,
                            'qty' => $qty,
                            'price' => $price,
                            'commission' => 0,
                            'discount' => 0,
                        ];
                    }
                }
            }
        }

        // Convert grouped lines into orders
        if (!empty($styleColorLines)) {
            $order = [
                'order_type' => 'commission',
                'incoterms' => 'FOB',
                'season' => $season,
                'order_date' => $orderDate,
                'purchase_order_number' => $poNumber,
                'lines' => [],
            ];

            foreach ($styleColorLines as $groupedLine) {
                $order['lines'][] = [
                    'colourway' => [
                        'style_number' => $this->normalizeStyleNumber($groupedLine['style_code']),
                        'colour_name' => $this->normalizeColourName($groupedLine['color_name']),
                        'product_description' => $groupedLine['style_name'],
                    ],
                    'customer_exfty_date' => $exFactoryDate ?: date('Y-m-d', strtotime('+30 days')),
                    'customer_into_wh' => date('Y-m-d', strtotime(($exFactoryDate ?: date('Y-m-d')) . ' +5 days')),
                    'quantities' => $groupedLine['quantities'],
                    'shipment_lines' => [
                        ['exfty' => $exFactoryDate ?: date('Y-m-d', strtotime('+30 days'))]
                    ],
                ];
            }

            $orders[] = $order;
        }

        if (empty($orders)) {
            throw new \RuntimeException('No valid orders found in PDF data');
        }

        return [
            'orders' => $orders,
            'metadata' => [
                'detected_season' => $season,
                'extraction_method' => 'tabula_pdf',
            ],
        ];
    }

    /**
     * Parse Excel data into standardized format - Pure specific
     */
    private function parseExcelData(array $rows): array
    {
        if (empty($rows)) {
            throw new \RuntimeException('No data found in Excel file');
        }

        $orders = [];
        $detectedSeason = null;

        // Extract season from document
        foreach (array_slice($rows, 0, 10) as $row) {
            $rowText = implode(' ', $row);
            $season = $this->extractSeasonFromText($rowText);
            if ($season) {
                $detectedSeason = $season;
                break;
            }
        }

        // TODO: Implement Pure-specific Excel parsing logic based on actual document structure

        return [
            'orders' => $orders,
            'metadata' => [
                'detected_season' => $detectedSeason,
                'extraction_method' => 'excel_sheets',
            ],
        ];
    }

    /**
     * Extract PO Number from document text (Pure specific)
     */
    private function extractPoNumber(string $text): ?string
    {
        // Pure uses "Purchase Order Number:" format
        $patterns = [
            '/Purchase Order Number:\s*(PO\s+[^\n]+)/i',
            '/PO Number:\s*(PO\s+[^\n]+)/i',
            '/Purchase Order:\s*(PO\s+[^\n]+)/i',
            '/PO\s+([A-Za-z0-9\s\.]+)/i', // Generic PO pattern
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                $poNumber = trim($matches[1]);

                // Validate that this looks like a legitimate PO number
                if (preg_match('/^PO\s+/i', $poNumber)) {
                    return $poNumber;
                }
            }
        }

        return null;
    }

    /**
     * Extract ex-factory date from document text (Pure specific)
     */
    private function extractExFactoryDate(string $text): ?string
    {
        $patterns = [
            '/Ex-Factory Date:\s*(\d{1,2}\/\d{1,2}\/\d{4})/',
            '/Ex Factory Date:\s*(\d{1,2}\/\d{1,2}\/\d{4})/',
            '/ExFactory Date:\s*(\d{1,2}\/\d{1,2}\/\d{4})/',
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                $dateString = trim($matches[1]);
                $parsedDate = $this->parseDate($dateString);
                if ($parsedDate) {
                    return $parsedDate;
                }
            }
        }

        return null;
    }

    /**
     * Extract order date from document text (Pure specific)
     */
    private function extractOrderDate(string $text): ?string
    {
        // Pure might use order date, printed date, or ex-factory date
        $patterns = [
            '/Order Date:\s*(\d{1,2}\/\d{1,2}\/\d{4})/',
            '/Date:\s*(\d{1,2}\/\d{1,2}\/\d{4})/',
            '/Printed:\s*(\d{1,2}\/\d{1,2}\/\d{4})/',
            '/(\d{1,2}\/\d{1,2}\/\d{4})/', // Any date format
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                $dateString = trim($matches[1]);
                $parsedDate = $this->parseDate($dateString);
                if ($parsedDate) {
                    return $parsedDate;
                }
            }
        }

        // If no date found, use today's date
        return date('Y-m-d');
    }

    /**
     * Parse individual Pure order line from table row
     */
    private function parsePureOrderLine(array $row, string $styleCode, string $styleName, ?string $exFactoryDate): ?array
    {
        // Pure table structure:
        // [0] = Color (Jade Green)
        // [1] = Size (8, 10, 12, etc.)
        // [2] = Item Number (469132)
        // [3] = EAN (5056836008326)
        // [4] = Quantity (21)
        // [5] = FOB (43.80)

        $colorName = trim($row[0] ?? '');
        $size = trim($row[1] ?? '');
        $itemNumber = trim($row[2] ?? '');
        $ean = trim($row[3] ?? '');
        $qty = (int)($row[4] ?? 0);
        $fob = trim($row[5] ?? '');

        if (empty($colorName) || empty($size) || $qty <= 0 || empty($fob)) {
            return null;
        }

        // Extract price from FOB string
        $priceString = preg_replace('/[^0-9.]/', '', $fob);
        $price = (float) $priceString;

        if ($price <= 0) {
            return null;
        }

        // Use ex-factory date if available, otherwise default
        $defaultDate = $exFactoryDate ?: date('Y-m-d', strtotime('+30 days'));

        return [
            'colourway' => [
                'style_number' => $this->normalizeStyleNumber($styleCode),
                'colour_name' => $this->normalizeColourName($colorName),
                'product_description' => $styleName,
            ],
            'customer_exfty_date' => $defaultDate,
            'customer_into_wh' => date('Y-m-d', strtotime($defaultDate . ' +5 days')),
            'quantities' => [
                [
                    'size' => $size,
                    'qty' => $qty,
                    'price' => $price,
                    'commission' => 0,
                    'discount' => 0,
                ]
            ],
            'shipment_lines' => [
                ['exfty' => $defaultDate]
            ],
            '_pure_metadata' => [
                'item_number' => $itemNumber,
                'ean' => $ean,
            ],
        ];
    }
}

