<?php

namespace App\Services\CommissionImports\Extractors;

use App\Services\CommissionImports\BaseExtractor;
use App\Services\VertexAiService;
use Illuminate\Support\Facades\Log;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

class MrPorterExtractor extends BaseExtractor
{
    protected VertexAiService $vertexAi;

    public function __construct()
    {
        $this->vertexAi = new VertexAiService();
    }

    public function getCustomerName(): string
    {
        return 'Mr Porter';
    }

    public function getCustomerId(): int
    {
        // Use USD customer profile if configured (Mr Porter ($))
        return 250;
    }

    public function getSupportedExtensions(): array
    {
        return ['pdf', 'xlsx', 'xls'];
    }

    public function extractData(TemporaryUploadedFile $file): array
    {
        $extension = strtolower($file->getClientOriginalExtension());

        if (!in_array($extension, $this->getSupportedExtensions())) {
            throw new \InvalidArgumentException("Unsupported file type for Mr Porter: {$extension}");
        }

        Log::info('Mr Porter - Starting Vertex AI extraction', [
            'file' => $file->getClientOriginalName(),
            'extension' => $extension,
            'size' => $file->getSize(),
        ]);

        // Get the file path
        $filePath = $file->getRealPath();
        
        // Calculate traditional total from raw file
        $traditionalTotal = $this->calculateTraditionalTotal($file, $extension);

        // Convert PDF to CSV for better table structure recognition
        $fileToProcess = $filePath;
        $mimeType = VertexAiService::getMimeType($extension);
        $convertedFile = null;
        
        if ($extension === 'pdf') {
            Log::info('Mr Porter - Converting PDF to CSV for better extraction');
            $convertedFile = $this->convertPdfToCsv($file);
            if ($convertedFile) {
                $fileToProcess = $convertedFile;
                $mimeType = 'text/csv';
                Log::info('Mr Porter - Successfully converted PDF to CSV');
            } else {
                Log::warning('Mr Porter - PDF conversion failed, using original PDF');
            }
        }

        // Create extraction prompt
        $prompt = $this->buildExtractionPrompt();

        try {
            // Extract data using Vertex AI
            $extractedData = $this->vertexAi->extractFromFile($fileToProcess, $prompt, $mimeType);

            if (!$extractedData) {
                throw new \RuntimeException('Vertex AI returned empty response');
            }

            // Validate and normalize the extracted data
            $normalizedData = $this->normalizeExtractedData($extractedData);

            // Calculate actual total units from extracted quantities
            $geminiTotal = 0;
            foreach ($normalizedData['orders'] ?? [] as $order) {
                foreach ($order['lines'] ?? [] as $line) {
                    foreach ($line['quantities'] ?? [] as $qty) {
                        $geminiTotal += (int)($qty['qty'] ?? 0);
                    }
                }
            }

            // Check if totals match
            $totalsMismatch = ($traditionalTotal !== null && $geminiTotal !== $traditionalTotal);
            $warningMessage = null;
            
            if ($totalsMismatch) {
                $warningMessage = "Total quantity mismatch! Gemini extracted: {$geminiTotal} units, but file shows: {$traditionalTotal} units. Please verify the data.";
                Log::warning('Mr Porter - Total mismatch detected', [
                    'gemini_total' => $geminiTotal,
                    'traditional_total' => $traditionalTotal,
                    'difference' => abs($geminiTotal - $traditionalTotal)
                ]);
            } else {
                Log::info('Mr Porter - Totals match or validation skipped', [
                    'gemini_total' => $geminiTotal,
                    'traditional_total' => $traditionalTotal,
                    'validation_status' => $traditionalTotal === null ? 'skipped (no total found)' : 'passed'
                ]);
            }

            Log::info('Mr Porter - Successfully extracted data', [
                'orders_count' => count($normalizedData['orders'] ?? []),
                'total_lines' => array_sum(array_map(fn($order) => count($order['lines'] ?? []), $normalizedData['orders'] ?? [])),
                'detected_season' => $normalizedData['season'] ?? null,
                'gemini_total' => $geminiTotal,
                'traditional_total' => $traditionalTotal,
                'totals_match' => !$totalsMismatch
            ]);

            // Add metadata for season detection and total validation
            $normalizedData['metadata'] = [
                'detected_season' => $normalizedData['season'] ?? null,
                'extraction_method' => 'vertex_ai_gemini',
                'gemini_total' => $geminiTotal,
                'traditional_total' => $traditionalTotal,
                'totals_match' => !$totalsMismatch,
                'total_warning' => $warningMessage,
                'filename' => $file->getClientOriginalName(),
            ];

            return $normalizedData;

        } catch (\Throwable $e) {
            Log::error('Mr Porter - Vertex AI extraction failed', [
                'error' => $e->getMessage(),
                'file' => $file->getClientOriginalName(),
                'memory_usage_mb' => round(memory_get_usage(true) / 1048576, 2),
            ]);
            throw new \RuntimeException("Failed to extract Mr Porter data: {$e->getMessage()}");
        } finally {
            // Clean up converted file if it exists
            if ($convertedFile && file_exists($convertedFile)) {
                @unlink($convertedFile);
                Log::debug('Mr Porter - Cleaned up converted file');
            }
        }
    }

    public function validateCustomerData(array $data): array
    {
        $errors = [];

        foreach (($data['orders'] ?? []) as $orderIndex => $order) {
            if (!empty($order['purchase_order_number'])) {
                // Allow alphanumeric PO like MrPMRPMPLPSS26I10
                if (!preg_match('/^[A-Za-z0-9-]+$/', $order['purchase_order_number'])) {
                    $errors[] = "Order {$orderIndex}: PO should be alphanumeric.";
                }
            }

            foreach (($order['lines'] ?? []) as $lineIndex => $line) {
                $styleNumber = $line['colourway']['style_number'] ?? '';
                if (!empty($styleNumber) && !preg_match('/^[A-Za-z0-9-]+$/', $styleNumber)) {
                    $errors[] = "Order {$orderIndex}, Line {$lineIndex}: Style number should be alphanumeric.";
                }
            }
        }

        return $errors;
    }

    /**
     * Calculate traditional total by reading the Excel/PDF file directly
     */
    protected function calculateTraditionalTotal(TemporaryUploadedFile $file, string $extension): ?int
    {
        try {
            if (in_array($extension, ['xlsx', 'xls'])) {
                return $this->calculateTotalFromExcel($file);
            } elseif ($extension === 'pdf') {
                return $this->calculateTotalFromPdf($file);
            }
        } catch (\Exception $e) {
            Log::warning('Mr Porter - Could not calculate traditional total', [
                'error' => $e->getMessage()
            ]);
        }
        
        return null;
    }

    /**
     * Calculate total from Excel file by looking for "Total Units" row
     */
    protected function calculateTotalFromExcel(TemporaryUploadedFile $file): ?int
    {
        $sheets = \Maatwebsite\Excel\Facades\Excel::toArray(null, $file);
        $mainSheet = $sheets[0] ?? [];
        
        // Look for "Total Units" in the first column
        foreach ($mainSheet as $row) {
            $firstCell = strtolower(trim((string)($row[0] ?? '')));
            if (str_contains($firstCell, 'total units')) {
                // Total is usually in the next cell or a few cells over
                for ($i = 1; $i < min(count($row), 10); $i++) {
                    $value = $row[$i] ?? null;
                    if (is_numeric($value) && $value > 0) {
                        return (int)$value;
                    }
                }
            }
        }
        
        return null;
    }

    /**
     * Calculate total from PDF by looking for "Total Units" text
     */
    protected function calculateTotalFromPdf(TemporaryUploadedFile $file): ?int
    {
        try {
            $text = \Spatie\PdfToText\Pdf::getText($file->getRealPath());
            
            // Try multiple patterns for total units
            $patterns = [
                '/Total\s+Units\s*:?\s*(\d+)/i',           // "Total Units: 123" or "Total Units 123"
                '/Total\s*:?\s*(\d+)\s+units?/i',          // "Total: 123 units" or "Total 123 unit"
                '/Grand\s+Total\s*:?\s*(\d+)/i',           // "Grand Total: 123"
                '/Total\s+Quantity\s*:?\s*(\d+)/i',        // "Total Quantity: 123"
                '/Total\s+Qty\s*:?\s*(\d+)/i',             // "Total Qty: 123"
                '/\bTotal\b[^\d]{0,20}(\d+)\s*$/mi',       // "Total" followed by number at end of line
            ];
            
            foreach ($patterns as $pattern) {
                if (preg_match($pattern, $text, $matches)) {
                    Log::info('Mr Porter - Found total in PDF using pattern', [
                        'pattern' => $pattern,
                        'total' => $matches[1]
                    ]);
                    return (int)$matches[1];
                }
            }
            
            Log::warning('Mr Porter - Could not find total units in PDF text', [
                'text_sample' => substr($text, 0, 500)
            ]);
        } catch (\Exception $e) {
            Log::debug('Could not extract text from PDF', ['error' => $e->getMessage()]);
        }
        
        return null;
    }

    /**
     * Build the extraction prompt for Vertex AI
     */
    protected function buildExtractionPrompt(): string
    {
        return <<<'PROMPT'
Extract purchase order data from this Mr Porter file.

COLUMNS YOU MAY FIND:
- PO Number: The COMPLETE alphanumeric string (e.g., MrPMRPMPLPSS26K24, MrPMRPMPLPSS26K30) - extract the ENTIRE string, not just part of it
- Date viewed / Order Date: Date near the PO number
- Style No: Product style code starting with "MP" (e.g., MPA242027, MPK252017) - DO NOT include size letters like "S", "M", "L" before the style code
- Description: Product name
- Designer Colour: Color name (DO NOT USE Designer Colour Code)
- Ship Date: Exfty date (DO NOT USE CANCEL DATE EVER)
- Discounted Cost / Original Cost: Unit price in USD
- Size columns: Headers like S, M, L, XL, XXL, or numeric sizes (28, 30, 32)
- Quantity cells: Numbers under each size column

CRITICAL RULES FOR READING QUANTITIES:
- Each size column is SEPARATE - do NOT combine adjacent cells
- Read each cell independently
- The size header tells you which size each quantity belongs to
- Pay careful attention to column alignment - each quantity must match its size header

UNDERSTAND THE DATA:
- Products may span multiple rows (info in one row, quantities in another)
- Size labels are in the header row, quantities are directly below in their own cells
- Each size/quantity pair is in its own column - do not merge them
- Price is per unit
- The PO Number contains a season code (like SS26 or AW24) within it, but you must extract the FULL PO string

OUTPUT (JSON only, no markdown):
{
  "purchase_order_number": "COMPLETE_PO_STRING_HERE",
  "order_date": "YYYY-MM-DD",
  "incoterms": "FOB",
  "orders": [{
    "order_date": "YYYY-MM-DD",
    "lines": [{
      "colourway": {
        "style_number": "...",
        "description": "...",
        "colour_name": "...",
        "colour_code": "..."
      },
      "customer_exfty_date": "YYYY-MM-DD",
      "quantities": [
        {"size": "S", "qty": 2, "price": 15.93},
        {"size": "M", "qty": 7, "price": 15.93}
      ]
    }]
  }]
}

Extract all products:
PROMPT;
    }

    /**
     * Normalize extracted data to match expected format
     */
    protected function normalizeExtractedData(array $data): array
    {
        // Get order date from top level or use current date as fallback
        $orderDate = $this->parseDate($data['order_date'] ?? '') ?? date('Y-m-d');
        
        $normalized = [
            'purchase_order_number' => $data['purchase_order_number'] ?? '',
            'season' => $data['season'] ?? $this->extractSeasonFromText($data['purchase_order_number'] ?? ''),
            'incoterms' => $data['incoterms'] ?? 'FOB',
            'orders' => []
        ];

        // Normalize orders
        foreach (($data['orders'] ?? []) as $order) {
            $normalizedOrder = [
                'order_date' => $this->parseDate($order['order_date'] ?? '') ?? $orderDate,
                'purchase_order_number' => $data['purchase_order_number'] ?? '',
                'lines' => []
            ];

            foreach (($order['lines'] ?? []) as $line) {
                // Use customer_exfty_date if available, otherwise fall back to ship_date
                $exftyDate = $this->parseDate($line['customer_exfty_date'] ?? $line['ship_date'] ?? '');
                
                $normalizedLine = [
                    'colourway' => [
                        'style_number' => $this->normalizeStyleNumber($line['colourway']['style_number'] ?? ''),
                        'description' => trim($line['colourway']['description'] ?? ''),
                        'colour_name' => $this->normalizeColourName($line['colourway']['colour_name'] ?? ''),
                        'colour_code' => trim($line['colourway']['colour_code'] ?? ''),
                    ],
                    'customer_exfty_date' => $exftyDate,
                    'quantities' => []
                ];

                // Normalize quantities and ensure numeric values
                foreach (($line['quantities'] ?? []) as $qty) {
                    $size = trim(strtoupper((string)($qty['size'] ?? '')));
                    $quantity = (int)($qty['qty'] ?? 0);
                    $price = (float)($qty['price'] ?? 0);

                    if ($size && $quantity > 0) {
                        $normalizedLine['quantities'][] = [
                            'size' => $size,
                            'qty' => $quantity,
                            'price' => $price,
                        ];
                    }
                }

                // Only add lines with quantities
                if (!empty($normalizedLine['quantities'])) {
                    $normalizedOrder['lines'][] = $normalizedLine;
                }
            }

            if (!empty($normalizedOrder['lines'])) {
                $normalized['orders'][] = $normalizedOrder;
            }
        }

        return $normalized;
    }

    /**
     * Convert PDF to CSV using tabula-py for better table extraction
     */
    protected function convertPdfToCsv(TemporaryUploadedFile $file): ?string
    {
        try {
            // Create temporary CSV file path
            $tempCsvPath = sys_get_temp_dir() . '/' . uniqid('mrporter_') . '.csv';
            
            // Helper to run a command with a hard timeout to avoid hanging workers
            $runWithTimeout = function (string $cmd, array &$out, int &$code, int $seconds = 60): void {
                $wrapped = sprintf('timeout %ds %s', $seconds, $cmd);
                $out = [];
                exec($wrapped, $out, $code);
            };
            
            // 1) Try stream mode first (better for text-based PDFs)
            $cmdStream = sprintf(
                'python3 -c "import tabula; tabula.convert_into(%s, %s, output_format=\'csv\', pages=\'all\', stream=True, guess=False)" 2>&1',
                escapeshellarg($file->getRealPath()),
                escapeshellarg($tempCsvPath)
            );
            $output = []; $returnCode = 0;
            $runWithTimeout($cmdStream, $output, $returnCode, 60);
            
            if ($returnCode === 0 && file_exists($tempCsvPath) && filesize($tempCsvPath) > 0) {
                $csvSample = implode("\n", array_slice(file($tempCsvPath), 0, 10));
                Log::info('Mr Porter - PDF converted to CSV successfully (stream mode)', [
                    'output_file' => $tempCsvPath,
                    'size' => filesize($tempCsvPath),
                    'csv_sample' => $csvSample
                ]);
                return $tempCsvPath;
            }
            
            Log::warning('Mr Porter - PDF to CSV (stream mode) failed, trying lattice', [
                'return_code' => $returnCode,
                'output' => implode("\n", $output)
            ]);
            
            // 2) Fallback: lattice mode (better for bordered tables)
            $cmdLattice = sprintf(
                'python3 -c "import tabula; tabula.convert_into(%s, %s, output_format=\'csv\', pages=\'all\', lattice=True)" 2>&1',
                escapeshellarg($file->getRealPath()),
                escapeshellarg($tempCsvPath)
            );
            $output2 = []; $returnCode2 = 0;
            $runWithTimeout($cmdLattice, $output2, $returnCode2, 60);
            
            if ($returnCode2 === 0 && file_exists($tempCsvPath) && filesize($tempCsvPath) > 0) {
                $csvSample = implode("\n", array_slice(file($tempCsvPath), 0, 10));
                Log::info('Mr Porter - PDF converted to CSV successfully (lattice mode)', [
                    'output_file' => $tempCsvPath,
                    'size' => filesize($tempCsvPath),
                    'csv_sample' => $csvSample
                ]);
                return $tempCsvPath;
            }
            
            Log::warning('Mr Porter - PDF to CSV conversion failed (both modes)', [
                'stream_code' => $returnCode,
                'stream_output' => implode("\n", $output),
                'lattice_code' => $returnCode2,
                'lattice_output' => implode("\n", $output2),
            ]);
            
            // Clean up failed conversion file
            if (file_exists($tempCsvPath)) {
                @unlink($tempCsvPath);
            }
            
            return null;
            
        } catch (\Throwable $e) {
            Log::error('Mr Porter - PDF conversion error', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }
}
