<?php

namespace App\Services\CommissionImports\Extractors;

use App\Services\CommissionImports\BaseExtractor;
use App\Services\CommissionImports\TabulaService;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Log;

class ChintiExtractor extends BaseExtractor
{
    private TabulaService $tabulaService;

    public function __construct()
    {
        $this->tabulaService = new TabulaService();
    }

    public function getCustomerName(): string
    {
        return 'Chinti & Parker';
    }

    public function getCustomerId(): int
    {
        return 30; // Chinti & Parker customer ID from database
    }

    public function extractData(TemporaryUploadedFile $file): array
    {
        $extension = strtolower($file->getClientOriginalExtension());

        if ($extension === 'pdf') {
            return $this->extractFromPdf($file);
        } elseif (in_array($extension, ['xlsx', 'xls'])) {
            return $this->extractFromExcel($file);
        }

        throw new \InvalidArgumentException("Unsupported file type: {$extension}");
    }


    public function validateCustomerData(array $data): array
    {
        $errors = [];

        foreach ($data['orders'] ?? [] as $orderIndex => $order) {
            // Validate Chinti-specific PO format: 6-digit numbers (like 101428)
            if (!empty($order['purchase_order_number'])) {
                if (!preg_match('/^\d{6}$/', $order['purchase_order_number'])) {
                    $errors[] = "Order {$orderIndex}: Chinti PO numbers should be 6-digit numbers";
                }
            }

            // Validate Chinti-specific style number format: AAW2312, KAW2451, KPS2322, KSS2501, etc.
            foreach ($order['lines'] ?? [] as $lineIndex => $line) {
                $styleNumber = $line['colourway']['style_number'] ?? '';
                if (!empty($styleNumber) && !preg_match('/^(AAW|KAW|KPS|KSS)\d{4}$/', $styleNumber)) {
                    $errors[] = "Order {$orderIndex}, Line {$lineIndex}: Chinti style numbers should follow format 'AAW2312', 'KAW2451', 'KPS2322', or 'KSS2501'";
                }
            }
        }

        return $errors;
    }

    /**
     * Extract data from PDF using Tabula with Chinti-specific parameters
     */
    private function extractFromPdf(TemporaryUploadedFile $file): array
    {
        // Chinti-specific Tabula extraction parameters
        // Get total number of pages and extract from all of them
        $pdfPath = $file->getRealPath();
        $pdfInfo = shell_exec("pdfinfo " . escapeshellarg($pdfPath) . " 2>&1");
        $totalPages = 9; // Default to 9 pages

        if (preg_match('/Pages:\s+(\d+)/', $pdfInfo, $matches)) {
            $totalPages = (int)$matches[1];
        }

        // Build array of all page numbers [1, 2, 3, ...]
        $pages = range(1, $totalPages);
        $mode = 'stream'; // Chinti uses less structured tables

        $rows = $this->tabulaService->extractTables($pdfPath, $pages, $mode);

        return $this->parsePdfTableData($rows, file_get_contents($pdfPath));
    }

    /**
     * Extract data from Excel with Chinti-specific logic
     */
    private function extractFromExcel(TemporaryUploadedFile $file): array
    {
        $sheets = Excel::toArray(null, $file);
        $mainSheet = $sheets[0] ?? []; // Chinti uses first sheet
        
        return $this->parseExcelData($mainSheet);
    }

    /**
     * Parse PDF table data into standardized format - Chinti specific
     */
    private function parsePdfTableData(array $rows, string $fullText): array
    {
        if (empty($rows)) {
            throw new \RuntimeException('No table data found in PDF');
        }

        // Extract metadata from full text
        $season = $this->extractSeasonFromText($fullText);
        $poNumber = $this->extractPoNumber($fullText);
        $orderDate = $this->extractOrderDate($fullText);

        if (!$poNumber) {
            // If no PO number found, this might be a different format PDF
            // For now, throw an error but this could be enhanced to use a default
            throw new \RuntimeException('Could not find purchase order number in PDF');
        }

        // Parse the table data - Chinti format has multiple tables per PDF
        $orders = [];

        foreach ($rows as $rowIndex => $row) {
            // Skip header rows and empty rows
            if (empty($row[0]) || strpos($row[0], 'First/Make') !== false || strpos($row[0], 'Delivery From') !== false) {
                continue;
            }

            // Look for style code rows - support multiple formats (AAW, KAW, KPS, KSS, etc.)
            if (preg_match('/^(AAW|KAW|KPS|KSS)\d{4}$/', $row[0] ?? '', $matches)) {
                // Start new order for this style code
                $currentStyleCode = $row[0];
                $currentOrder = [
                    'order_type' => 'commission',
                    'incoterms' => 'FOB',
                    'season' => $season,
                    'order_date' => $orderDate,
                    'purchase_order_number' => $poNumber,
                    'lines' => [],
                ];
            } elseif (isset($currentOrder) && count($row) >= 8 && is_numeric($row[5] ?? '')) {
                // Parse order line data - this contains the product, color, and quantities
                // Handles both simple format (8 cols) and detailed format (13 cols)
                $line = $this->parseChintiOrderLine($row, $poNumber, $currentStyleCode);
                if ($line) {
                    $currentOrder['lines'][] = $line;
                    // Add this order to the orders array since we have at least one line
                    $orders[] = $currentOrder;
                }
            }
        }

        if (empty($orders)) {
            throw new \RuntimeException('No valid orders found in PDF data');
        }

        // Log successful extraction for audit trail (commented out for testing)
        // Log::info('Chinti PDF extraction completed', [
        //     'orders_found' => count($orders),
        //     'po_number' => $poNumber,
        //     'season' => $season,
        //     'order_date' => $orderDate,
        // ]);

        return [
            'orders' => $orders,
            'metadata' => [
                'detected_season' => $season,
                'extraction_method' => 'tabula_pdf',
            ],
        ];
    }

    /**
     * Parse Excel data into standardized format - Chinti specific
     */
    private function parseExcelData(array $rows): array
    {
        if (empty($rows)) {
            throw new \RuntimeException('No data found in Excel file');
        }

        $orders = [];
        $detectedSeason = null;

        // Extract season from document
        foreach (array_slice($rows, 0, 10) as $row) {
            $rowText = implode(' ', $row);
            $season = $this->extractSeasonFromText($rowText);
            if ($season) {
                $detectedSeason = $season;
                break;
            }
        }

        // TODO: Implement Chinti-specific Excel parsing logic based on actual document structure
        // This will need to be customized based on Chinti's actual Excel format

        return [
            'orders' => $orders,
            'metadata' => [
                'detected_season' => $detectedSeason,
                'extraction_method' => 'excel_sheets',
            ],
        ];
    }

    /**
     * Extract PO Number from document text (Chinti specific)
     */
    private function extractPoNumber(string $text): ?string
    {
        // Multiple patterns to match PO numbers in Chinti documents
        $patterns = [
            // Standard docket number format
            '/Docket Number:\s*(\d{6})/',
            '/Docket Number\s*(\d{6})/',

            // Alternative formats that might appear in different PDF layouts
            '/PO Number:\s*(\d{6})/',
            '/PO Number\s*(\d{6})/',
            '/Order Number:\s*(\d{6})/',
            '/Order Number\s*(\d{6})/',
            '/Purchase Order:\s*(\d{6})/',
            '/Purchase Order\s*(\d{6})/',
            '/Invoice Number:\s*(\d{6})/',
            '/Invoice Number\s*(\d{6})/',
            '/Reference:\s*(\d{6})/',
            '/Reference\s*(\d{6})/',

            // Look for 6-digit numbers that appear prominently (like in headers or titles)
            '/\b(10\d{4})\b/', // Numbers starting with 10 (common for Chinti POs)

            // Fallback: any 6-digit number (but prefer the first prominent one)
            '/\b(\d{6})\b/',
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                $poNumber = $matches[1];

                // Validate that this looks like a legitimate PO number
                if (preg_match('/^\d{6}$/', $poNumber)) {
                    return $poNumber;
                }
            }
        }

        return null;
    }

    /**
     * Extract season from document text (Chinti specific override)
     */
    protected function extractSeasonFromText(string $text): ?string
    {
        // Multiple patterns for season extraction in Chinti documents
        $patterns = [
            // Standard AW/SS format
            '/\b(AW|SS)(\d{2,4})\b/i',
            // Season mentioned in context
            '/Season:\s*(AW|SS)(\d{2,4})/i',
            '/Season\s*(AW|SS)(\d{2,4})/i',
            // Collection or range format
            '/Collection:\s*(AW|SS)(\d{2,4})/i',
            '/Range:\s*(AW|SS)(\d{2,4})/i',

            // Full season names (WINTER, SPRING, SUMMER, AUTUMN)
            '/\b(WINTER|SPRING|SUMMER|AUTUMN)\s*(\d{2,4})\b/i',
            '/\b(Winter|Spring|Summer|Autumn)\s*(\d{2,4})\b/i',

            // Season names in context
            '/Season:\s*(WINTER|SPRING|SUMMER|AUTUMN)\s*(\d{2,4})/i',
            '/Season\s*(WINTER|SPRING|SUMMER|AUTUMN)\s*(\d{2,4})/i',

            // Collection with full season names
            '/Collection:\s*(WINTER|SPRING|SUMMER|AUTUMN)\s*(\d{2,4})/i',

            // Additional patterns for different PDF layouts
            '/Docket Reference:\s*(WINTER|SPRING|SUMMER|AUTUMN)\s*(\d{2,4})/i',
            '/Reference:\s*(WINTER|SPRING|SUMMER|AUTUMN)\s*(\d{2,4})/i',
            '/Title:\s*(WINTER|SPRING|SUMMER|AUTUMN)\s*(\d{2,4})/i',

            // Case-insensitive and flexible spacing
            '/\b(winter|spring|summer|autumn)\s*(\d{2,4})\b/i',
            '/\b(WINTER|SPRING|SUMMER|AUTUMN)\s*25\b/i',
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                $seasonName = strtoupper($matches[1]);
                $year = $matches[2];

                // Convert 4-digit years to 2-digit
                if (strlen($year) === 4) {
                    $year = substr($year, -2);
                }

                // Map full season names to AW/SS codes
                $seasonCode = $this->mapSeasonNameToCode($seasonName);

                return $seasonCode . $year;
            }
        }

        return null;
    }

    /**
     * Map full season names to AW/SS codes
     */
    private function mapSeasonNameToCode(string $seasonName): string
    {
        $upperSeason = strtoupper($seasonName);

        if (in_array($upperSeason, ['WINTER', 'AUTUMN'])) {
            return 'AW'; // Winter and Autumn map to AW
        } elseif (in_array($upperSeason, ['SPRING', 'SUMMER'])) {
            return 'SS'; // Spring and Summer map to SS
        } else {
            return strtoupper(substr($seasonName, 0, 2)); // Fallback to first 2 chars
        }
    }

    /**
     * Extract order date from document text (Chinti specific)
     */
    private function extractOrderDate(string $text): ?string
    {
        // Multiple date patterns for Chinti documents
        $patterns = [
            // Standard docket date format
            '/Docket Date:\s*(\d{1,2}\s+\w+\s+\d{4})/',
            '/Docket Date\s*(\d{1,2}\s+\w+\s+\d{4})/',

            // Alternative formats
            '/Order Date:\s*(\d{1,2}\s+\w+\s+\d{4})/',
            '/Order Date\s*(\d{1,2}\s+\w+\s+\d{4})/',
            '/Date:\s*(\d{1,2}\s+\w+\s+\d{4})/',
            '/Date\s*(\d{1,2}\s+\w+\s+\d{4})/',

            // Delivery dates as fallback
            '/Delivery From:\s*(\d{1,2}\s+\w+\s+\d{4})/',
            '/Delivery To:\s*(\d{1,2}\s+\w+\s+\d{4})/',
            '/Delivery Date:\s*(\d{1,2}\s+\w+\s+\d{4})/',

            // Printed date (often appears at bottom of documents)
            '/Printed:\s*(\d{1,2}\s+\w+\s+\d{4})/',
            '/Printed:\s*(\d{1,2}\s+\w+\s+\d{4}\s+-\s+\d{1,2}:\d{2}:\d{2})/',

            // Look for dates in the format DD Month YYYY
            '/(\d{1,2}\s+(?:January|February|March|April|May|June|July|August|September|October|November|December)\s+\d{4})/',
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                $dateString = trim($matches[1]);
                $parsedDate = $this->parseDate($dateString);
                if ($parsedDate) {
                    return $parsedDate;
                }
            }
        }

        // If no date found, use a default date (today) to prevent import failures
        return date('Y-m-d');
    }

    /**
     * Parse individual Chinti order line from table row
     */
    private function parseChintiOrderLine(array $row, string $poNumber, string $styleCode = ''): ?array
    {
        $numColumns = count($row);
        $productName = trim($row[0] ?? '');
        $colorName = trim($row[1] ?? '');

        if (empty($productName) || empty($colorName)) {
            return null;
        }

        $quantities = [];
        $unitPrice = 0;

        // Detect format based on number of columns
        if ($numColumns >= 13) {
            // Detailed format (13 columns): product, color, del_from, del_to, XS, S, M, L, LL, 3L, total_qty, cost, total_cost
            // Size columns: [4]=XS, [5]=S, [6]=M, [7]=L, [8]=LL, [9]=3L
            $sizeColumns = [
                'XS' => 4,
                'S' => 5,
                'M' => 6,
                'L' => 7,
                'LL' => 8,
                '3L' => 9,
            ];

            $cost = trim($row[11] ?? ''); // Cost is in column 11

            foreach ($sizeColumns as $size => $columnIndex) {
                $qty = (int)($row[$columnIndex] ?? 0);
                if ($qty > 0) {
                    $quantities[] = [
                        'size' => $size,
                        'qty' => $qty,
                        'price' => 0, // Will be set below
                        'commission' => 0,
                        'discount' => 0,
                    ];
                }
            }
        } else {
            // Simple format (8 columns): product, color, del_from, del_to, size_header, qty, cost, total_cost
            $size = 'OS'; // Default to one-size
            $qty = (int)($row[5] ?? 0);
            $cost = trim($row[6] ?? '');

            if ($qty > 0) {
                $quantities[] = [
                    'size' => $size,
                    'qty' => $qty,
                    'price' => 0, // Will be set below
                    'commission' => 0,
                    'discount' => 0,
                ];
            }
        }

        if (empty($quantities) || empty($cost)) {
            return null;
        }

        // Extract price from cost string (Chinti specific format: "US$ 7.63")
        $priceString = preg_replace('/[^0-9.]/', '', $cost);
        $unitPrice = (float) $priceString;

        if ($unitPrice <= 0) {
            return null;
        }

        // Set price for all quantities
        foreach ($quantities as &$quantity) {
            $quantity['price'] = $unitPrice;
        }

        return [
            'colourway' => [
                'style_number' => $this->normalizeStyleNumber($styleCode),
                'colour_name' => $this->normalizeColourName($colorName),
                'product_description' => $productName,
            ],
            'customer_exfty_date' => date('Y-m-d', strtotime('+30 days')), // Default
            'customer_into_wh' => date('Y-m-d', strtotime('+35 days')), // Default
            'quantities' => $quantities,
            'shipment_lines' => [
                ['exfty' => date('Y-m-d', strtotime('+30 days'))]
            ],
        ];
    }
}