<?php

namespace App\Services\CommissionImports\Extractors;

use App\Services\CommissionImports\BaseExtractor;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Log;

class CharlesTyrwhittExtractor extends BaseExtractor
{
    public function getCustomerName(): string
    {
        return 'Charles Tyrwhitt';
    }

    public function getCustomerId(): int
    {
        return 69; // Charles Tyrwhitt customer ID from database
    }

    public function extractData(TemporaryUploadedFile $file): array
    {
        $extension = strtolower($file->getClientOriginalExtension());

        if (in_array($extension, ['xlsx', 'xls'])) {
            return $this->extractFromExcel($file);
        }

        throw new \InvalidArgumentException("Unsupported file type for Charles Tyrwhitt: {$extension}. Please use Excel files.");
    }

    public function getSupportedExtensions(): array
    {
        return ['xlsx', 'xls']; // Charles Tyrwhitt only uses Excel
    }

    public function validateCustomerData(array $data): array
    {
        $errors = [];

        foreach ($data['orders'] ?? [] as $orderIndex => $order) {
            // Validate Charles Tyrwhitt-specific PO format (e.g., "LPO0056989")
            if (!empty($order['purchase_order_number'])) {
                if (!preg_match('/^LPO\d{7}$/', $order['purchase_order_number'])) {
                    $errors[] = "Order {$orderIndex}: Charles Tyrwhitt PO numbers should follow format 'LPO0056989'";
                }
            }

            // Validate Charles Tyrwhitt-specific style number format (e.g., KNT0016CAM)
            foreach ($order['lines'] ?? [] as $lineIndex => $line) {
                $styleNumber = $line['colourway']['style_number'] ?? '';
                if (!empty($styleNumber) && !preg_match('/^[A-Z]{3}\d{4}[A-Z]{3}$/', $styleNumber)) {
                    $errors[] = "Order {$orderIndex}, Line {$lineIndex}: Charles Tyrwhitt style numbers should follow format 'KNT0016CAM'";
                }
            }
        }

        return $errors;
    }

    /**
     * Extract data from Excel with Charles Tyrwhitt-specific logic
     */
    private function extractFromExcel(TemporaryUploadedFile $file): array
    {
        $sheets = Excel::toArray(null, $file);
        $mainSheet = $sheets[0] ?? [];

        return $this->parseExcelData($mainSheet);
    }

    /**
     * Parse Excel data into standardized format - Charles Tyrwhitt specific
     */
    private function parseExcelData(array $rows): array
    {
        if (empty($rows)) {
            throw new \RuntimeException('No data found in Excel file');
        }

        // Charles Tyrwhitt Excel structure:
        // Column 0: PO number
        // Column 1: Category
        // Column 2: Subcategory
        // Column 3: Item number (base style code like KNT0016CAM)
        // Column 4: Product name
        // Column 5: Ex-Factory Date (Excel date serial)
        // Column 6: Delivery date (Excel date serial)
        // Column 7: SKU (includes size, like KNT0016CAMXS)
        // Column 8: EAN
        // Column 9: Qty

        $detectedSeason = null;
        $styleColorLines = []; // Group by item number (which represents style+color)
        $poNumbers = [];
        $exFactoryDates = [];

        // Skip header row and process data
        foreach (array_slice($rows, 1) as $rowIndex => $row) {
            $poNumber = trim($row[0] ?? '');
            $category = trim($row[1] ?? '');
            $subcategory = trim($row[2] ?? '');
            $itemNumber = trim($row[3] ?? ''); // Base style code (KNT0016CAM)
            $productName = trim($row[4] ?? '');
            $exFactorySerial = $row[5] ?? null;
            $deliverySerial = $row[6] ?? null;
            $sku = trim($row[7] ?? ''); // Full SKU with size (KNT0016CAMXS)
            $ean = trim($row[8] ?? '');
            $qty = (int)($row[9] ?? 0);

            if (empty($poNumber) || empty($itemNumber) || empty($sku) || $qty <= 0) {
                continue;
            }

            // Extract size from SKU (e.g., KNT0016CAMXS -> XS)
            $size = $this->extractSizeFromSku($sku, $itemNumber);

            // Convert Excel date serial to actual date
            $exFactoryDate = $this->convertExcelDate($exFactorySerial);
            $deliveryDate = $this->convertExcelDate($deliverySerial);

            // Collect PO numbers and dates
            if (!in_array($poNumber, $poNumbers)) {
                $poNumbers[] = $poNumber;
            }
            if ($exFactoryDate && !in_array($exFactoryDate, $exFactoryDates)) {
                $exFactoryDates[] = $exFactoryDate;
            }

            // Group by item number (style+color combination)
            if (!isset($styleColorLines[$itemNumber])) {
                $styleColorLines[$itemNumber] = [
                    'po_number' => $poNumber,
                    'style_code' => $itemNumber,
                    'product_name' => $productName,
                    'category' => $category,
                    'subcategory' => $subcategory,
                    'ex_factory_date' => $exFactoryDate,
                    'delivery_date' => $deliveryDate,
                    'quantities' => [],
                ];
            }

            $styleColorLines[$itemNumber]['quantities'][] = [
                'size' => $size,
                'qty' => $qty,
                'price' => 1.00, // Default price - will need to be updated manually or from customer settings
                'commission' => 0,
                'discount' => 0,
            ];
        }

        // Extract season from first product name or use default
        if (!empty($styleColorLines)) {
            $firstLine = reset($styleColorLines);
            $detectedSeason = $this->extractSeasonFromText($firstLine['product_name'] ?? '');
        }

        // Convert grouped lines into orders
        $orders = [];

        // Group by PO number
        foreach ($poNumbers as $poNumber) {
            $orderLines = [];

            foreach ($styleColorLines as $itemNumber => $lineData) {
                if ($lineData['po_number'] === $poNumber) {
                    $orderLines[] = [
                        'colourway' => [
                            'style_number' => $this->normalizeStyleNumber($lineData['style_code']),
                            'colour_name' => $this->extractColorFromStyleCode($lineData['style_code']),
                            'product_description' => $lineData['product_name'],
                        ],
                        'customer_exfty_date' => $lineData['ex_factory_date'] ?: date('Y-m-d', strtotime('+30 days')),
                        'customer_into_wh' => $lineData['delivery_date'] ?: date('Y-m-d', strtotime('+35 days')),
                        'quantities' => $lineData['quantities'],
                        'shipment_lines' => [
                            ['exfty' => $lineData['ex_factory_date'] ?: date('Y-m-d', strtotime('+30 days'))]
                        ],
                    ];
                }
            }

            if (!empty($orderLines)) {
                $orders[] = [
                    'order_type' => 'commission',
                    'incoterms' => 'FOB',
                    'season' => $detectedSeason,
                    'order_date' => date('Y-m-d'), // Use today's date
                    'purchase_order_number' => $poNumber,
                    'lines' => $orderLines,
                ];
            }
        }

        if (empty($orders)) {
            throw new \RuntimeException('No valid orders found in Excel data');
        }

        return [
            'orders' => $orders,
            'metadata' => [
                'detected_season' => $detectedSeason,
                'extraction_method' => 'excel_sheets',
            ],
        ];
    }

    /**
     * Extract size from SKU by removing the base item number
     */
    private function extractSizeFromSku(string $sku, string $itemNumber): string
    {
        // SKU format: KNT0016CAMXS where KNT0016CAM is item number and XS is size
        $size = str_replace($itemNumber, '', $sku);

        // Normalize size names
        $sizeMap = [
            'XXXL' => '3XL',
            'XXL' => '2XL',
            'XL' => 'XL',
            'L' => 'L',
            'M' => 'M',
            'S' => 'S',
            'XS' => 'XS',
        ];

        foreach ($sizeMap as $pattern => $normalized) {
            if ($size === $pattern) {
                return $normalized;
            }
        }

        return $size ?: 'M'; // Default to M if size can't be determined
    }

    /**
     * Extract color name from style code
     */
    private function extractColorFromStyleCode(string $styleCode): string
    {
        // Style code format: KNT0016CAM where last 3 letters are color code
        if (preg_match('/([A-Z]{3})$/', $styleCode, $matches)) {
            $colorCode = $matches[1];

            // Map common color codes to names
            $colorMap = [
                'CAM' => 'Camel',
                'NAV' => 'Navy',
                'STL' => 'Steel Blue',
                'BLK' => 'Black',
                'WHT' => 'White',
                'GRY' => 'Grey',
                'BLU' => 'Blue',
                'RED' => 'Red',
                'GRN' => 'Green',
            ];

            return $colorMap[$colorCode] ?? $colorCode;
        }

        return 'Unknown';
    }

    /**
     * Convert Excel date serial number to Y-m-d format
     */
    private function convertExcelDate($serial): ?string
    {
        if (empty($serial) || !is_numeric($serial)) {
            return null;
        }

        // Excel date serial starts from 1900-01-01
        // Note: Excel incorrectly treats 1900 as a leap year, so we need to account for that
        $unixTimestamp = ($serial - 25569) * 86400;
        return date('Y-m-d', $unixTimestamp);
    }
}

