<?php

namespace App\Services\CommissionImports\Extractors;

use App\Services\CommissionImports\BaseExtractor;
use App\Services\CommissionImports\TabulaService;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Maatwebsite\Excel\Facades\Excel;
use Spatie\PdfToText\Pdf;
use Illuminate\Support\Facades\Log;

class BodenExtractor extends BaseExtractor
{
    private TabulaService $tabulaService;

    public function __construct()
    {
        $this->tabulaService = new TabulaService();
    }

    public function getCustomerName(): string
    {
        return 'Boden';
    }

    public function getCustomerId(): int
    {
        return 71; // Boden customer ID from database
    }

    public function extractData(TemporaryUploadedFile $file): array
    {
        $extension = strtolower($file->getClientOriginalExtension());

        if ($extension === 'pdf') {
            return $this->extractFromPdf($file);
        } elseif (in_array($extension, ['xlsx', 'xls'])) {
            return $this->extractFromExcel($file);
        }

        throw new \InvalidArgumentException("Unsupported file type: {$extension}");
    }

    public function validateCustomerData(array $data): array
    {
        $errors = [];

        foreach ($data['orders'] ?? [] as $orderIndex => $order) {
            // Validate Boden-specific PO format: 6-8 digit numbers
            if (!empty($order['purchase_order_number'])) {
                if (!preg_match('/^\d{6,8}$/', $order['purchase_order_number'])) {
                    $errors[] = "Order {$orderIndex}: Boden PO numbers should be 6-8 digit numbers";
                }
            }

            // Validate Boden-specific style number format: flexible format (e.g., K1250)
            foreach ($order['lines'] ?? [] as $lineIndex => $line) {
                $styleNumber = $line['colourway']['style_number'] ?? '';
                if (!empty($styleNumber) && !preg_match('/^[A-Z]+\d+$/', $styleNumber)) {
                    $errors[] = "Order {$orderIndex}, Line {$lineIndex}: Boden style numbers should follow format like 'K1250' (letters followed by numbers)";
                }
            }
        }

        return $errors;
    }

    /**
     * Extract data from PDF using Tabula with Boden-specific parameters
     */
    private function extractFromPdf(TemporaryUploadedFile $file): array
    {
        // Extract text content from PDF first
        $fullText = Pdf::getText($file->getRealPath());
        
        // Boden-specific Tabula extraction parameters
        // Extract from all pages since order data can span multiple pages
        $pages = []; // Empty array means all pages
        $mode = 'stream'; // Boden documents work better with stream mode
        
        $rows = $this->tabulaService->extractTables($file->getRealPath(), $pages, $mode);
        
        return $this->parsePdfTableData($rows, $fullText);
    }

    /**
     * Extract data from Excel with Boden-specific logic
     */
    private function extractFromExcel(TemporaryUploadedFile $file): array
    {
        $sheets = Excel::toArray(null, $file);
        $mainSheet = $sheets[0] ?? []; // Boden uses first sheet
        
        return $this->parseExcelData($mainSheet);
    }

    /**
     * Parse PDF table data into standardized format - Boden specific
     */
    private function parsePdfTableData(array $rows, string $fullText): array
    {
        if (empty($rows)) {
            throw new \RuntimeException('No table data found in PDF');
        }

        // Extract summary information from full text (metadata section)
        $season = $this->extractSeasonFromText($fullText);
        $orderDate = $this->extractOrderDate($fullText);
        $poNumber = $this->extractPoNumberFromMetadata($fullText);
        $styleCode = $this->extractStyleCodeFromMetadata($fullText);
        $productDescription = $this->extractProductDescriptionFromMetadata($fullText);
        
        Log::info('Boden PDF metadata extracted', [
            'po_number' => $poNumber,
            'style_code' => $styleCode,
            'product_description' => $productDescription,
            'season' => $season,
            'order_date' => $orderDate,
            'total_rows' => count($rows),
        ]);
        
        // Find the orders table and extract data using new format
        $orderLines = $this->parseOrderTableNewFormat($rows, $poNumber, $styleCode, $productDescription);
        
        if (empty($orderLines)) {
            throw new \RuntimeException('No order lines found in document');
        }

        // Group order lines by PO number
        $ordersByPo = [];
        foreach ($orderLines as $line) {
            $linePo = $line['_po_number'] ?? $poNumber;
            unset($line['_po_number']);
            unset($line['_order_line_number']); // Remove internal field
            
            if (!isset($ordersByPo[$linePo])) {
                $ordersByPo[$linePo] = [
                    'order_type' => 'commission',
                    'incoterms' => 'FOB',
                    'season' => $season,
                    'order_date' => $orderDate,
                    'purchase_order_number' => $linePo,
                    'lines' => [],
                ];
            }
            
            $ordersByPo[$linePo]['lines'][] = $line;
        }

        $orders = array_values($ordersByPo);
        
        // Log successful extraction for audit trail
        Log::info('Boden PDF extraction completed', [
            'orders_found' => count($orders),
            'total_lines' => count($orderLines),
            'season' => $season,
            'order_date' => $orderDate,
        ]);
        
        return [
            'orders' => $orders,
            'metadata' => [
                'detected_season' => $season,
                'extraction_method' => 'tabula_pdf',
            ],
        ];
    }

    /**
     * Parse Excel data into standardized format - Boden specific
     */
    private function parseExcelData(array $rows): array
    {
        if (empty($rows)) {
            throw new \RuntimeException('No data found in Excel file');
        }

        $orders = [];
        $detectedSeason = null;
        
        // Extract season from first few rows
        foreach (array_slice($rows, 0, 10) as $row) {
            $rowText = implode(' ', $row);
            $season = $this->extractSeasonFromText($rowText);
            if ($season) {
                $detectedSeason = $season;
                break;
            }
        }

        // TODO: Implement Boden-specific Excel parsing logic based on actual document structure
        // This will need to be customized based on Boden's actual Excel format

        return [
            'orders' => $orders,
            'metadata' => [
                'detected_season' => $detectedSeason,
                'extraction_method' => 'excel_sheets',
            ],
        ];
    }

    /**
     * Extract PO Number from document text
     */
    private function extractPoNumber(string $text): ?string
    {
        // Try multiple patterns for PO Number
        $patterns = [
            '/PO Number\s+(\d+)/',
            '/PO Number\s*:\s*(\d+)/',
            '/PO Number\s*\n\s*(\d+)/',
            '/PO\s+Number\s+(\d+)/',
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                return $matches[1];
            }
        }
        
        // If still not found, log the context for debugging
        if (preg_match('/PO Number.{0,100}/i', $text, $matches)) {
            \Log::info('PO Number context not matched: ' . $matches[0]);
        }
        
        return null;
    }

    /**
     * Extract PO Number from document metadata section
     */
    private function extractPoNumberFromMetadata(string $text): ?string
    {
        // Match "PO Number 668061" or "PO Number: 668061" format
        $patterns = [
            '/PO\s*Number\s+(\d{6,8})/',
            '/PO\s*Number\s*:\s*(\d{6,8})/',
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                return $matches[1];
            }
        }
        
        return null;
    }

    /**
     * Extract style code from document metadata section
     * Pattern like "Products / Materials Eva Cashmere Crew Cardigan K0862"
     */
    private function extractStyleCodeFromMetadata(string $text): ?string
    {
        // Look for style code patterns like K0862, K1250 etc
        // Usually after "Products / Materials" or standalone
        $patterns = [
            '/Products\s*\/\s*Materials\s+.*?([A-Z]\d{3,5})/s',
            '/\b([A-Z]\d{4,5})\b/', // Standalone style code like K0862
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                return $matches[1];
            }
        }
        
        return null;
    }

    /**
     * Extract product description from document metadata section
     */
    private function extractProductDescriptionFromMetadata(string $text): ?string
    {
        // Look for product description after "Products / Materials"
        if (preg_match('/Products\s*\/\s*Materials\s+(.+?)(?:[A-Z]\d{4,5}|\n\n)/s', $text, $matches)) {
            return trim(preg_replace('/\s+/', ' ', $matches[1]));
        }
        
        return null;
    }

    /**
     * Parse order table in new Boden format
     * Order lines start with 8-digit numbers like 66806101, 66806201
     */
    private function parseOrderTableNewFormat(array $rows, ?string $poNumber, ?string $styleCode, ?string $productDescription): array
    {
        $orderLines = [];
        
        foreach ($rows as $index => $row) {
            $firstColumn = trim($row[0] ?? '');
            
            // Skip empty rows, header rows, and summary rows
            if (empty($firstColumn)) {
                continue;
            }
            
            // Skip header rows (contain "Order", "Code", "Report", etc.)
            if (in_array(strtolower($firstColumn), ['order', 'code', 'report', '']) || 
                strpos(strtolower($firstColumn), 'order code') !== false) {
                continue;
            }
            
            // Skip subtotal rows like "01 (3)", "02 (3)"
            if (preg_match('/^\d{1,2}\s*\(\d+\)$/', $firstColumn)) {
                continue;
            }
            
            // Look for order line rows - 8-digit numbers starting with 6
            if (preg_match('/^6\d{7}$/', $firstColumn)) {
                $line = $this->parseBodenOrderLineNewFormat($row, $poNumber, $styleCode, $productDescription);
                if ($line) {
                    $orderLines[] = $line;
                    Log::debug('Boden order line parsed', [
                        'row_index' => $index,
                        'order_line' => $firstColumn,
                        'color_code' => $row[1] ?? 'N/A',
                    ]);
                }
            }
        }

        return $orderLines;
    }

    /**
     * Parse individual Boden order line from new format table row
     * Column structure:
     * 0: Order line number (e.g., 66806101)
     * 1: Color code (e.g., LGR, BLK)
     * 3: Product name part (e.g., "Eva")
     * 4: Color name (e.g., "Sapling", "Black")
     * 6: Order Qty (e.g., 235)
     * 7-11: Size quantities (XS, S, M, L, XL)
     * 12: Supplier CP1 price (e.g., 43.15)
     * 17-19: FOB Dates
     * 20: Delivery Date
     */
    private function parseBodenOrderLineNewFormat(array $row, ?string $poNumber, ?string $styleCode, ?string $productDescription): ?array
    {
        $orderLineNumber = trim($row[0] ?? '');
        $colourCode = trim($row[1] ?? '');
        $productName = trim($row[3] ?? '');
        $colourName = trim($row[4] ?? '');
        
        // Extract order qty from column 6
        $orderQty = (int) ($row[6] ?? 0);
        
        // Extract size quantities from columns 7-11
        $quantities = [];
        $price = (float) ($row[12] ?? 0);
        
        // Fallback price if not found
        if ($price <= 0) {
            $price = 43.15; // Default from documents
        }
        
        $sizeMapping = [
            'XS' => 7,
            'S' => 8,
            'M' => 9,
            'L' => 10,
            'XL' => 11,
        ];
        
        foreach ($sizeMapping as $size => $columnIndex) {
            $qty = (int) ($row[$columnIndex] ?? 0);
            if ($qty > 0) {
                $quantities[] = [
                    'size' => $size,
                    'qty' => $qty,
                    'price' => $price,
                    'commission' => 0,
                    'discount' => 0,
                ];
            }
        }
        
        // Skip if no quantities found
        if (empty($quantities)) {
            return null;
        }
        
        // Skip if no colour code
        if (empty($colourCode)) {
            return null;
        }
        
        // Extract dates from row
        $fobDate = $this->parseDateFromRow($row, 17) ?? $this->parseDateFromRow($row, 18) ?? $this->parseDateFromRow($row, 19);
        $deliveryDate = $this->parseDateFromRow($row, 20);
        
        // Build product description from row data if not from metadata
        $lineProductDescription = $productDescription ?? trim($productName . ' ' . ($row[7] ?? ''));
        
        return [
            '_po_number' => $poNumber,
            '_order_line_number' => $orderLineNumber,
            'colourway' => [
                'style_number' => $this->normalizeStyleNumber($styleCode ?? ''),
                'colour_code' => $colourCode,
                'colour_name' => $this->normalizeColourName($colourName),
                'product_description' => $lineProductDescription,
            ],
            'customer_exfty_date' => $fobDate ?? date('Y-m-d', strtotime('+30 days')),
            'customer_into_wh' => $deliveryDate ?? date('Y-m-d', strtotime('+60 days')),
            'quantities' => $quantities,
            'shipment_lines' => [
                ['exfty' => $fobDate ?? date('Y-m-d', strtotime('+30 days'))]
            ],
        ];
    }

    /**
     * Parse date from a specific row column
     */
    private function parseDateFromRow(array $row, int $columnIndex): ?string
    {
        $dateStr = trim($row[$columnIndex] ?? '');
        if (empty($dateStr)) {
            return null;
        }
        
        // Try parsing DD/MM/YYYY format
        if (preg_match('/(\d{1,2})\/(\d{1,2})\/(\d{4})/', $dateStr, $matches)) {
            $day = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
            $month = str_pad($matches[2], 2, '0', STR_PAD_LEFT);
            $year = $matches[3];
            return "{$year}-{$month}-{$day}";
        }
        
        return null;
    }

    /**
     * Extract order date from document text  
     */
    private function extractOrderDate(string $text): ?string
    {
        if (preg_match('/Issued On\s+(\d{1,2}\/\d{1,2}\/\d{4})/', $text, $matches)) {
            return $this->parseDate($matches[1]);
        }
        return null;
    }

    /**
     * Parse the orders table from Tabula extracted rows
     */
    private function parseOrderTable(array $rows): array
    {
        $orderLines = [];
        
        // Look for data rows that contain style codes
        foreach ($rows as $index => $row) {
            $firstColumn = $row[0] ?? '';
            
            // Look for rows that contain style codes like K1250
            if (preg_match('/([A-Z]+\d+)/', $firstColumn, $matches)) {
                $line = $this->parseBodenOrderLine($row);
                if ($line) {
                    $orderLines[] = $line;
                }
            }
        }

        return $orderLines;
    }

    /**
     * Find the header row of the orders table
     */
    private function findOrderTableHeader(array $rows): int
    {
        foreach ($rows as $index => $row) {
            $rowText = strtolower(implode(' ', $row));
            // Look for the row that contains "Code" and "Col Code" 
            if (strpos($rowText, 'code') !== false && strpos($rowText, 'col') !== false) {
                return $index;
            }
        }
        return -1;
    }

    /**
     * Create column mapping for Boden table structure
     */
    private function createBodenColumnMapping(array $headers): array
    {
        $map = ['sizes' => []];
        
        foreach ($headers as $index => $header) {
            $header = strtolower(trim($header));
            
            // Based on the actual table structure observed
            if ($header === 'code' || (strpos($header, 'code') !== false && strpos($header, 'col') === false)) {
                $map['style_code'] = $index;
            } elseif ($header === 'col' || strpos($header, 'col') !== false) {
                $map['colour_code'] = $index;
            } elseif (strpos($header, 'po product') !== false || strpos($header, 'product') !== false) {
                $map['product_description'] = $index;
            } elseif (strpos($header, 'supplier cp1') !== false || strpos($header, 'cp1') !== false) {
                $map['price'] = $index;
            } elseif (strpos($header, 'fob date plan') !== false) {
                $map['fob_date'] = $index;
            } elseif (strpos($header, 'delivery date contract') !== false) {
                $map['delivery_date'] = $index;
            } elseif (in_array($header, ['xs', 's', 'm', 'l', 'xl', 'xxl'])) {
                $map['sizes'][$header] = $index;
            }
        }
        
        return $map;
    }

    /**
     * Parse individual Boden order line from table row
     */
    private function parseBodenOrderLine(array $row): ?array
    {
        // Extract PO number and style code from the first column
        // Format: "66579801 K1250" where 66579801 is PO number and K1250 is style code
        $firstColumn = $row[0] ?? '';
        $poNumber = '';
        $styleCode = '';
        
        if (preg_match('/(\d{8})\s+([A-Z]+\d+)/', $firstColumn, $matches)) {
            $poNumber = $matches[1];
            $styleCode = $matches[2];
        } elseif (preg_match('/([A-Z]+\d+)/', $firstColumn, $matches)) {
            // Fallback if format is different
            $styleCode = $matches[1];
            // Look for PO number in nearby columns or use a default
            $poNumber = $this->extractPoFromRow($row);
        }
        
        // Color code is in column 1 (Col Code)
        $colourCode = $row[1] ?? '';
        
        // Product description spans multiple columns and rows - extract from text
        $productDescription = $this->extractProductDescription($row);
        
        if (empty($styleCode) || empty($colourCode) || empty($poNumber)) {
            return null;
        }

        // Extract quantities based on actual table structure
        $quantities = $this->extractQuantitiesFromBodenRow($row);
        
        if (empty($quantities)) {
            return null;
        }

        // For now, use default dates - these will need to be extracted from the full document
        $defaultDate = date('Y-m-d', strtotime('+30 days'));
        
        return [
            '_po_number' => $poNumber, // Internal field for grouping
            'colourway' => [
                'style_number' => $this->normalizeStyleNumber($styleCode),
                'colour_name' => $this->normalizeColourName($colourCode),
                'product_description' => $productDescription,
            ],
            'customer_exfty_date' => $defaultDate,
            'customer_into_wh' => $defaultDate,
            'quantities' => $quantities,
            'shipment_lines' => [
                ['exfty' => $defaultDate]
            ],
        ];
    }

    /**
     * Extract product description from row data
     */
    private function extractProductDescription(array $row): string
    {
        // Product description is typically in columns 2-4
        $description = '';
        for ($i = 2; $i <= 4; $i++) {
            if (!empty($row[$i])) {
                $description .= ' ' . $row[$i];
            }
        }
        return trim($description);
    }

    /**
     * Extract quantities from Boden table row based on actual structure
     */
    private function extractQuantitiesFromBodenRow(array $row): array
    {
        $quantities = [];
        $defaultCommission = 0; // From customer settings
        $defaultDiscount = 0; // From customer settings
        
        // Extract price from column 12 (Supplier CP1) based on actual table structure
        $price = (float)($row[12] ?? 0);
        
        // Fallback if price not found in expected column
        if ($price <= 0) {
            $price = 53.90; // Default fallback
        }
        
        // Based on actual column analysis:
        // Col 6: 276 (total), Col 7: 26 (XS), Col 8: 81 (S), Col 9: 102 (M), Col 10: 47 (L), Col 11: 20 (XL)
        $sizeMapping = [
            'XS' => 7,  // Correct column indexes based on actual table structure
            'S' => 8,
            'M' => 9, 
            'L' => 10,
            'XL' => 11,
        ];
        
        foreach ($sizeMapping as $size => $columnIndex) {
            $qty = (int) ($row[$columnIndex] ?? 0);
            if ($qty > 0) {
                $quantities[] = [
                    'size' => $size,
                    'qty' => $qty,
                    'price' => $price,
                    'commission' => $defaultCommission,
                    'discount' => $defaultDiscount,
                ];
            }
        }
        
        // If no quantities found in expected columns, try to find them elsewhere
        if (empty($quantities)) {
            // Look for any numeric values that could be quantities
            for ($i = 7; $i < min(20, count($row)); $i++) {
                $value = $row[$i] ?? '';
                if (is_numeric($value) && (int)$value > 0 && (int)$value < 1000) {
                    $quantities[] = [
                        'size' => 'M', // Default size if structure is unclear
                        'qty' => (int)$value,
                        'price' => $price,
                        'commission' => $defaultCommission,
                        'discount' => $defaultDiscount,
                    ];
                    break; // Just take the first quantity found
                }
            }
        }
        
        return $quantities;
    }

    /**
     * Extract PO number from row data as fallback
     */
    private function extractPoFromRow(array $row): string
    {
        // Look for 8-digit numbers in the row
        foreach ($row as $cell) {
            if (preg_match('/(\d{8})/', $cell, $matches)) {
                return $matches[1];
            }
        }
        
        // Fallback to any 6+ digit number
        foreach ($row as $cell) {
            if (preg_match('/(\d{6,})/', $cell, $matches)) {
                return $matches[1];
            }
        }
        
        return '';
    }
}