<?php

namespace App\Services\CommissionImports;

use App\Models\Seasons;
use App\Models\Customer;
use App\Models\Departments;
use App\Models\CustomerOrders;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrderLineQuantities;
use App\Models\CustomerOrderFiles;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use App\Models\Sizes;
use App\Models\CustomerAddress;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

class CommissionImportService
{
    private StyleMatchingService $styleMatchingService;

    public function __construct(StyleMatchingService $styleMatchingService)
    {
        $this->styleMatchingService = $styleMatchingService;
    }

    /**
     * Process commission import data and create orders
     */
    public function processImport(
        array $standardizedData,
        int $customerId,
        int $seasonId,
        int $departmentId,
        array $uploadedFiles = [],
        array $duplicateResolution = []
    ): array {
        $results = [
            'success' => false,
            'orders_created' => 0,
            'lines_created' => 0,
            'errors' => [],
        ];

        try {
            DB::beginTransaction();

            foreach ($standardizedData['orders'] as $orderData) {
                $po = strtoupper(trim($orderData['purchase_order_number'] ?? ''));
                $action = $duplicateResolution[$po]['action'] ?? null;
                $existingOrderId = $duplicateResolution[$po]['existing_order_id'] ?? null;

                if ($action === 'cancel') {
                    // Skip this order entirely
                    continue;
                }

                // Defensive: check DB for an existing order with same customer + PO to avoid duplicate insert
                $existingOrder = CustomerOrders::where('customers_id', $customerId)
                    ->where('customer_po', $po)
                    ->first();

                $createdNew = false;

                if ($action === 'overwrite' && ($existingOrderId || $existingOrder)) {
                    $order = $existingOrderId ? CustomerOrders::findOrFail($existingOrderId) : $existingOrder;
                    $this->clearOrderLines($order);
                    $order->update([
                        'seasons_id' => $seasonId,
                        'departments_id' => $departmentId,
                        'order_date' => Carbon::parse($orderData['order_date'])->format('Y-m-d'),
                        'incoterms' => $orderData['incoterms'] ?? 'FOB',
                    ]);
                } elseif ($action === 'add' && ($existingOrderId || $existingOrder)) {
                    $order = $existingOrderId ? CustomerOrders::findOrFail($existingOrderId) : $existingOrder;
                } else {
                    if ($existingOrder) {
                        // No explicit action but existing order found: default to add
                        $order = $existingOrder;
                    } else {
                        $order = $this->createOrder($orderData, $customerId, $seasonId, $departmentId);
                        $createdNew = true;
                    }
                }

                if ($createdNew) {
                    $results['orders_created']++;
                }

                // Store uploaded files
                $this->storeOrderFiles($order, $uploadedFiles);

                // Process order lines
                foreach ($orderData['lines'] as $lineData) {
                    $this->processOrderLine($order, $lineData, $customerId, $seasonId, $departmentId);
                    $results['lines_created']++;
                }
            }

            DB::commit();
            $results['success'] = true;

        } catch (\Exception $e) {
            DB::rollBack();
            $results['errors'][] = $e->getMessage();
            throw $e;
        }

        return $results;
    }

    /**
     * Remove all lines and children from an existing order (keeps the order record)
     */
    private function clearOrderLines(CustomerOrders $order): void
    {
        $lineIds = CustomerOrderLines::where('customer_orders_id', $order->id)->pluck('id');

        if ($lineIds->isEmpty()) return;

        // Delete quantities
        CustomerOrderLineQuantities::whereIn('customer_order_lines_id', $lineIds)->delete();

        // Delete shipment line sizes and shipment lines
        $shipmentLineIds = ShipmentLine::whereIn('customer_order_lines_id', $lineIds)->pluck('id');
        if ($shipmentLineIds->isNotEmpty()) {
            ShipmentLineSizes::whereIn('shipment_line_id', $shipmentLineIds)->delete();
            ShipmentLine::whereIn('id', $shipmentLineIds)->delete();
        }

        // Delete order lines
        CustomerOrderLines::whereIn('id', $lineIds)->delete();
    }

    /**
     * Validate standardized import data
     */
    public function validateImportData(array $data): array
    {
        $errors = [];

        if (!isset($data['orders']) || !is_array($data['orders'])) {
            $errors[] = 'Invalid data structure: orders array is required';
            return $errors;
        }

        foreach ($data['orders'] as $orderIndex => $order) {
            $orderErrors = $this->validateOrderData($order, $orderIndex);
            $errors = array_merge($errors, $orderErrors);
        }

        return $errors;
    }

    /**
     * Detect season from import data
     */
    public function detectSeason(array $data): ?Seasons
    {
        $detectedSeason = $data['metadata']['detected_season'] ?? null;
        
        if ($detectedSeason) {
            return Seasons::where('description', 'like', '%' . $detectedSeason . '%')->first();
        }

        // Try to extract from order data
        foreach ($data['orders'] as $order) {
            if (!empty($order['season'])) {
                $season = Seasons::where('description', 'like', '%' . $order['season'] . '%')->first();
                if ($season) {
                    return $season;
                }
            }
        }

        return null;
    }

    /**
     * Create customer order
     */
    private function createOrder(
        array $orderData,
        int $customerId,
        int $seasonId,
        int $departmentId
    ): CustomerOrders {
        $customerAddressId = null;
        
        // Get or create customer address if needed
        if (!empty($orderData['delivery_address'])) {
            $customerAddress = CustomerAddress::firstOrCreate([
                'name' => $orderData['delivery_address'],
                'customer_id' => $customerId,
            ], [
                'line1' => $orderData['delivery_address'],
                'countries_id' => Customer::find($customerId)->countries_id ?? 1,
            ]);
            $customerAddressId = $customerAddress->id;
        }

        return CustomerOrders::create([
            'customers_id' => $customerId,
            'seasons_id' => $seasonId,
            'departments_id' => $departmentId,
            'order_type' => 'commission',
            'customer_po' => $orderData['purchase_order_number'],
            'order_date' => Carbon::parse($orderData['order_date'])->format('Y-m-d'),
            'incoterms' => $orderData['incoterms'] ?? 'FOB',
            'customer_addresses_id' => $customerAddressId,
        ]);
    }

    /**
     * Process single order line
     */
    private function processOrderLine(
        CustomerOrders $order,
        array $lineData,
        int $customerId,
        int $seasonId,
        int $departmentId
    ): void {
        // Find or create style and colourway
        $styleData = [];
        if (!empty($lineData['colourway']['product_description'])) {
            $styleData['description'] = $lineData['colourway']['product_description'];
        }
        
        $style = $this->styleMatchingService->findOrCreateStyle(
            $lineData['colourway']['style_number'],
            $customerId,
            $seasonId,
            $departmentId,
            $styleData
        );

        $styleVersion = $this->styleMatchingService->getFirstStyleVersion($style);
        
        $colourway = $this->styleMatchingService->findOrCreateColourway(
            $lineData['colourway']['colour_name'],
            $styleVersion
        );

        // Create order line
        $orderLine = CustomerOrderLines::create([
            'customer_orders_id' => $order->id,
            'colourways_id' => $colourway->id,
            'factory_cust_date' => !empty($lineData['customer_exfty_date']) 
                ? Carbon::parse($lineData['customer_exfty_date'])->format('Y-m-d')
                : null,
        ]);

        // Process quantities
        $this->processQuantities($orderLine, $lineData['quantities']);

        // Process shipment lines
        $this->processShipmentLines($orderLine, $lineData);
    }

    /**
     * Process order line quantities
     */
    private function processQuantities(CustomerOrderLines $orderLine, array $quantities): void
    {
        // Create quantities in the correct order (by size ID)
        $sizesWithQuantities = [];
        
        // First, collect all sizes with their quantities
        foreach ($quantities as $quantityData) {
            $size = Sizes::firstOrCreate(['name' => $quantityData['size']]);
            $sizesWithQuantities[$size->id] = [
                'size' => $size,
                'data' => $quantityData,
            ];
        }
        
        // Sort by size ID to maintain database order
        ksort($sizesWithQuantities);
        
        // Create order line quantities in the correct order
        foreach ($sizesWithQuantities as $sizeData) {
            $quantityData = $sizeData['data'];
            
            CustomerOrderLineQuantities::create([
                'customer_order_lines_id' => $orderLine->id,
                'sizes_id' => $sizeData['size']->id,
                'qty' => $quantityData['qty'] ?? 0,
                'price' => $quantityData['price'] ?? 0,
                'commission' => $quantityData['commission'] ?? 0,
                'discount' => $quantityData['discount'] ?? 0,
            ]);
        }
    }

    /**
     * Process shipment lines
     */
    private function processShipmentLines(CustomerOrderLines $orderLine, array $lineData): void
    {
        $shipmentData = $lineData['shipment_lines'][0] ?? [];
        $exftyDate = $shipmentData['exfty'] ?? $lineData['customer_exfty_date'];

        $shipmentLine = ShipmentLine::create([
            'customer_order_lines_id' => $orderLine->id,
            'exfty' => Carbon::parse($exftyDate)->format('Y-m-d'),
        ]);

        // Create shipment line sizes in the correct order (by size ID)
        $sizesWithQuantities = [];
        
        // First, collect all sizes with their quantities
        foreach ($lineData['quantities'] as $quantityData) {
            $size = Sizes::firstOrCreate(['name' => $quantityData['size']]);
            $sizesWithQuantities[$size->id] = [
                'size' => $size,
                'qty' => $quantityData['qty'] ?? 0,
            ];
        }
        
        // Sort by size ID to maintain database order
        ksort($sizesWithQuantities);
        
        // Create shipment line sizes in the correct order
        foreach ($sizesWithQuantities as $sizeData) {
            ShipmentLineSizes::create([
                'shipment_line_id' => $shipmentLine->id,
                'sizes_id' => $sizeData['size']->id,
                'qty' => $sizeData['qty'],
            ]);
        }
    }

    /**
     * Store uploaded files with order
     */
    private function storeOrderFiles(CustomerOrders $order, array $uploadedFiles): void
    {
        foreach ($uploadedFiles as $file) {
            if ($file instanceof TemporaryUploadedFile) {
                $filename = $file->getClientOriginalName();
                $path = $file->store('customer-orders', 'public');

                CustomerOrderFiles::create([
                    'customer_orders_id' => $order->id,
                    'filename' => $filename,
                    'file' => $path,
                ]);
            }
        }
    }

    /**
     * Validate individual order data
     */
    private function validateOrderData(array $order, int $index): array
    {
        $errors = [];
        $prefix = "Order {$index}: ";

        if (empty($order['purchase_order_number'])) {
            $errors[] = $prefix . 'Purchase order number is required';
        }

        if (empty($order['order_date'])) {
            $errors[] = $prefix . 'Order date is required';
        } elseif (!strtotime($order['order_date'])) {
            $errors[] = $prefix . 'Invalid order date format';
        }

        if (!isset($order['lines']) || !is_array($order['lines']) || empty($order['lines'])) {
            $errors[] = $prefix . 'At least one order line is required';
        } else {
            foreach ($order['lines'] as $lineIndex => $line) {
                $lineErrors = $this->validateOrderLineData($line, $index, $lineIndex);
                $errors = array_merge($errors, $lineErrors);
            }
        }

        return $errors;
    }

    /**
     * Validate individual order line data
     */
    private function validateOrderLineData(array $line, int $orderIndex, int $lineIndex): array
    {
        $errors = [];
        $prefix = "Order {$orderIndex}, Line {$lineIndex}: ";

        if (empty($line['colourway']['style_number'])) {
            $errors[] = $prefix . 'Style number is required';
        }

        if (empty($line['colourway']['colour_name'])) {
            $errors[] = $prefix . 'Colour name is required';
        }

        if (empty($line['customer_exfty_date'])) {
            $errors[] = $prefix . 'Customer ex-factory date is required';
        } elseif (!strtotime($line['customer_exfty_date'])) {
            $errors[] = $prefix . 'Invalid customer ex-factory date format';
        }

        if (!isset($line['quantities']) || !is_array($line['quantities']) || empty($line['quantities'])) {
            $errors[] = $prefix . 'At least one quantity is required';
        } else {
            $totalQty = 0;
            foreach ($line['quantities'] as $qtyIndex => $qty) {
                if (empty($qty['size'])) {
                    $errors[] = $prefix . "Quantity {$qtyIndex}: Size is required";
                }
                if (!isset($qty['qty']) || !is_numeric($qty['qty']) || $qty['qty'] <= 0) {
                    $errors[] = $prefix . "Quantity {$qtyIndex}: Valid quantity > 0 is required";
                } else {
                    $totalQty += $qty['qty'];
                }
                if (!isset($qty['price']) || !is_numeric($qty['price']) || $qty['price'] <= 0) {
                    $errors[] = $prefix . "Quantity {$qtyIndex}: Valid price > 0 is required";
                }
            }
            
            if ($totalQty <= 0) {
                $errors[] = $prefix . 'Total quantity must be greater than 0';
            }
        }

        return $errors;
    }
}

