<?php

namespace App\Services\CommissionImports;

use App\Services\CommissionImports\Extractors\BodenExtractor;
use App\Services\CommissionImports\Extractors\ChintiExtractor;
use App\Services\CommissionImports\Extractors\EwmExtractor;
use App\Services\CommissionImports\Extractors\MrPorterExtractor;
use App\Services\CommissionImports\Extractors\PureExtractor;
use App\Services\CommissionImports\Extractors\CharlesTyrwhittExtractor;

class CommissionExtractorFactory
{
    /**
     * Available extractors mapped by customer name
     */
    private static array $extractors = [
        'boden' => BodenExtractor::class,
        'chinti' => ChintiExtractor::class,
        'ewm' => EwmExtractor::class,
        'edinburgh woollen mill' => EwmExtractor::class,
        'mr porter' => MrPorterExtractor::class,
        'pure' => PureExtractor::class,
        'pure collection' => PureExtractor::class,
        'charles tyrwhitt' => CharlesTyrwhittExtractor::class,
    ];

    /**
     * Create extractor instance for given customer
     */
    public static function make(string $customerName): BaseExtractor
    {
        $customerName = strtolower(trim($customerName));
        
        if (!isset(self::$extractors[$customerName])) {
            throw new \InvalidArgumentException("No extractor found for customer: {$customerName}");
        }

        $extractorClass = self::$extractors[$customerName];
        
        if (!class_exists($extractorClass)) {
            throw new \RuntimeException("Extractor class does not exist: {$extractorClass}");
        }

        return new $extractorClass();
    }

    /**
     * Get list of available customers
     */
    public static function getAvailableCustomers(): array
    {
        $customers = [];
        $seenNames = [];

        foreach (self::$extractors as $key => $class) {
            if (!class_exists($class)) continue;

            $instance = new $class();
            $name = $instance->getCustomerName();

            if (in_array($name, $seenNames, true)) continue; // avoid duplicate labels

            $customers[$key] = $name;
            $seenNames[] = $name;
        }

        return $customers;
    }

    /**
     * Register a new customer extractor
     */
    public static function register(string $customerKey, string $extractorClass): void
    {
        if (!is_subclass_of($extractorClass, BaseExtractor::class)) {
            throw new \InvalidArgumentException("Extractor class must extend BaseExtractor");
        }
        
        self::$extractors[$customerKey] = $extractorClass;
    }

    /**
     * Check if customer extractor exists
     */
    public static function hasExtractor(string $customerName): bool
    {
        $customerName = strtolower(trim($customerName));
        return isset(self::$extractors[$customerName]);
    }
}


