<?php

namespace App\Services\CommissionImports;

use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

abstract class BaseExtractor
{
    /**
     * Extract data from uploaded file and return standardized format
     */
    abstract public function extractData(TemporaryUploadedFile $file): array;

    /**
     * Get the customer name this extractor handles
     */
    abstract public function getCustomerName(): string;

    /**
     * Get the customer ID from database
     */
    abstract public function getCustomerId(): int;

    /**
     * Validate customer-specific data rules
     */
    abstract public function validateCustomerData(array $data): array;

    /**
     * Get supported file extensions for this customer
     */
    public function getSupportedExtensions(): array
    {
        return ['pdf', 'xlsx', 'xls'];
    }

    /**
     * Check if file extension is supported
     */
    public function isFileSupported(TemporaryUploadedFile $file): bool
    {
        $extension = strtolower($file->getClientOriginalExtension());
        return in_array($extension, $this->getSupportedExtensions());
    }

    /**
     * Extract season patterns from text content
     */
    protected function extractSeasonFromText(string $text): ?string
    {
        // Look for patterns like AW24, SS25, AW2024, SS2025
        if (preg_match('/(AW|SS)(\d{2,4})/i', $text, $matches)) {
            $season = strtoupper($matches[1]);
            $year = $matches[2];
            
            // Convert 4-digit years to 2-digit
            if (strlen($year) === 4) {
                $year = substr($year, -2);
            }
            
            return $season . $year;
        }
        
        return null;
    }

    /**
     * Normalize style number format
     */
    protected function normalizeStyleNumber(string $styleNumber): string
    {
        $normalized = trim(strtoupper($styleNumber));
        
        // Remove leading size letters that might be incorrectly prepended (e.g., "SMPK252017" -> "MPK252017")
        // Mr Porter style numbers typically start with "MP"
        if (preg_match('/^([SMLX]+)(MP[A-Z0-9]+)$/i', $normalized, $matches)) {
            return $matches[2];
        }
        
        return $normalized;
    }

    /**
     * Normalize colour name
     */
    protected function normalizeColourName(string $colourName): string
    {
        return trim(ucwords(strtolower($colourName)));
    }

    /**
     * Parse date from various formats
     */
    protected function parseDate(string $dateString): ?string
    {
        $dateString = trim($dateString);
        
        if (empty($dateString)) {
            return null;
        }

        // Try common date formats
        $formats = [
            'Y-m-d',
            'd/m/Y',
            'm/d/Y',
            'd-m-Y',
            'm-d-Y',
            'Y/m/d',
            'd.m.Y',
            'j/n/Y',
            'j-n-Y',
            'j F Y', // 19 May 2025
            'd F Y', // 19 May 2025
        ];

        foreach ($formats as $format) {
            $date = \DateTime::createFromFormat($format, $dateString);
            if ($date && $date->format($format) === $dateString) {
                return $date->format('Y-m-d');
            }
        }

        // Try strtotime as fallback
        $timestamp = strtotime($dateString);
        if ($timestamp !== false) {
            return date('Y-m-d', $timestamp);
        }

        return null;
    }

    /**
     * Clean numeric value
     */
    protected function cleanNumeric(string $value): float
    {
        // Remove currency symbols, spaces, and commas
        $cleaned = preg_replace('/[£$€,\s]/', '', $value);
        return (float) $cleaned;
    }
}


