<?php

namespace App\Notifications;

use App\Models\CommissionImportBatch;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class CommissionImportComplete extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public CommissionImportBatch $batch
    ) {}

    public function via($notifiable): array
    {
        return ['mail'];
    }

    public function toMail($notifiable): MailMessage
    {
        $batch = $this->batch;
        $order = $batch->commissionOrder;

        $matchedCount = $batch->matched_lines_count;
        $unmatchedCount = $batch->unmatched_lines_count;
        $totalLines = $matchedCount + $unmatchedCount;
        $matchPercent = $totalLines > 0 ? round(($matchedCount / $totalLines) * 100) : 0;

        $url = route('commission.skeleton-orders') . '?batch=' . $batch->id;

        $message = (new MailMessage)
            ->subject("Commission Import Complete: {$batch->batch_name}")
            ->greeting("Hello {$notifiable->name},")
            ->line("Your commission order import has been processed.")
            ->line("**Batch:** {$batch->batch_name}")
            ->line("**Customer:** {$batch->customer_name}")
            ->line("**Files processed:** {$batch->processed_files} of {$batch->total_files}")
            ->line("**Lines matched:** {$matchedCount}/{$totalLines} ({$matchPercent}%)");

        if ($unmatchedCount > 0) {
            $message->line("⚠️ **{$unmatchedCount} lines need manual matching**");
        }

        $message->action('View Skeleton Order', $url)
            ->line('You can review and forward this order to PD for confirmation.');

        return $message;
    }
}


