<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\BaseModel;
use Illuminate\Database\Eloquent\SoftDeletes;
use OwenIt\Auditing\Contracts\Auditable;

class YarnOrder extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;


    protected static function booted(): void
    {
        parent::booted();
        static::deleting(function($model) {
            $model->yarn_order_lines?->each?->delete();
            $model->yarn_order_files?->each?->delete();
        });
        static::restoring(function($model) {
            $model->yarn_order_lines()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->yarn_order_files()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
        });
    }


    protected $casts = [
        'date' => 'date:Y-m-d',
        'created_at' => 'date:Y-m-d',
        'updated_at' => 'date:Y-m-d',
        'deleted_at' => 'date:Y-m-d',
    ];

    protected $fillable = [
        'suppliers_id',
        'date',
        'created_at',
        'updated_at',
        'comments',
        'invoice_to_id',
        'deliver_to_id',
        'customers_id',
        'seasons_id',
        'deleted_at',
    ];

    public function scopeSearch($query, $value)
    {
        $sanitizedValue = preg_replace('/[^a-zA-Z0-9\s-]/', '', $value);
        // Split the search string into multiple terms
        $terms = preg_split('/\s+/', trim($value));

        $query->where(function($query) use ($terms) {
            foreach ($terms as $term) {
                // For each term, require at least one matching condition
                $query->where(function($q) use ($term) {
                    $q->where('yarn_orders.id', 'like', "%{$term}%")
                    ->orWhereRelation('yarn_order_lines.yarn_colours', 'description', 'like', "%{$term}%")
                    ->orWhereRelation('yarn_order_lines.yarn_colours.yarn', 'description', 'like', "%{$term}%")
                    ->orWhereRelation('yarn_order_lines.yarn_colours.yarn.counts', 'count', 'like', "%{$term}%")
                    ->orWhereRelation('yarn_order_lines.yarn_order_line_styles.customer_order_lines.colourways', 'name', 'like', "%{$term}%")
                    ->orWhereRelation('yarn_order_lines.yarn_order_line_styles.customer_order_lines.colourways.style_versions.factories', 'name', 'like', "%{$term}%")
                    ->orWhereRelation('yarn_order_lines.yarn_order_line_styles.customer_order_lines.colourways.style_versions.styles', 'customer_ref', 'like', "%{$term}%")
                    ->orWhereRelation('yarn_order_lines.yarn_order_line_styles.customer_order_lines.colourways.style_versions.styles', 'designs_id', 'like', "%{$term}%")
                    ->orWhereRelation('yarn_order_lines.yarn_order_line_styles.customer_order_lines.colourways.style_versions.styles.designs', 'description', 'like', "%{$term}%")
                    ->orWhereRelation('customers', 'name', 'like', "%{$term}%")
                    ->orWhereRelation('suppliers', 'name', 'like', "%{$term}%");
                });
            }
        });
    }

	public function customers()
    {
        return $this->belongsTo(Customer::class);
    }
    public function suppliers()
    {
        return $this->belongsTo(Suppliers::class);
    }
	public function seasons()
    {
        return $this->belongsTo(Seasons::class);
    }
    public function invoice_to()
    {
        return $this->belongsTo(Suppliers::class);
    }
    public function deliver_to()
    {
        return $this->belongsTo(Suppliers::class);
    }
    public function yarn_order_lines()
    {
        return $this->hasMany(YarnOrderLine::class);
    }
    public function yarn_order_files()
    {
        return $this->hasMany(YarnOrderFiles::class);
    }
}
