<?php

namespace App\Models;

use App\Casts\TitleCase;
use \Bkwld\Cloner\Cloneable;
use App\Models\BaseModel;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

/**
 * Yarn model
 * 
 * Represents yarn specifications used in the system.
 */
class Yarn extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use Cloneable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'suppliers_id',
        'description',
        'counts_id',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'description' => TitleCase::class,
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::deleting(function ($model) {
            $model->yarn_compositions?->each?->delete();
            $model->yarn_colours?->each?->delete();
        });

        static::restoring(function ($model) {
            $model->yarn_compositions()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->yarn_colours()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
        });

        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('description');
        });
    }

    /**
     * Get the supplier for this yarn.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function suppliers()
    {
        return $this->belongsTo(Suppliers::class);
    }

    /**
     * Get the count for this yarn.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function counts()
    {
        return $this->belongsTo(Count::class);
    }

    /**
     * Get the yarn compositions for this yarn.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function yarn_compositions()
    {
        return $this->hasMany(YarnComposition::class);
    }

    /**
     * Get the yarn colours for this yarn.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function yarn_colours()
    {
        return $this->hasMany(YarnColours::class);
    }

    /**
     * Scope a query to search for yarns.
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $value
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, string $value)
    {
        return $query->where('description', 'like', '%' . $value . '%')
            ->orWhereRelation('counts', 'count', 'like', '%' . $value . '%')
            ->orWhereRelation('yarn_colours', 'reference', 'like', '%' . $value . '%')
            ->orWhereRelation('yarn_colours', 'description', 'like', '%' . $value . '%')
            ->orWhereRelation('suppliers', 'name', 'like', '%' . $value . '%');
    }
}