<?php

namespace App\Models;

use Laravel\Jetstream\HasTeams;
use Laravel\Sanctum\HasApiTokens;
use OwenIt\Auditing\Models\Audit;
use Laravel\Jetstream\HasProfilePhoto;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Laravel\Fortify\TwoFactorAuthenticatable;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;

class User extends Authenticatable
{
    use SoftDeletes;
    use HasApiTokens;
    use HasFactory;
    use HasProfilePhoto;
    use HasTeams;
    use Notifiable;
    use TwoFactorAuthenticatable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'email_verified_at',
        'password',
        'two_factor_secret',
        'two_factor_recovery_codes',
        'two_factor_confirmed_at',
        'remember_token',
        'current_team_id',
        'profile_photo_path',
        'settings',
        'phone',
        'zoho_refresh_token',
        'google_access_token',
        'google_refresh_token',
        'google_token_expires_at',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_recovery_codes',
        'two_factor_secret',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'settings' => 'array',
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
        'google_token_expires_at' => 'datetime',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array<int, string>
     */
    protected $appends = [
        'profile_photo_url',
    ];

    /**
     * Get the latest audit for this user.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function latestAudit()
    {
        return $this->hasOne(Audit::class)->latestOfMany();
    }

    /**
     * Get the teams for this user.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function teams()
    {
        return $this->belongsToMany(Team::class)->withPivot('role');
    }

    /**
     * Retrieve a setting with a given name or fall back to the default.
     * 
     * @param string $name
     * @param mixed $default
     * @return mixed
     */
    public function setting(string $name, $default = null)
    {
        if ($this->settings != null && array_key_exists($name, $this->settings)) {
            return $this->settings[$name];
        }
        return $default;
    }

    /**
     * Update one or more settings and then optionally save the model.
     * 
     * @param array $revisions
     * @param bool $save
     * @return self
     */
    public function settings(array $revisions, bool $save = true): self
    {
        if ($this->settings != null) {
            $this->settings = array_merge($this->settings, $revisions);
        } else {
            $this->settings = $revisions;
        }
        
        if ($save) {
            $this->save();
        }
        
        return $this;
    }

    /**
     * Get the user departments for this user.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function user_departments()
    {
        return $this->hasMany(UserDepartment::class, 'users_id');
    }

    /**
     * Get the functional roles for this user.
     */
    public function functionalRoles()
    {
        return $this->belongsToMany(FunctionalRole::class, 'user_functional_roles');
    }

    /**
     * Check if user has a specific functional role.
     */
    public function hasRole(string $roleName): bool
    {
        return $this->functionalRoles()->where('name', $roleName)->exists();
    }
}
