<?php

namespace App\Models;

use App\Casts\Upper;
use App\Casts\TitleCase;
use App\Models\BaseModel;
use App\Models\YarnOrder;
use Illuminate\Support\Facades\Cache;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

/**
 * Supplier model
 * 
 * Represents suppliers/factories in the system.
 */
class Suppliers extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::saved(function ($supplier) {
            if ($supplier->isDirty('currency')) {
                Cache::tags(['yarn_costs'])->flush();
                Cache::tags(["prices"])->flush();
            }
            self::invalidatePriceResolutions($supplier);
        });

        static::deleted(function ($supplier) {
            self::invalidatePriceResolutions($supplier);
        });

        static::deleting(function ($model) {
            $model->shipments?->each?->delete();
            $model->yarns?->each?->delete();
            $model->supplier_addresses?->each?->delete();
            $model->style_versions?->each?->delete();
            $model->yarn_orders?->each?->delete();
        });

        static::restoring(function ($model) {
            $model->shipments()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->yarns()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->supplier_addresses()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->style_versions()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->yarn_orders()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
        });

        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('name');
        });

        static::updated(function ($factory) {
            ShipmentLine::where('factory_id', $factory->id)->update([
                'factory_id' => $factory->id,
            ]);
        });
    }

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'name' => TitleCase::class,
        'contact' => TitleCase::class,
        'finance_contact' => TitleCase::class,
        'mid' => Upper::class,
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'type',
        'name',
        'countries_id',
        'contact',
        'email',
        'phone',
        'finance_contact',
        'finance_email',
        'finance_phone',
        'payment_terms',
        'currency',
        'customs',
        'mid',
        'logo',
        'yc_email',
    ];

    /**
     * Get the country for this supplier.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function countries()
    {
        return $this->belongsTo(Countries::class);
    }

    /**
     * Get the shipments for this supplier.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function shipments()
    {
        return $this->hasMany(Shipment::class);
    }

    /**
     * Get the yarns for this supplier.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function yarns()
    {
        return $this->hasMany(Yarn::class);
    }

    /**
     * Get the supplier addresses for this supplier.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function supplier_addresses()
    {
        return $this->hasMany(SupplierAddress::class);
    }

    /**
     * Get the style versions for this supplier (as factory).
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function style_versions()
    {
        return $this->hasMany(StyleVersions::class, 'factory_id');
    }

    /**
     * Get the yarn orders for this supplier.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function yarn_orders()
    {
        return $this->hasMany(YarnOrder::class);
    }


    /**
     * Scope a query to search for suppliers.
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $value
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, string $value)
    {
        return $query->where('name', 'like', "%{$value}%")
            ->orWhere('id', $value)
            ->orWhere('currency', $value)
            ->orWhereRelation('countries', 'country', 'like', "%{$value}%")
            ->orWhereRelation('countries', 'code', 'like', "%{$value}%");
    }

    /**
     * Invalidate price resolutions for a supplier.
     * 
     * @param \App\Models\Suppliers $supplier
     * @return void
     */
    protected static function invalidatePriceResolutions($supplier): void
    {
        try {
            $service = app(\App\Services\PriceResolutionService::class);
            $service->invalidateAndWarmupBySupplier($supplier->id);
        } catch (\Exception $e) {
            \Log::error('Failed to invalidate price resolutions for supplier', [
                'supplier_id' => $supplier->id,
                'error' => $e->getMessage()
            ]);
        }
    }
}
