<?php

namespace App\Models;

use App\Casts\Boolean;
use \Bkwld\Cloner\Cloneable;
use Illuminate\Support\Facades\Auth;
use App\Models\BaseModel;
use App\Jobs\SuggestCommodityCode;
use Illuminate\Database\Eloquent\Scope;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Styles extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use Cloneable;

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::deleting(function ($model) {
            $model->colourways?->each?->delete();
            $model->prices?->each?->delete();
            $model->style_versions?->each?->delete();
        });

        static::restoring(function ($model) {
            $model->style_versions()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
        });

        // Dispatch AI commodity code suggestion when a style is created
        static::created(function ($style) {
            SuggestCommodityCode::dispatch($style->id);
        });
    }

    /**
     * Trigger commodity code suggestion for this style.
     * Called from related models (e.g., ColourwayYarns) when yarn changes occur.
     */
    public function suggestCommodityCode(bool $forceUpdate = false): void
    {
        SuggestCommodityCode::dispatch($this->id, $forceUpdate);
    }



    protected $cloneable_relations = [
        'style_versions',
    ];

	/**
     * Attributes to include in the Audit.
     *
     * @var array
     */
	protected $auditInclude = [];

    protected $casts = [
        'cancelled'  => Boolean::class,
        'carryover'  => Boolean::class,
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];

    /**
     * Normalize customer_ref to empty string if null.
     * Prevents duplicate styles bypassing the unique constraint
     * (MySQL treats NULL and '' as different values in unique indexes).
     */
    public function setCustomerRefAttribute($value): void
    {
        $this->attributes['customer_ref'] = $value ?? '';
    }
    protected $fillable = [
        'designs_id',
        'departments_id',
        'seasons_id',
        'customers_id',
        'customer_ref',
        'commodity_codes_id',
        'notes',
        'created_at',
        'updated_at',
        'carryover',
        'category',
        'cancelled',
        'deleted_at',
        'customer_samp_no',
        'green_seal_approval_comments',
        'photo_shoot_sample_comments',
        'intake_id'
    ];


    public function getHasOrdersAttribute(){
        return $this->style_versions()
            ->whereHas('colourways.customer_order_lines')
            ->exists();
    }

    public function scopeSearch($query, $value, $includePrices = false)
    {
        $sanitizedValue = preg_replace('/[^a-zA-Z0-9\s-]/', '', $value);
        // Split the search string into multiple terms
        $terms = preg_split('/\s+/', trim($value));

        $query->where(function($query) use ($terms, $includePrices) {
            foreach ($terms as $term) {
                // For each term, require at least one matching condition
                $query->where(function($q) use ($term, $includePrices) {
                    $q->where('styles.customer_ref', 'like', "%{$term}%")
                      ->orWhere('styles.designs_id', 'like', "%{$term}%")
                      ->orWhere('category', 'like', "%{$term}%")
                      ->orWhereRelation('customers', 'name', 'like', "%{$term}%")
                      ->orWhereRelation('designs', 'description', 'like', "%{$term}%")
                      ->orWhereRelation('style_versions.factories', 'name', 'like', "%{$term}%")
                      ->orWhereRelation('style_versions.colourways.colourway_yarns.yarn_colours.yarn', 'description', 'like', "%{$term}%")
                      ->orWhereRelation('style_versions.colourways.colourway_yarns.yarn_colours.yarn.counts', 'count', 'like', "%{$term}%")
                      ->orWhereRelation('style_versions.colourways.colourway_yarns.yarn_colours', 'reference', 'like', "%{$term}%")
                      ->orWhereRelation('style_versions.colourways.colourway_yarns.yarn_colours', 'description', 'like', "%{$term}%");

                    if($includePrices){
                        $q->orWhereRelation('style_versions.prices', 'name', 'like', "%{$term}%")
                        ->orWhereRelation('style_versions.prices', 'quote', 'like', "%{$term}%")
                        ->orWhereRelation('style_versions.prices', 'cmt', 'like', "%{$term}%");
                    }
                });
            }
        });
    }

    public function scopeFilterBy($query, $filters)
    {
        return $query
            ->when($filters['season'], fn ($q) => $q->where('season', $filters['season']))
            ->when($filters['search'], fn ($q) => $q->where('description', 'like', "%{$filters['search']}%"))
            ->when($filters['customer'], fn ($q) => $q->where('customer', $filters['customer']));
    }

    public function firstImageColourway()
    {
        return $this->hasOneThrough(
            Colourways::class,
            StyleVersions::class,
            'styles_id',          // Foreign key on style_versions table
            'style_versions_id',  // Foreign key on colourways table
            'id',                 // Local key on styles table
            'id'                  // Local key on style_versions table
        )
        ->whereNotNull('colourways.img_thumb')
        ->where('colourways.img_thumb', '!=', '')
        ->orderBy('colourways.id') // Pick the first valid image thumbnail
        ->select([
            'colourways.id as colourway_id',
            'colourways.img_thumb'
        ]);
    }





    public function getImage(){ //OLD???
        foreach($this->style_versions as $version){
            foreach($version->colourways as $colourway){
                if($colourway->image != NULL)
                    return $version->getImage();
            }
        }
    }

    public function getCountPricesAttribute(){
        $i = 0;
        foreach($this->style_versions as $version){
            $i = $i + $version->prices->count();
        }
        return $i;
    }

    public function getGotPricesAttribute(){
        $i = 0;
        foreach($this->style_versions as $version){
            $i = $i + $version->prices->count();
        }
        if($i > 0)
            return 1;
        else
            return 0;
    }

    public function saveWithUser(){
        $this->last_updated_by = auth()->user()->id;
        $this->save();
    }
    public function quote($currency){
        $prices = Price::whereRelation('style_versions', 'styles_id', '=', $this->id)->where('sizes_id', 32)->get();
        $return = [];
        if($prices->where('quote_agreed', '=', TRUE)->count() == 1){
            return [$prices->where('quote_agreed', '=', TRUE)->first()->quoteCurrency('£')];
        }
        foreach($prices as $price){
            $return[] = $price->quoteCurrency($currency);
        }
        return $return;
    }
    public function quoteStatus(){
        $prices = Price::whereRelation('style_versions', 'styles_id', '=', $this->id)->where('quote_agreed', '=', TRUE)->count();
        if($prices == 0)
            return 'Not Accepted';
        elseif($prices == 1)
            return 'Confirmed';
        else
            return 'Error - Multiple prices accepted';
    }
    public function quoteStatusFast($prices){
        $prices = $prices->where('style_versions.styles_id', $this->id)->where('quote_agreed', '=', TRUE)->count();
        if($prices == 0)
            return 'Not Accepted';
        elseif($prices == 1)
            return 'Confirmed';
        else
            return 'Error - Multiple prices accepted';
    }

    public function intakes()
    {
        return $this->belongsTo(Intake::class);
    }
    public function departments()
    {
        return $this->belongsTo(Departments::class);
    }
    public function seasons()
    {
        return $this->belongsTo(Seasons::class);
    }
    public function customers()
    {
        return $this->belongsTo(Customer::class);
    }
    public function commodity_codes_id()
    {
        return $this->belongsTo(CommodityCodes::class);
    }
    public function designs()
    {
        return $this->belongsTo(Design::class);
    }
    public function version_to_use()
    {
        return $this->belongsTo(StyleVersions::class, 'use_version');
    }

	public function style_versions()
    {
        return $this->hasMany(StyleVersions::class, 'styles_id');
    }

}


// class UserDepartmentScope implements Scope
// {
//     public function apply(Builder $builder, Model $model)
//     {
//         $user = Auth::user(); // Assuming you have user authentication

//         if ($user) {
//             $builder->whereHas('departments', function ($query) use ($user) {
//                 $query->whereIn('departments_id', $user->user_departments->pluck('departments_id')); // Replace with your logic for department ID
//             });
//         }
//     }
// }
