<?php

namespace App\Models;

use Bkwld\Cloner\Cloneable;
use App\Models\BaseModel;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Services\TotalCacheService;

class ShipmentLineSizes extends BaseModel
{
    use SoftDeletes;
    use HasFactory;
    use Cloneable;
    protected $casts = [
        'qty'         => 'integer',
        'shipped_qty' => 'integer',
        'created_at'  => 'datetime:Y-m-d',
        'updated_at'  => 'datetime:Y-m-d',
        'deleted_at'  => 'datetime:Y-m-d',
    ];
	protected $fillable = [
        'sizes_id',
        'shipment_line_id',
        'qty',
        'created_at',
        'updated_at',
        'shipped_qty',
        'deleted_at',
    ];



    public function sizes()
    {
        return $this->belongsTo(Sizes::class);
    }
    public function shipment_lines()
    {
        return $this->belongsTo(ShipmentLine::class, 'shipment_line_id');
    }

    protected static function booted(): void
    {
        parent::booted();
        // Total cache invalidation hooks
        static::saved(function($model) {
            $model->invalidateTotalCache();
        });

        static::updated(function($model) {
            $model->invalidateTotalCache();
        });

        static::deleted(function($model) {
            $model->invalidateTotalCache();
        });
    }

    /**
     * Invalidate and warm up total cache for related entities.
     */
    public function invalidateTotalCache(): void
    {
        try {
            $service = app(TotalCacheService::class);
            
            // Invalidate and warm up related shipment line cache
            if ($this->shipment_line_id) {
                $service->invalidateEntity('shipment_line', $this->shipment_line_id);
                
                // Get the shipment line to invalidate and warm up related caches
                $shipmentLine = ShipmentLine::find($this->shipment_line_id);
                if ($shipmentLine) {
                    // Warm up shipment line cache
                    $service->warmupShipmentLine($shipmentLine);
                    
                    // Invalidate and warm up related customer order line cache
                    if ($shipmentLine->customer_order_lines_id) {
                        $service->invalidateEntity('customer_order_line', $shipmentLine->customer_order_lines_id);
                        $service->warmupCustomerOrderLine($shipmentLine->customer_order_lines);
                    }
                    
                    // Invalidate and warm up related customer order cache
                    if ($shipmentLine->customer_order_lines && $shipmentLine->customer_order_lines->customer_orders_id) {
                        $service->invalidateEntity('customer_order', $shipmentLine->customer_order_lines->customer_orders_id);
                        $service->warmupCustomerOrder($shipmentLine->customer_order_lines->customer_orders);
                    }
                    
                    // Invalidate and warm up related shipment cache
                    if ($shipmentLine->shipment_id) {
                        $service->invalidateEntity('shipment', $shipmentLine->shipment_id);
                        $shipment = $shipmentLine->shipment;
                        if ($shipment) {
                            $service->warmupShipment($shipment);
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            \Log::error("Failed to invalidate and warm up total cache for shipment line size {$this->id}: " . $e->getMessage());
        }
    }
}
