<?php

namespace App\Models;

use App\Casts\Boolean;
use App\Casts\Currency;
use App\Casts\TitleCase;
use Bkwld\Cloner\Cloneable;
use App\Models\BaseModel;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Services\TotalCacheService;

class Shipment extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use Cloneable;


    public function scopeSearch($query, $value)
    {
        $sanitizedValue = preg_replace('/[^a-zA-Z0-9\s-]/', '', $value);
        // Split the search string into multiple terms
        $terms = preg_split('/\s+/', trim($value));

        $query->where(function($query) use ($terms) {
            foreach ($terms as $term) {
                // For each term, require at least one matching condition
                $query->where(function($q) use ($term) {
                    $q->where('id', 'like', "%{$term}%")
                    ->orwhereRelation('shipment_lines.customer_order_lines.colourways', 'name', 'like', "%{$term}%")
                    ->orwhereRelation('shipment_lines.customer_order_lines.colourways.style_versions', 'name', 'like', "%{$term}%")
                    ->orwhereRelation('shipment_lines.customer_order_lines.colourways.style_versions.styles', 'designs_id', 'like', "%{$term}%")
                    ->orwhereRelation('shipment_lines.customer_order_lines.colourways.style_versions.styles', 'customer_ref', 'like', "%{$term}%")
                    ->orwhereRelation('shipment_lines.customer_order_lines.colourways.style_versions.styles.designs', 'description', 'like', "%{$term}%");
                });
            }
        });
    }

    protected static function booted(): void
    {
        parent::booted();
        static::deleting(function($model) {
            $model->shipment_lines?->each?->delete();
            $model->shipment_files?->each?->delete();
        });

        // Total cache invalidation hooks
        static::saved(function($model) {
            $model->invalidateTotalCache();
        });

        static::updated(function($model) {
            $model->invalidateTotalCache();
        });

        static::deleted(function($model) {
            $model->invalidateTotalCache();
        });
        static::restoring(function($model) {
            $model->shipment_lines()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->shipment_files()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
        });
        // static::addGlobalScope('order', function (Builder $builder) {
        //     $builder->orderBy('name');
        // });
    }
    protected $casts = [
        "organiser"         => TitleCase::class,
        "cost"              => Currency::class,
        "additional_cost"   => Currency::class,
        'customs_submitted' => Boolean::class,
        'truck_first_collection' => 'date:Y-m-d',
        'created_at'        => 'datetime:Y-m-d',
        'updated_at'        => 'datetime:Y-m-d',
        'deleted_at'        => 'datetime:Y-m-d',
    ];
	protected $fillable = [
        'organiser',
        'transporter_id',
        'cost',
        'additional_cost',
        'customs_submitted',
        'transporter_invoice',
        'comments',
        // 'last_updated_by',
        'created_at',
        'updated_at',
        'customs_submitted_text',
        'no_cartons',
        'weight',
        'no_trucks',
        'truck_first_collection',
        'deleted_at',
    ];

    public function getTotalsAttribute()
    {
        $totals = [
            'goods_value' => 0,
            'transport_budget' => 0,
            'goods_qty' => 0,
            'cmt_total' => 0,
        ];
        
        foreach ($this->shipment_lines as $sl) {
            $order = $sl->customer_order_lines->customer_orders;
            $isWholesale = $order->order_type === 'wholesale';

            // Get price model data for this line
            $priceModel = $sl->customer_order_lines->price_model;
            
            foreach ($sl->customer_order_lines->customer_order_line_quantities as $colq) {
                // Use qty from shipment_line_sizes if available, otherwise use ordered quantity
                $shipmentLineSize = $sl->shipment_line_sizes->where('sizes_id', $colq->sizes_id)->first();
                $qty = $shipmentLineSize?->qty ?? $colq->qty;

                $totals['goods_qty'] += $qty;
                
                if ($priceModel) {
                    $totals['goods_value'] += ($isWholesale ? $priceModel['quote_base'] : $colq->price) * $qty;
                    $totals['transport_budget'] += ($isWholesale ? $priceModel['transport_budget_base'] : $colq->price) * $qty;
                    $totals['cmt_total'] += $priceModel['cmt'] * $qty;
                } else {
                    // Fallback to old method if no price model
                    $totals['goods_value'] += ($isWholesale ? $colq->quote_base : $colq->price) * $qty;
                    $totals['transport_budget'] += ($isWholesale ? $colq->transport_budget : $colq->price) * $qty;
                    if (isset($colq->cmt)) {
                        $totals['cmt_total'] += $colq->cmt * $qty;
                    }
                }
            }
        }

        return $totals;
    }

    public function getGoodsValueAttribute()
    {
        return $this->totals['goods_value'];
    }

    public function getTransportBudgetAttribute()
    {
        return $this->totals['transport_budget'];
    }

    public function getGoodsQtyAttribute()
    {
        return $this->totals['goods_qty'];
    }

    public function getCmtTotalAttribute()
    {
        return $this->totals['cmt_total'];
    }




    public function transporters()
    {
        return $this->belongsTo(Suppliers::class, 'transporter_id');
    }

    public function shipment_lines()
    {
        return $this->hasMany(ShipmentLine::class, 'shipment_id');
    }
    public function shipment_files()
    {
        return $this->hasMany(ShipmentFiles::class);
    }

    /**
     * Invalidate total cache for this shipment and related entities.
     * Dispatches background jobs to recalculate totals.
     */
    public function invalidateTotalCache(): void
    {
        try {
            $service = app(TotalCacheService::class);
            
            // Invalidate and dispatch warmup for this shipment
            $service->invalidateAndDispatchWarmup('shipment', $this->id);
            
            // Invalidate and dispatch warmup for related shipment line caches
            $shipmentLineIds = $this->shipment_lines()->pluck('id')->toArray();
            if (!empty($shipmentLineIds)) {
                $service->invalidateAndDispatchWarmupBatch('shipment_line', $shipmentLineIds);
                
                // Also invalidate and dispatch warmup for related customer order line caches
                $orderLineIds = ShipmentLine::whereIn('id', $shipmentLineIds)->pluck('customer_order_lines_id')->filter()->unique()->toArray();
                if (!empty($orderLineIds)) {
                    $service->invalidateAndDispatchWarmupBatch('customer_order_line', $orderLineIds);
                    
                    // Also invalidate and dispatch warmup for related customer order caches
                    $orderIds = CustomerOrderLines::whereIn('id', $orderLineIds)->pluck('customer_orders_id')->filter()->unique()->toArray();
                    if (!empty($orderIds)) {
                        $service->invalidateAndDispatchWarmupBatch('customer_order', $orderIds);
                    }
                }
            }
        } catch (\Exception $e) {
            \Log::error("Failed to invalidate total cache for shipment {$this->id}: " . $e->getMessage());
        }
    }
}
