<?php

namespace App\Models;

use App\Casts\Upper;
use App\Casts\Percentage;
use App\Models\CommissionOrder;
use \Bkwld\Cloner\Cloneable;
use Illuminate\Support\Facades\Cache;
use App\Models\BaseModel;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

/**
 * Season model
 * 
 * Represents seasons used in the system for organizing styles and orders.
 */
class Seasons extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use Cloneable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'description',
        'start_date',
        'end_date',
        'euro_rate',
        'usd_rate',
        'gbp_rate',
        'markup',
        'settings',
        'locked',
        'no_days_yarn_interest',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'description' => Upper::class,
        'markup' => Percentage::class,
        'euro_rate' => 'decimal:4',
        'usd_rate' => 'decimal:4',
        'gbp_rate' => 'decimal:4',
        'start_date' => 'datetime:Y-m-d',
        'end_date' => 'datetime:Y-m-d',
        'settings' => 'array',
        'locked' => 'boolean',
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::deleting(function ($model) {
            $model->yarn_orders?->each?->delete();
            $model->styles?->each?->delete();
            $model->customer_orders?->each?->delete();
        });

        static::restoring(function ($model) {
            $model->yarn_orders()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->styles()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->customer_orders()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
        });

        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('created_at', 'desc');
        });

        static::updated(function ($season) {
            ShipmentLine::where('season_id', $season->id)->update([
                'season_id' => $season->id,
            ]);
        });

        static::saved(function ($season) {
            if ($season->isDirty(['euro_rate', 'usd_rate', 'gbp_rate', 'markup'])) {
                // Only clear prices for styles that belong to this season
                Cache::tags(['prices', "season:{$season->id}"])->flush();
                
                // CRITICAL: Delete all PriceResolutions for this season since exchange rates changed
                // This ensures no stale pricing data remains
                \DB::table('price_resolutions')
                    ->where('season_id', $season->id)
                    ->delete();
                
                // Trigger warmup for all affected colourways
                try {
                    $service = app(\App\Services\PriceResolutionService::class);
                    $colourways = \App\Models\Colourways::whereHas('style_versions.styles', function($q) use ($season) {
                        $q->where('seasons_id', $season->id);
                    })->pluck('id');
                    
                    foreach ($colourways as $colourwayId) {
                        $service->warmupColourway($colourwayId);
                    }
                } catch (\Exception $e) {
                    \Log::error('Failed to warmup price resolutions after season rate change', [
                        'season_id' => $season->id,
                        'error' => $e->getMessage()
                    ]);
                }
            }
        });
    }

    /**
     * Get the yarn orders for this season.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function yarn_orders()
    {
        return $this->hasMany(YarnOrder::class);
    }

    /**
     * Get the styles for this season.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function styles()
    {
        return $this->hasMany(Styles::class);
    }

    /**
     * Get the customer orders for this season.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function customer_orders()
    {
        return $this->hasMany(CustomerOrders::class);
    }

    /**
     * Get the commission orders for this season.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function commissionOrders()
    {
        return $this->hasMany(CommissionOrder::class, 'seasons_id');
    }


    /**
     * Retrieve a setting with a given name or fall back to the default.
     * 
     * @param string $name
     * @param mixed $default
     * @return mixed
     */
    public function setting(string $name, $default = null)
    {
        if ($this->settings != null && array_key_exists($name, $this->settings)) {
            return $this->settings[$name];
        }
        return $default;
    }

    /**
     * Update one or more settings and then optionally save the model.
     * 
     * @param array $revisions
     * @param bool $save
     * @return self
     */
    public function settings(array $revisions, bool $save = true): self
    {
        if ($this->settings != null) {
            $this->settings = array_merge($this->settings, $revisions);
        } else {
            $this->settings = $revisions;
        }
        
        if ($save) {
            $this->save();
        }
        
        return $this;
    }

    /**
     * Scope a query to search for seasons.
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $value
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, string $value)
    {
        return $query->where('description', 'like', "%{$value}%");
    }
}
