<?php

namespace App\Models;

use GuzzleHttp\Client;
use App\Casts\Currency;
use App\Casts\TitleCase;
use App\Casts\Percentage;
use App\Models\BaseModel;
use OwenIt\Auditing\Audit;
use App\Helper\Conversions;
use \Bkwld\Cloner\Cloneable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Cache;
use Illuminate\Database\Eloquent\Model;
use OwenIt\Auditing\Contracts\Auditable;
use Symfony\Component\DomCrawler\Crawler;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;


class Price extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use Cloneable;

    protected static function boot()
    {
        parent::boot();

        static::saving(function ($sample) {
            if ($sample->isDirty('weight') && empty($sample->costed_weight)) {
                $sample->costed_weight = round($sample->weight * 1.10, 3); // 10% increase, round if you want
            }
        });

        static::saved(function ($price) {
            // Note: clearPriceCacheForStyleVersion is now called inside invalidatePriceResolutions
            // to ensure proper ordering (cache clear BEFORE warmup)
            self::invalidatePriceResolutions($price);
        });

        static::deleted(function ($price) {
            self::clearPriceCacheForStyleVersion($price->style_versions_id, $price->colourways_id);
            self::invalidatePriceResolutions($price);
        });
    }


    protected static function booted(): void
    {
        static::creating(function ($price) {
            foreach ([
                'yarn_trans_currency',
                'accessory_currency',
                'label_currency',
                'gmt_trans_currency',
                'embroidery_currency',
            ] as $field) {
                if (is_null($price->$field)) {
                    $price->$field = '€';
                }
            }
        });
    }

    protected static function clearPriceCacheForStyleVersion($styleVersionId, $colourwayId)
    {
        // Get season ID for targeted cache clearing
        $styleVersion = StyleVersions::find($styleVersionId);
        $seasonId = $styleVersion?->styles?->seasons_id;
        
        $cacheTags = [
            "prices",
            "styleversion:{$styleVersionId}",
        ];
        
        // Only add colourway tag if colourwayId is not null
        if ($colourwayId !== null) {
            $cacheTags[] = "colourway:{$colourwayId}";
        } else {
            // For NULL colourway prices, we need to clear cache for all colourways of this style version
            // Get all colourways for this style version and clear their caches
            $colourways = \App\Models\Colourways::where('style_versions_id', $styleVersionId)->pluck('id');
            foreach ($colourways as $cwId) {
                $cacheTags[] = "colourway:{$cwId}";
            }
        }
        
        if ($seasonId) {
            $cacheTags[] = "season:{$seasonId}";
        }
        
        Cache::tags($cacheTags)->flush();
    }

    protected static function invalidatePriceResolutions($price)
    {
        try {
            $priceResolutionService = app(\App\Services\PriceResolutionService::class);
            $totalCacheService = app(\App\Services\TotalCacheService::class);
            
            // IMPORTANT: Clear cache BEFORE invalidating price resolutions
            // This ensures the Colourways->price() method will fetch fresh prices
            static::clearPriceCacheForStyleVersion($price->style_versions_id, $price->colourways_id);
            
            // Invalidate caches synchronously (fast operations - just marking records as stale)
            $priceResolutionService->invalidateByPrice($price->id);
            $totalCacheService->invalidateByPrice($price);
            
            // If this is a general price (NULL colourway), also invalidate all resolutions for this style version
            if ($price->colourways_id === null) {
                $priceResolutionService->invalidateByStyleVersion($price->style_versions_id);
            }
            
            // Dispatch warmup synchronously for ASOS to ensure immediate updates
            // This blocks the save operation but guarantees prices are fresh
            \App\Jobs\WarmupPriceCacheJob::dispatchSync(
                $price->id,
                $price->colourways_id === null // isStyleVersionWarmup
            );
        } catch (\Exception $e) {
            \Log::error('Failed to invalidate price resolutions', [
                'price_id' => $price->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    protected $appends = [
        'yarn_cost_kg',
        'yarn_value_euro',
        'subtotal',
        'subtotal_base',
        'cmt_base',
        'customs',
        'factoring',
        'quote_base',
        'margin',
        'discount_price',
        'discount_price_base',
        'last_updated_by',
        'quote_base_raw',
    ];

    protected $cloneable_relations = [
    ];
    protected $casts = [
        'name' => TitleCase::class,
        'yarn_trans' => Currency::class,
        'yarn_interest' => Percentage::class,
        'cmt' => Currency::class,
        'gmt_trans' => Currency::class,
        'aoc' => Percentage::class,
        'yarn' => Currency::class,
        'fabric' => Currency::class,
        'accessory' => Currency::class,
        'target' => Currency::class,
        'label' => Currency::class,
        'yarn_value' => Currency::class,
        'embroidery' => Currency::class,
        'duty' => Currency::class,
        'testing' => Currency::class,
        'testing_courier' => Currency::class,
        'quote' => Currency::class,

        'quotes_sent' => 'array',
        'valid_until' => 'date:Y-m-d',
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
        'flag_issue' => 'boolean',
    ];

    protected $fillable = [
        'style_versions_id',
        'name',
        'model',
        'yarn_trans',
        'yarn_trans_currency',
        'yarn_interest',
        'cmt',
        'cmt_currency',
        'gmt_trans',
        'gmt_trans_currency',
        'notes',
        'aoc',
        'weight',
        'yarn',
        'fabric',
        'accessory',
        'created_at',
        'updated_at',
        'yarn_currency',
        'fabric_currency',
        'accessory_currency',
        'target',
        'costed_weight',
        'min_qty',
        'label',
        'label_currency',
        'yarn_value',
        'yarn_value_currency',
        'sizes_id',
        'customer_notes',
        'embroidery',
        'embroidery_currency',
        'testing',
        'testing_courier',
        'weight_comments',
        'cost_comments',
        'comments_toggle',
        'colourways_id',
        'cmt_status',
        'quote_status',
        'valid_until',
        'colour_type',
        'quote',
        'quotes_sent',
        'deleted_at',
        'discount',
        'duty',
        'phase_id',
        'yarnkg',
        'yarnkg_currency',
        'flag_issue'
    ];

    protected $attributes = [
        'id' => NULL,
        'style_versions_id' => NULL,
        'name' => NULL,
        'model' => 'standard',
        'yarn_trans' => NULL,
        'yarn_trans_currency' => NULL,
        'cmt' => NULL,
        // 'cmt_currency' => NULL,
        'gmt_trans' => NULL,
        'gmt_trans_currency' => NULL,
        'notes' => NULL,
        'aoc' => 0,
        'weight' => NULL,
        'yarn' => NULL,
        // 'fabric' => NULL,
        'accessory' => NULL,
        'created_at' => NULL,
        'updated_at' => NULL,
        'yarn_currency' => NULL,
        // 'fabric_currency' => NULL,
        'accessory_currency' => NULL,
        'target' => NULL,
        'costed_weight' => NULL,
        'min_qty' => NULL,
        'label' => NULL,
        'label_currency' => NULL,
        'yarn_value' => NULL,
        'yarn_value_currency' => NULL,
        'sizes_id' => NULL,
        'customer_notes' => NULL,
        'embroidery' => NULL,
        'embroidery_currency' => NULL,
        'testing' => NULL,
        'testing_courier' => NULL,
        'weight_comments' => NULL,
        'cost_comments' => NULL,
        'comments_toggle' => NULL,
        'colourways_id' => NULL,
        'cmt_status' => 'not_confirmed',
        'quote_status' => 'not_confirmed',
        'valid_until' => NULL,
        'colour_type' => NULL,
        'quote' => NULL,
        'quotes_sent' => NULL,
        'deleted_at' => NULL,
        'discount' => NULL,
        'duty' => 0,
        'phase_id' => NULL,
        'yarnkg' => NULL,
        'yarnkg_currency' => NULL,
        'flag_issue' => false,
    ];

    //RELATIONS
    public function style_versions()
    {
        return $this->belongsTo(StyleVersions::class);
    }
    public function sizes()
    {
        return $this->belongsTo(Sizes::class);
    }
    public function last_updated_by_user()
    {
        return $this->belongsTo(User::class, 'last_updated_by');
    }
    public function customerOrderLineQuantities()
    {
        return $this->hasMany(CustomerOrderLineQuantities::class, 'prices_id');
    }
    public function colourways()
    {
        return $this->belongsTo(Colourways::class);
    }




    /**
     * Fetch the Season.
     *
     */
    protected $cached_season;

    public function getSeasonAttribute()
    {
        if (!isset($this->cached_season)) {
            $cacheKey = "season:style_version_{$this->style_versions_id}";

            $this->cached_season = Cache::remember($cacheKey, now()->addMonths(6), function () {
                return $this->style_versions->styles->seasons ?? null;
            });
        }

        return $this->cached_season;
    }


    /**
     * Fetch the Bank of England base rate (cached for one week).
     *
     * @return float|null
     */
    public static function getBaseRateAttribute()
    {
        $cacheKey = 'boe_base_rate_value';
        return Cache::remember($cacheKey, now()->addWeek(), function () {
            $client = new Client();
            $response = $client->request('GET', 'https://www.bankofengland.co.uk/boeapps/database/Bank-Rate.asp#');
            if ($response->getStatusCode() === 200) {
                $html = $response->getBody()->getContents();
                $crawler = new Crawler($html);
                $rate = $crawler->filter('.stat-figure')->text();
                $rate = str_replace('%', '', $rate);
                $rate = (float) trim($rate);
                return $rate / 100;
            }
            return null;
        });
    }


    /*-----------------------------------------------
    | Instance (Computed) Methods
    |-----------------------------------------------*/


    protected $cached_yarn_cost_kg;

    public function getYarnCostKgAttribute()
    {
        $styleVersionId = $this->style_versions_id ?? null;

        if (!isset($this->cached_yarn_cost_kg) && $styleVersionId) {
            $cacheKey = "yarn_cost_kg:style_version_{$styleVersionId}:colour_type_{$this->colour_type}";

            return $this->cached_yarn_cost_kg = Cache::tags(['yarn_costs', 'style_version:' . $styleVersionId])
                ->remember($cacheKey, now()->addWeek(), function () {
                    if ($this->model === 'full_factored') {
                        return 0;
                    }

                $colourways = $this->style_versions->colourways;
                
                if ($colourways->isEmpty()) {
                    return 0;
                }
                
                // Filter by colour_type if specified
                // Include both exact matches AND NULL colour_type colourways as fallback
                if (!is_null($this->colour_type)) {
                    $matchingColourways = $colourways->where('colour_type', $this->colour_type);
                    // If no exact matches, fall back to NULL colour_type colourways
                    if ($matchingColourways->isEmpty()) {
                        $colourways = $colourways->whereNull('colour_type');
                    } else {
                        $colourways = $matchingColourways;
                    }
                }
                
                // Filter to only include colourways with full yarn information
                // (all yarns have prices and percentages sum to 100%)
                $colourwaysWithFullInfo = $colourways->filter(function ($colourway) {
                        // Check if colourway has yarn data
                        if ($colourway->colourway_yarns->isEmpty()) {
                            return false;
                        }
                        
                        // Check if all yarns have prices
                        $allHavePrices = $colourway->colourway_yarns->every(function ($yarn) {
                            return !is_null($yarn->price) && $yarn->price > 0;
                        });
                        
                        if (!$allHavePrices) {
                            return false;
                        }
                        
                        // Check if percentages add up to 100%
                        $totalPercentage = $colourway->colourway_yarns->sum('percentage');
                        
                        // Allow for small rounding differences (within 0.5%)
                        return abs($totalPercentage - 100) < 0.5;
                    });
                    
                    // If no colourways with full info, return 0
                    if ($colourwaysWithFullInfo->isEmpty()) {
                        return 0;
                    }
                    
                    // Calculate average yarn cost per kg from all matching colourways
                    $totalCost = 0;
                    $count = 0;
                    
                    foreach ($colourwaysWithFullInfo as $colourway) {
                        $totalCost += $colourway->yarn_cost_kg_eur;
                        $count++;
                    }
                    
                    return $count > 0 ? ($totalCost / $count) : 0;
                });
        }

        return $this->cached_yarn_cost_kg;
    }


    /**
     * Calculate the yarn value in Euro.
     *
     * @return string
     */
    public function getYarnValueEuroAttribute()
    {
        $cacheKey = "price_{$this->id}_yarn_value_euro";
        return Cache::remember($cacheKey, 1, function () {
            if ($this->model == 'standard') {
                $yarnCostKgEur = $this->yarnCostKg ?? 0;
            } elseif ($this->model == 'manual') {
                // Get the Season from the related Style via style_version.
                $season = $season = $this->season;;
                $seasonId = $season ? $season->id : $this->seasons_id;
                $rate = $this->{ currencyToRate($this->yarnkg_currency) } ?? 0;
                $yarnCostKgEur = convertCurrency($this->yarnkg_currency, '€', $this->yarn, $seasonId, $rate);
            } else {
                $yarnCostKgEur = 0;
            }
            
            // Convert yarn_trans to Euro based on its currency
            $season = $this->season;
            $seasonId = $season ? $season->id : $this->seasons_id;
            $yarnTransRate = $this->{ currencyToRate($this->yarn_trans_currency) } ?? null;
            $yarnTransEur = convertCurrency($this->yarn_trans_currency, '€', $this->yarn_trans, $seasonId, $yarnTransRate);
            
            // Calculate base yarn value
            $val = ($yarnCostKgEur + $yarnTransEur) * ((float)$this->costed_weight);
            
            // Apply yarn interest if applicable
            $yarnInterest = (float)($this->yarn_interest ?? 0);
            if ($yarnInterest > 0) {
                // Get number of days from season
                $noDays = $season ? (int)($season->no_days_yarn_interest ?? 365) : 365;
                
                // Apply AER formula: A = P * (1 + r)^(days/365)
                // Convert percentage to decimal (e.g., 5% becomes 0.05)
                $interestRate = $yarnInterest / 100;
                $val = $val * pow(1 + $interestRate, $noDays / 365);
            }
            
            // Round to 2 decimal places for consistency in calculations
            return round($val, 2);
        });
    }

        /**
     * Calculate the yarn value in Euro.
     *
     * @return string
     */
    public function getYarnValueBaseAttribute()
    {
        $cacheKey = "price_{$this->id}_yarn_value_base";
        return Cache::remember($cacheKey, 1, function () {
            // Get Season from related Style.
            $season = $season = $this->season;;
            $seasonId = $season ? $season->id : $this->seasons_id;
            // Get fact_currency from related Factory.
            $factCurrency = $this->style_versions->factories->currency ?? $this->fact_currency;
            // Optionally, you could cache euro and pound rates here.
            $euroRate = $this->{ currencyToRate('€') } ?? null;
            $poundRate = $this->{ currencyToRate('£') } ?? null;

            return number_format(convertCurrency($factCurrency, '£', $this->yarn_value, $seasonId, $this->{ currencyToRate($factCurrency) } ?? null), 2);
        });
    }

    /**
     * Calculate the cmt in GBP.
     *
     * @return string
     */
    public function getCmtBaseAttribute()
    {
        $cacheKey = "price_{$this->id}_cmt_base";
        return Cache::remember($cacheKey, 1, function () {
            // Get Season from related Style.
            $season = $season = $this->season;;
            $seasonId = $season ? $season->id : $this->seasons_id;
            // Get fact_currency from related Factory.
            $factCurrency = $this->style_versions->factories->currency ?? $this->fact_currency;
            // Optionally, you could cache euro and pound rates here.
            $euroRate = $this->{ currencyToRate('€') } ?? null;
            $poundRate = $this->{ currencyToRate('£') } ?? null;

            return number_format(convertCurrency($factCurrency, '£', $this->cmt, $seasonId, $this->{ currencyToRate($factCurrency) } ?? null), 2);
        });
    }

    /**
     * Calculate the subtotal in Euro.
     *
     * @return string
     */
    public function getSubtotalAttribute()
    {
        $cacheKey = "price_{$this->id}_subtotal";
        return Cache::remember($cacheKey, 1, function () {
            // Get Season from related Style.
            $season = $season = $this->season;;
            $seasonId = $season ? $season->id : $this->seasons_id;
            // Get fact_currency from related Factory.
            $factCurrency = $this->style_versions->factories->currency ?? $this->fact_currency;
            $factCurrencyRate = $this->{ currencyToRate($factCurrency) } ?? null;
            // Get customer currency from related Style's customer.
            $custCurrency = $this->style_versions->styles->customers->currency ?? $this->cust_currency;
            // Optionally, you could cache euro and pound rates here.
            $euroRate = $this->{ currencyToRate('€') } ?? null;
            $poundRate = $this->{ currencyToRate('£') } ?? null;

            // if(!is_numeric($this->yarnValueEuro)){
            //     dd($this->yarnValueEuro);
            // }

            $subtotalEuro = $this->yarnValueEuro +
                $this->cmt +
                $this->cmt * ($this->aoc / 100) +
                convertCurrency($this->accessory_currency, $factCurrency, $this->accessory, $seasonId, $factCurrencyRate) +
                convertCurrency($this->embroidery_currency, $factCurrency, $this->embroidery, $seasonId, $factCurrencyRate) +
                convertCurrency($this->label_currency, $factCurrency, $this->label, $seasonId, $factCurrencyRate) +
                convertCurrency($this->gmt_trans_currency, $factCurrency, $this->gmt_trans, $seasonId, $factCurrencyRate) +
                convertCurrency('£', $factCurrency, $this->testing, $seasonId, $factCurrencyRate) +
                convertCurrency('£', $factCurrency, $this->testing_courier, $seasonId, $factCurrencyRate);

            $subtotalEuro *= (($this->duty / 100) + 1);

            if ($this->customer_customs || $this->factory_customs) {
                $subtotalEuro += convertCurrency('£', $factCurrency, env('CUSTOMS'), $seasonId, $factCurrencyRate);
            }

            $subtotalEuro += $this->factoring;
            return number_format($subtotalEuro, 2, '.', '');
        });
    }

    /**
     * Calculate the subtotal converted to GBP.
     *
     * @return string
     */
    public function getSubtotalBaseAttribute()
    {
        $cacheKey = "price_{$this->id}_subtotal_base";
        return Cache::remember($cacheKey, 1, function () {
            $season = $season = $this->season;;
            $seasonId = $season ? $season->id : $this->seasons_id;
            $factCurrency = $this->style_versions->factories->currency ?? $this->fact_currency;
            return number_format(convertCurrency($factCurrency, '£', $this->subtotal, $seasonId, $this->{ currencyToRate($factCurrency) } ?? null), 2);
        });
    }

    /**
     * Calculate customs cost.
     *
     * @return string
     */
    public function getCustomsAttribute()
    {
        $cacheKey = "price_{$this->id}_customs";
        return Cache::remember($cacheKey, 1, function () {
            if ($this->customer_customs || $this->factory_customs) {
                return number_format(env('CUSTOMS'), 2);
            } else {
                return 0;
            }
        });
    }

    /**
     * Calculate the factoring cost.
     *
     * @return string
     */
    public function getCustomerTermsAttribute()
    {
        $cacheKey = "price_{$this->id}_customer_terms";
        return Cache::remember($cacheKey, 1, function () {
            return $this->style_versions->styles->customers->customer_payment_terms;
        });
    }

    /**
     * Calculate the factoring cost.
     *
     * @return string
     */
    public function getFactoringAttribute()
    {
        $cacheKey = "price_{$this->id}_factoring";
        return Cache::remember($cacheKey, 1, function () {
            $itemFee = env('FACTORING_CHARGE');
            $itemFee += ($this->quote * 1.2 * env('FACTORING_CHARGE_PERCENTAGE'));
            $aer = $this->baseRate + env('FACTORING_INTEREST');
            $days_in_year = 365;
            $total_interest_fee = 0;
            foreach ($this->customer_terms as $term) {
                if (!isset($term->days, $term->percentage)) {
                    continue;
                }
                $days = $term->days;
                $percentage = $term->percentage / 100;
                $daily_rate = pow(1 + $aer, 1 / $days_in_year) - 1;
                $interest_for_days = pow(1 + $daily_rate, $days) - 1;
                $interest_fee = ($this->quote * 1.2) * $interest_for_days * $percentage;
                $total_interest_fee += $interest_fee;
            }
            $total_factoring_cost = $itemFee + $total_interest_fee;
            return number_format($total_factoring_cost, 2);
        });
    }

    /**
     * Calculate the quote in GBP (raw numeric value for calculations).
     *
     * @return float
     */
    public function getQuoteBaseRawAttribute()
    {
        $cacheKey = "price_{$this->id}_quote_base_raw";
        return Cache::remember($cacheKey, 1, function () {
            $season = $this->season;
            $seasonId = $season ? $season->id : $this->seasons_id;
            $custCurrency = $this->style_versions->styles->customers->currency ?? $this->cust_currency;
            $factCurrency = $this->style_versions->factories->currency ?? $this->fact_currency;
            $rawValue = convertCurrency($custCurrency, '£', $this->quote, $seasonId);
            return is_numeric($rawValue) ? (float) $rawValue : 0.0;
        });
    }

    /**
     * Calculate the quote in GBP (formatted for display).
     *
     * @return string
     */
    public function getQuoteBaseAttribute()
    {
        $cacheKey = "price_{$this->id}_quote_base";
        return Cache::remember($cacheKey, 1, function () {
            $rawValue = $this->quote_base_raw;
            return number_format($rawValue, 2);
        });
    }

    /**
     * Calculate the margin percentage.
     *
     * @return string|int
     */
    public function getMarginAttribute()
    {
        $cacheKey = "price_{$this->id}_margin";
        return Cache::remember($cacheKey, 1, function () {
            if ($this->quote > 0) {
                // Calculate using raw numeric values to avoid type errors
                $discountPriceBaseRaw = 0;
                $subtotalBaseRaw = 0;
                
                // Get raw values for discount price base calculation
                $quoteBaseRaw = $this->getQuoteBaseRawAttribute();
                if ($quoteBaseRaw > 0 && isset($this->cust_discount) && $this->cust_discount > 0) {
                    $custDiscount = is_numeric($this->cust_discount) ? (float) $this->cust_discount : 0.0;
                    $discountPriceBaseRaw = $quoteBaseRaw * ((100 - $custDiscount) / 100);
                } elseif ($quoteBaseRaw > 0) {
                    $discountPriceBaseRaw = $quoteBaseRaw;
                }
                
                // Get raw value for subtotal base
                $season = $this->season;
                $seasonId = $season ? $season->id : $this->seasons_id;
                $factCurrency = $this->style_versions->factories->currency ?? $this->fact_currency;
                $subtotalBaseRaw = (float) convertCurrency($factCurrency, '£', $this->subtotal, $seasonId, $this->{ currencyToRate($factCurrency) } ?? null);
                
                if ($discountPriceBaseRaw > 0) {
                    return number_format(100 * (($discountPriceBaseRaw - $subtotalBaseRaw) / $discountPriceBaseRaw), 1);
                }
            }
            return '0.0';
        });
    }

    public function getCustDiscountAttribute(){
        if(!isset($this->style_versions->styles->customers->default_discount)){
            $this->load('style_versions.styles.customers');
        }
        $discount = $this->style_versions->styles->customers->default_discount;
        return is_numeric($discount) ? (float) $discount : 0.0;
    }

    /**
     * Calculate the discounted price.
     *
     * @return string
     */
    public function getDiscountPriceAttribute()
    {
        $cacheKey = "price_{$this->id}_discount_price";
        return Cache::remember($cacheKey, 1, function () {
            if ($this->quote && isset($this->cust_discount) && $this->cust_discount > 0) {
                // Ensure we have numeric values for calculations
                $quote = is_numeric($this->quote) ? (float) $this->quote : 0.0;
                $custDiscount = is_numeric($this->cust_discount) ? (float) $this->cust_discount : 0.0;
                
                if ($custDiscount > 0) {
                    return number_format($quote * ((100 - $custDiscount) / 100), 2);
                }
            }
            
            if ($this->quote) {
                $quote = is_numeric($this->quote) ? (float) $this->quote : 0.0;
                return number_format($quote, 2);
            }
            
            return '';
        });
    }

    /**
     * Calculate the discounted quote in GBP.
     *
     * @return string
     */
    public function getDiscountPriceBaseAttribute()
    {
        $cacheKey = "price_{$this->id}_discount_price_base";
        return Cache::remember($cacheKey, 1, function () {
            if ($this->quoteBase && isset($this->cust_discount) && $this->cust_discount > 0) {
                // Ensure we have numeric values for calculations
                $quoteBase = is_numeric($this->quoteBase) ? (float) $this->quoteBase : 0.0;
                $custDiscount = is_numeric($this->cust_discount) ? (float) $this->cust_discount : 0.0;
                
                if ($custDiscount > 0) {
                    return number_format($quoteBase * ((100 - $custDiscount) / 100), 2);
                }
            }
            
            if ($this->quoteBase) {
                $quoteBase = is_numeric($this->quoteBase) ? (float) $this->quoteBase : 0.0;
                return number_format($quoteBase, 2);
            }
            
            return '';
        });
    }

    /**
     * Get last updater.
     *
     * @return string
     */
    public function getLastUpdatedByAttribute()
    {
        return "";
        // return $this->audits()->with('user:id,name')->latest()->first()?->user?->name;
    }

    /**
     * Get old prices.
     *
     * @return string
     */
    public function getHistoricAttribute()
    {
        // Ensure the necessary relationships are available.
        // Using the current Price model's relationships:
        // - Design: via $this->style_versions->styles->designs_id
        // - Season: via $this->style_versions->styles->seasons->id
        // - Sizes: current price's sizes_id
        if (
            !isset($this->style_versions) ||
            !isset($this->style_versions->styles) ||
            !isset($this->style_versions->styles->seasons)
        ) {
            return json_encode([]);
        }

        $designsId = $this->style_versions->styles->designs_id;
        $currentSeasonId = $this->style_versions->styles->seasons->id;
        $sizesId = $this->sizes_id; // or $this->sizes_id ?? null

        // Query previous prices matching the criteria.
        $historic = \DB::table('prices as prev_p')
            ->join('style_versions as prev_sv', 'prev_p.style_versions_id', '=', 'prev_sv.id')
            ->join('styles as prev_s', 'prev_sv.styles_id', '=', 'prev_s.id')
            ->join('seasons as prev_season', 'prev_season.id', '=', 'prev_s.seasons_id')
            ->where('prev_s.designs_id', $designsId)
            ->where('prev_s.seasons_id', '<', $currentSeasonId)
            ->when($sizesId, function($query) use ($sizesId) {
                return $query->where('prev_p.sizes_id', $sizesId);
            })
            ->where('prev_p.quote_status', 'confirmed')
            ->orderBy('prev_season.id', 'desc')
            ->limit(3)
            ->get([
                'prev_season.description as season',
                'prev_p.id as price_id',
                'prev_p.cmt',
                'prev_p.quote'
            ]);

        return $historic->toJson();
    }

}
