<?php

namespace App\Models;

use App\Models\Gauge;
use DateTimeInterface;
use App\Casts\TitleCase;
use Illuminate\Support\Str;
use App\Models\BaseModel;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Design extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::deleting(function ($model) {
            $model->styles?->each?->delete();
        });

        static::restoring(function ($model) {
            $model->styles()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
        });

        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('id', 'desc');
        });

        static::updated(function ($design) {
            ShipmentLine::where('design_id', $design->id)->update([
                'design_id' => $design->id,
            ]);
        });
    }

    protected $casts = [
        'description' => TitleCase::class,
        'yarn' => TitleCase::class,
        'issued' => 'datetime:Y-m-d',
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];

	protected $fillable = [
        'description',
        'customers_id',
        'issued',
        'gauge_id',
        'yarn',
        'factory_id',
        'created_at',
        'updated_at',
        // 'last_updated_by',
        // 'customer_ref',
        'deleted_at',

    ];

    protected function Description(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => Str::title($value),
        );
    }




	public function customers()
    {
        return $this->belongsTo(Customer::class);
    }
    public function factories()
    {
        return $this->belongsTo(Suppliers::class, 'factory_id');
    }
	public function gauges(){
		return $this->belongsTo(Gauge::class, 'gauge_id');
	}

	public function styles()
    {
        return $this->hasMany(Styles::class, 'designs_id');
    }


    public function scopeSearch($query, $value)
    {
        // Split the search string into multiple terms
        $terms = preg_split('/\s+/', trim($value));

        $query->where(function($query) use ($terms) {
            foreach ($terms as $term) {
                // For each term, require at least one matching condition
                $query->where(function($q) use ($term) {
                    $q->where('description', 'like', "%{$term}%")
                    ->orWhere('yarn', 'like', "%{$term}%")
                    ->orwhere('id', 'like', "%{$term}%")
                    ->orWhereRelation('gauges', 'gauge', 'like', "%{$term}%")
                    ->orWhereRelation('customers', 'name', 'like', "%{$term}%")
                    ->orWhereRelation('factories', 'name', 'like', "%{$term}%");
                });
            }
        });
    }
}
