<?php

namespace App\Models;

use App\Casts\Upper;
use App\Models\BaseModel;
use Bkwld\Cloner\Cloneable;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

/**
 * Department model
 * 
 * Represents departments within the organization.
 */
class Departments extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use Cloneable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'description',
        'zoho_name',
        'type',
        'hidden',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'description' => Upper::class,
        'hidden' => 'boolean',
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::deleting(function ($model) {
            $model->styles?->each?->delete();
            $model->customer_orders?->each?->delete();
        });

        static::restoring(function ($model) {
            $model->styles()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->customer_orders()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
        });

        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('created_at', 'desc');
        });
    }

    /**
     * Get the styles for this department.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function styles()
    {
        return $this->hasMany(Styles::class);
    }

    /**
     * Get the customer orders for this department.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function customer_orders()
    {
        return $this->hasMany(CustomerOrders::class);
    }

    /**
     * Get the user departments for this department.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function user_departments()
    {
        return $this->hasMany(UserDepartment::class, 'users_id');
    }

    /**
     * Scope a query to search for departments.
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $value
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, string $value)
    {
        return $query->where('description', 'like', "%{$value}%")
            ->orWhere('type', 'like', "%{$value}%");
    }
}
