<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use OwenIt\Auditing\Contracts\Auditable;

class CommissionOrderLine extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;

    protected $table = 'commission_order_lines';

    protected $casts = [
        'cancelled' => 'boolean',
        'ai_match_confidence' => 'decimal:2',
        'commission_percent' => 'decimal:2',
        'discount_percent' => 'decimal:2',
        'imported_raw_data' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    protected $fillable = [
        'commission_orders_id',
        'colourways_id',
        'imported_style_ref',
        'imported_style_description',
        'imported_colour',
        'match_status',
        'commission_percent',
        'discount_percent',
        'ai_suggested_colourway_id',
        'ai_match_confidence',
        'imported_raw_data',
        'cancelled',
    ];

    // ==================
    // Relationships
    // ==================

    public function order()
    {
        return $this->belongsTo(CommissionOrder::class, 'commission_orders_id');
    }

    public function colourway()
    {
        return $this->belongsTo(Colourways::class, 'colourways_id');
    }

    public function suggestedColourway()
    {
        return $this->belongsTo(Colourways::class, 'ai_suggested_colourway_id');
    }

    public function quantities()
    {
        return $this->hasMany(CommissionOrderLineQuantity::class, 'commission_order_lines_id');
    }

    public function drops()
    {
        return $this->hasMany(CommissionDrop::class, 'commission_order_lines_id');
    }

    // ==================
    // Scopes
    // ==================

    public function scopeUnmatched($query)
    {
        return $query->where('match_status', 'unmatched');
    }

    public function scopeMatched($query)
    {
        return $query->whereIn('match_status', ['ai_matched', 'pd_matched', 'confirmed']);
    }

    public function scopeConfirmed($query)
    {
        return $query->where('match_status', 'confirmed');
    }

    public function scopeActive($query)
    {
        return $query->where('cancelled', false);
    }

    // ==================
    // Accessors
    // ==================

    public function getTotalQtyAttribute()
    {
        return $this->quantities->sum('qty');
    }

    public function getTotalValueAttribute()
    {
        return $this->quantities->sum(function ($qty) {
            return $qty->qty * ($qty->price ?? 0);
        });
    }

    public function getIsMatchedAttribute(): bool
    {
        return $this->colourways_id !== null;
    }

    // ==================
    // Helper methods
    // ==================

    /**
     * Get style info - either from linked colourway or imported data
     */
    public function getStyleRef(): ?string
    {
        if ($this->colourway) {
            return $this->colourway->style_versions->styles->customer_ref ?? null;
        }
        return $this->imported_style_ref;
    }

    public function getStyleDescription(): ?string
    {
        if ($this->colourway) {
            return $this->colourway->style_versions->styles->designs->description ?? null;
        }
        return $this->imported_style_description;
    }

    public function getColourName(): ?string
    {
        if ($this->colourway) {
            return $this->colourway->name;
        }
        return $this->imported_colour;
    }

    // ==================
    // Commission/Discount Accessors
    // ==================

    /**
     * Get effective commission percent (default to 0 if not set)
     */
    public function getEffectiveCommissionAttribute(): float
    {
        return (float) ($this->commission_percent ?? 0);
    }

    /**
     * Get effective discount percent (default to 0 if not set)
     */
    public function getEffectiveDiscountAttribute(): float
    {
        return (float) ($this->discount_percent ?? 0);
    }
}













