<?php

namespace App\Models;

use App\Casts\Upper;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use OwenIt\Auditing\Contracts\Auditable;

class CommissionOrder extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;

    protected $table = 'commission_orders';

    protected $casts = [
        'customer_po' => Upper::class,
        'order_date' => 'date:Y-m-d',
        'cancelled' => 'boolean',
        'pd_email_sent_at' => 'datetime',
        'pd_reviewed_at' => 'datetime',
        'confirmed_at' => 'datetime',
        'imported_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    protected $fillable = [
        'commission_import_batch_id',
        'customers_id',
        'seasons_id',
        'departments_id',
        'status',
        'customer_po',
        'order_date',
        'source_file',
        'source_file_path',
        'imported_by',
        'imported_at',
        'pd_email_sent_at',
        'forwarded_to_user_id',
        'forward_notes',
        'pd_reviewed_by',
        'pd_reviewed_at',
        'pd_notes',
        'confirmed_by',
        'confirmed_at',
        'merch_notes',
        'cancelled',
    ];

    // ==================
    // Relationships
    // ==================

    public function customer()
    {
        return $this->belongsTo(Customer::class, 'customers_id');
    }

    public function season()
    {
        return $this->belongsTo(Seasons::class, 'seasons_id');
    }

    public function department()
    {
        return $this->belongsTo(Departments::class, 'departments_id');
    }

    public function importedBy()
    {
        return $this->belongsTo(User::class, 'imported_by');
    }

    public function forwardedTo()
    {
        return $this->belongsTo(User::class, 'forwarded_to_user_id');
    }

    public function pdReviewedBy()
    {
        return $this->belongsTo(User::class, 'pd_reviewed_by');
    }

    public function confirmedBy()
    {
        return $this->belongsTo(User::class, 'confirmed_by');
    }

    public function lines()
    {
        return $this->hasMany(CommissionOrderLine::class, 'commission_orders_id');
    }

    public function importBatch()
    {
        return $this->belongsTo(CommissionImportBatch::class, 'commission_import_batch_id');
    }

    // ==================
    // Scopes
    // ==================

    public function scopeSkeleton($query)
    {
        return $query->where('status', 'skeleton');
    }

    public function scopePendingPd($query)
    {
        return $query->where('status', 'pending_pd');
    }

    public function scopePendingMerch($query)
    {
        return $query->where('status', 'pending_merch');
    }

    public function scopeConfirmed($query)
    {
        return $query->where('status', 'confirmed');
    }

    public function scopeActive($query)
    {
        return $query->where('cancelled', false);
    }

    // ==================
    // Accessors
    // ==================

    public function getTotalQtyAttribute()
    {
        return $this->lines->sum(function ($line) {
            return $line->quantities->sum('qty');
        });
    }

    public function getTotalValueAttribute()
    {
        return $this->lines->sum(function ($line) {
            return $line->quantities->sum(function ($qty) {
                return $qty->qty * ($qty->price ?? 0);
            });
        });
    }

    // ==================
    // Status helpers
    // ==================

    public function isSkeleton(): bool
    {
        return $this->status === 'skeleton';
    }

    public function isPendingPd(): bool
    {
        return $this->status === 'pending_pd';
    }

    public function isPendingMerch(): bool
    {
        return $this->status === 'pending_merch';
    }

    public function isConfirmed(): bool
    {
        return $this->status === 'confirmed';
    }

    public function canBeEdited(): bool
    {
        return !$this->cancelled && in_array($this->status, ['skeleton', 'pending_pd', 'pending_merch']);
    }

    // ==================
    // File attachments
    // ==================

    /**
     * Get array of attached source file paths
     */
    public function getSourceFilesAttribute(): array
    {
        if (empty($this->source_file_path)) {
            return [];
        }

        // Handle JSON array or single path
        $decoded = json_decode($this->source_file_path, true);
        if (is_array($decoded)) {
            return $decoded;
        }

        // Legacy single path
        return [$this->source_file_path];
    }

    /**
     * Check if order has attached files
     */
    public function hasSourceFiles(): bool
    {
        return !empty($this->source_files);
    }
}

