<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use OwenIt\Auditing\Contracts\Auditable;

class CommissionInvoiceGroup extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;

    protected $table = 'commission_invoice_groups';

    protected $casts = [
        'imported_raw_data' => 'array',
        'invoice_date' => 'date:Y-m-d',
        'invoice_uploaded_at' => 'datetime',
        'sent_to_zoho_at' => 'datetime',
        'total_value' => 'decimal:2',
        'total_qty' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    protected $fillable = [
        'customers_id',
        'reference',
        'status',
        'invoice_file_name',
        'invoice_file_path',
        'invoice_uploaded_at',
        'invoice_uploaded_by',
        'imported_raw_data',
        'invoice_date',
        'supplier_invoice_ref',
        'zoho_invoice_id',
        'sent_to_zoho_at',
        'total_value',
        'total_qty',
        'notes',
    ];

    // ==================
    // Relationships
    // ==================

    public function customer()
    {
        return $this->belongsTo(Customer::class, 'customers_id');
    }

    public function uploadedBy()
    {
        return $this->belongsTo(User::class, 'invoice_uploaded_by');
    }

    public function drops()
    {
        return $this->hasMany(CommissionDrop::class, 'commission_invoice_groups_id');
    }

    // ==================
    // Scopes
    // ==================

    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    public function scopeReady($query)
    {
        return $query->where('status', 'ready');
    }

    public function scopeSentToZoho($query)
    {
        return $query->where('status', 'sent_to_zoho');
    }

    public function scopeComplete($query)
    {
        return $query->where('status', 'complete');
    }

    // ==================
    // Status helpers
    // ==================

    public function isDraft(): bool
    {
        return $this->status === 'draft';
    }

    public function isReady(): bool
    {
        return $this->status === 'ready';
    }

    public function isSentToZoho(): bool
    {
        return $this->status === 'sent_to_zoho';
    }

    public function isComplete(): bool
    {
        return $this->status === 'complete';
    }

    // ==================
    // Methods
    // ==================

    /**
     * Recalculate and cache totals from drops
     */
    public function recalculateTotals(): void
    {
        $this->total_qty = $this->drops->sum(function ($drop) {
            return $drop->sizes->sum('shipped_qty') ?: $drop->sizes->sum('qty');
        });

        $this->total_value = $this->drops->sum(function ($drop) {
            return $drop->line->quantities->sum(function ($qty) use ($drop) {
                $dropQty = $drop->sizes->where('sizes_id', $qty->sizes_id)->first();
                $shippedQty = $dropQty ? ($dropQty->shipped_qty ?: $dropQty->qty) : 0;
                return $shippedQty * ($qty->price ?? 0);
            });
        });

        $this->save();
    }
}

























