<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Storage;

class CommissionImportBatchFile extends Model
{
    protected $fillable = [
        'batch_id',
        'original_filename',
        'stored_filename',
        'file_path',
        'status',
        'extracted_data',
        'matched_lines',
        'verification_issues',
        'metadata',
        'error_message',
        'skip_reason',
        'total_orders',
        'total_lines',
        'total_quantity',
        'extracted_at',
        'reviewed_at',
        'imported_at',
    ];

    protected $casts = [
        'extracted_data' => 'array',
        'matched_lines' => 'array',
        'verification_issues' => 'array',
        'metadata' => 'array',
        'extracted_at' => 'datetime',
        'reviewed_at' => 'datetime',
        'imported_at' => 'datetime',
    ];

    public function batch(): BelongsTo
    {
        return $this->belongsTo(CommissionImportBatch::class, 'batch_id');
    }

    public function getFullPathAttribute(): string
    {
        return Storage::path($this->file_path);
    }

    public function hasExtractedData(): bool
    {
        return !empty($this->extracted_data) && is_array($this->extracted_data);
    }

    public function isReadyForReview(): bool
    {
        return $this->status === 'extracted' && $this->hasExtractedData();
    }

    public function isImported(): bool
    {
        return $this->status === 'imported';
    }

    public function hasFailed(): bool
    {
        return $this->status === 'failed';
    }

    public function isSkipped(): bool
    {
        return $this->status === 'skipped';
    }

    public function markAsExtracted(array $data, array $metadata): void
    {
        $this->update([
            'status' => 'extracted',
            'extracted_data' => $data,
            'metadata' => $metadata,
            'total_orders' => count($data['orders'] ?? []),
            'total_lines' => array_sum(array_map(fn($order) => count($order['lines'] ?? []), $data['orders'] ?? [])),
            'total_quantity' => $this->calculateTotalQuantity($data),
            'extracted_at' => now(),
        ]);
    }

    public function markAsFailed(string $errorMessage): void
    {
        $this->update([
            'status' => 'failed',
            'error_message' => $errorMessage,
        ]);
    }

    public function markAsImported(): void
    {
        $this->update([
            'status' => 'imported',
            'imported_at' => now(),
        ]);
    }

    public function markAsSkipped(string $reason): void
    {
        $this->update([
            'status' => 'skipped',
            'skip_reason' => $reason,
        ]);
    }

    protected function calculateTotalQuantity(array $data): int
    {
        $total = 0;
        foreach ($data['orders'] ?? [] as $order) {
            foreach ($order['lines'] ?? [] as $line) {
                foreach ($line['quantities'] ?? [] as $qty) {
                    $total += (int)($qty['qty'] ?? 0);
                }
            }
        }
        return $total;
    }
}
