<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class CommissionImportBatch extends Model
{
    protected $fillable = [
        'batch_name',
        'customer_name',
        'customer_id',
        'season_id',
        'department_id',
        'user_id',
        'status',
        'workflow_status',
        'total_files',
        'processed_files',
        'successful_files',
        'failed_files',
        'started_at',
        'completed_at',
        // New workflow fields
        'commission_order_id',
        'forwarded_to_user_id',
        'forwarded_at',
        'pd_approved_by',
        'pd_approved_at',
        'pd_notes',
        'merch_approved_by',
        'merch_approved_at',
        'merch_notes',
        // Multi-agent results
        'extraction_result',
        'matching_result',
        'verification_result',
    ];

    protected $casts = [
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
        'forwarded_at' => 'datetime',
        'pd_approved_at' => 'datetime',
        'merch_approved_at' => 'datetime',
        'extraction_result' => 'array',
        'matching_result' => 'array',
        'verification_result' => 'array',
    ];

    // ==================
    // Relationships
    // ==================

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    public function season(): BelongsTo
    {
        return $this->belongsTo(Seasons::class);
    }

    public function department(): BelongsTo
    {
        return $this->belongsTo(Departments::class);
    }

    public function files(): HasMany
    {
        return $this->hasMany(CommissionImportBatchFile::class, 'batch_id');
    }

    public function commissionOrder(): BelongsTo
    {
        return $this->belongsTo(CommissionOrder::class, 'commission_order_id');
    }

    /**
     * Get all orders created from this batch (one per PO)
     */
    public function commissionOrders(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(CommissionOrder::class, 'commission_import_batch_id');
    }

    public function forwardedTo(): BelongsTo
    {
        return $this->belongsTo(User::class, 'forwarded_to_user_id');
    }

    public function pdApprovedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'pd_approved_by');
    }

    public function merchApprovedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'merch_approved_by');
    }

    // ==================
    // Workflow Status Helpers
    // ==================

    public function isPending(): bool
    {
        return $this->workflow_status === 'pending';
    }

    public function isExtracted(): bool
    {
        return in_array($this->workflow_status, ['extracted', 'matched', 'verified', 'forwarded_pd', 'pd_approved', 'merch_approved', 'completed']);
    }

    public function isMatched(): bool
    {
        return in_array($this->workflow_status, ['matched', 'verified', 'forwarded_pd', 'pd_approved', 'merch_approved', 'completed']);
    }

    public function isVerified(): bool
    {
        return in_array($this->workflow_status, ['verified', 'forwarded_pd', 'pd_approved', 'merch_approved', 'completed']);
    }

    public function isForwardedToPd(): bool
    {
        return in_array($this->workflow_status, ['forwarded_pd', 'pd_approved', 'merch_approved', 'completed']);
    }

    public function isPdApproved(): bool
    {
        return in_array($this->workflow_status, ['pd_approved', 'merch_approved', 'completed']);
    }

    public function isMerchApproved(): bool
    {
        return in_array($this->workflow_status, ['merch_approved', 'completed']);
    }

    public function isFullyApproved(): bool
    {
        return $this->workflow_status === 'completed' && $this->pd_approved_at && $this->merch_approved_at;
    }

    // ==================
    // Statistics
    // ==================

    public function getActualProcessedFilesAttribute(): int
    {
        return $this->files()
            ->whereIn('status', ['extracted', 'imported', 'failed', 'skipped'])
            ->count();
    }

    public function getActualFailedFilesAttribute(): int
    {
        return $this->files()
            ->where('status', 'failed')
            ->count();
    }

    public function getSkippedFilesAttribute(): int
    {
        return $this->files()
            ->where('status', 'skipped')
            ->count();
    }

    public function getProgressPercentageAttribute(): int
    {
        if ($this->total_files === 0) {
            return 0;
        }
        
        $actualProcessed = $this->actual_processed_files;
        return (int) round(($actualProcessed / $this->total_files) * 100);
    }

    public function getMatchedLinesCountAttribute(): int
    {
        $count = 0;
        foreach ($this->files as $file) {
            if ($file->matched_lines) {
                $count += collect($file->matched_lines)
                    ->where('match_status', '!=', 'unmatched')
                    ->count();
            }
        }
        return $count;
    }

    public function getUnmatchedLinesCountAttribute(): int
    {
        $count = 0;
        foreach ($this->files as $file) {
            if ($file->matched_lines) {
                $count += collect($file->matched_lines)
                    ->where('match_status', 'unmatched')
                    ->count();
            }
        }
        return $count;
    }

    public function getTotalLinesCountAttribute(): int
    {
        return $this->matched_lines_count + $this->unmatched_lines_count;
    }

    public function getMatchPercentageAttribute(): int
    {
        $total = $this->total_lines_count;
        if ($total === 0) return 0;
        return (int) round(($this->matched_lines_count / $total) * 100);
    }

    public function isComplete(): bool
    {
        return $this->actual_processed_files >= $this->total_files;
    }

    public function markAsCompleted(): void
    {
        $this->update([
            'status' => 'completed',
            'completed_at' => now(),
        ]);
    }

    // ==================
    // Workflow Actions
    // ==================

    public function forwardToPd(int $pdUserId, ?string $notes = null): void
    {
        $this->update([
            'workflow_status' => 'forwarded_pd',
            'forwarded_to_user_id' => $pdUserId,
            'forwarded_at' => now(),
        ]);

        // Update the commission order status too
        if ($this->commissionOrder) {
            $this->commissionOrder->update([
                'status' => 'pending_pd',
                'pd_email_sent_at' => now(),
            ]);
        }
    }

    public function approvePd(?string $notes = null): void
    {
        $this->update([
            'workflow_status' => 'pd_approved',
            'pd_approved_by' => auth()->id(),
            'pd_approved_at' => now(),
            'pd_notes' => $notes,
        ]);

        // Update the commission order status
        if ($this->commissionOrder) {
            $this->commissionOrder->update([
                'status' => 'pending_merch',
                'pd_reviewed_by' => auth()->id(),
                'pd_reviewed_at' => now(),
                'pd_notes' => $notes,
            ]);
        }
    }

    public function approveMerch(?string $notes = null): void
    {
        $this->update([
            'workflow_status' => 'completed',
            'merch_approved_by' => auth()->id(),
            'merch_approved_at' => now(),
            'merch_notes' => $notes,
        ]);

        // Update the commission order to confirmed
        if ($this->commissionOrder) {
            $this->commissionOrder->update([
                'status' => 'confirmed',
                'confirmed_by' => auth()->id(),
                'confirmed_at' => now(),
                'merch_notes' => $notes,
            ]);
        }
    }
}
