<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use OwenIt\Auditing\Contracts\Auditable;

class CommissionDrop extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;

    protected $table = 'commission_drops';

    protected $casts = [
        'exfty' => 'date:Y-m-d',
        'shipped_date' => 'date:Y-m-d',
        'pl_uploaded_at' => 'datetime',
        'shipped' => 'boolean',
        'invoiced' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    protected $fillable = [
        'commission_order_lines_id',
        'commission_invoice_groups_id',
        'exfty',
        'shipped_date',
        'pl_uploaded_at',
        'shipped',
        'invoiced',
        'notes',
        'fn_notes',
        'factory_invoice',
        'rt_invoice',
    ];

    // ==================
    // Relationships
    // ==================

    public function line()
    {
        return $this->belongsTo(CommissionOrderLine::class, 'commission_order_lines_id');
    }

    public function invoiceGroup()
    {
        return $this->belongsTo(CommissionInvoiceGroup::class, 'commission_invoice_groups_id');
    }

    public function sizes()
    {
        return $this->hasMany(CommissionDropSize::class, 'commission_drops_id');
    }

    // ==================
    // Scopes
    // ==================

    public function scopeShipped($query)
    {
        return $query->where('shipped', true);
    }

    public function scopeUnshipped($query)
    {
        return $query->where('shipped', false);
    }

    public function scopeInvoiced($query)
    {
        return $query->where('invoiced', true);
    }

    public function scopeUninvoiced($query)
    {
        return $query->where('invoiced', false);
    }

    // ==================
    // Accessors
    // ==================

    public function getTotalQtyAttribute()
    {
        return $this->sizes->sum('qty');
    }

    public function getTotalShippedQtyAttribute()
    {
        return $this->sizes->sum('shipped_qty') ?: 0;
    }

    public function getTotalValueAttribute()
    {
        $lineQuantities = $this->line->quantities->keyBy('sizes_id');
        
        return $this->sizes->sum(function ($size) use ($lineQuantities) {
            $qty = $size->shipped_qty ?: $size->qty;
            $price = $lineQuantities->get($size->sizes_id)?->price ?? 0;
            return $qty * $price;
        });
    }

    /**
     * Get the order through the line relationship
     */
    public function getOrderAttribute()
    {
        return $this->line?->order;
    }

    /**
     * Get the colourway through the line relationship
     */
    public function getColourwayAttribute()
    {
        return $this->line?->colourway;
    }
}

























