<?php

namespace App\Models;

use App\Models\BaseModel;
use Illuminate\Database\Eloquent\Factories\HasFactory;

/**
 * Cashflow Line model
 * 
 * Represents individual lines within a cashflow.
 */
class CashflowLine extends BaseModel
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'cmt',
        'cmt_terms',
        'cmt_currency',
        'total_cmt_base',
        'yarn',
        'yarn_terms',
        'yarn_currency',
        'total_yarn_base',
        'sale',
        'sale_terms',
        'sale_currency',
        'total_quote_base',
        'vat',
        'vat_base',
        'customer_po',
        'customer_ref',
        'description',
        'colour',
        'incoterms',
        'exfty',
        'cashflows_id',
        'shipment_lines_id',
        'seasons_id',
        'suppliers_id',
        'customers_id',
        'designs_id',
        'factored',
        'qty',
        'is_complete',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'cmt' => 'decimal:2',
        'total_cmt_base' => 'decimal:2',
        'yarn' => 'decimal:2',
        'total_yarn_base' => 'decimal:2',
        'sale' => 'decimal:2',
        'total_quote_base' => 'decimal:2',
        'vat' => 'decimal:2',
        'vat_base' => 'decimal:2',
        'qty' => 'integer',
        'factored' => 'boolean',
        'is_complete' => 'boolean',
    ];

    /**
     * Get the cashflow for this cashflow line.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function cashflow()
    {
        return $this->belongsTo(Cashflow::class);
    }

    /**
     * Get the shipment line for this cashflow line.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function shipmentLine()
    {
        return $this->belongsTo(ShipmentLine::class);
    }

    /**
     * Get the season for this cashflow line.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function season()
    {
        return $this->belongsTo(Seasons::class);
    }

    /**
     * Get the supplier for this cashflow line.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function suppliers()
    {
        return $this->belongsTo(Suppliers::class);
    }

    /**
     * Get the customer for this cashflow line.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function customers()
    {
        return $this->belongsTo(Customer::class);
    }

    /**
     * Get the design for this cashflow line.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function design()
    {
        return $this->belongsTo(Design::class);
    }
}
