<?php

namespace App\Livewire\Forms;

use Livewire\Form;
use App\Models\Design;
use App\Models\YarnOrder;
use App\Models\YarnOrderLine;
use Livewire\Attributes\Rule;
use App\Models\ColourwayYarns;
use App\Models\YarnOrderFiles;
use App\Models\CustomerOrderLines;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use App\Models\YarnOrderLineStyles;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;

class YarnOrderEditForm extends Form
{
	public ?YarnOrder $order;

	public $orderEdit = [], $uploadFiles = [];

	public $deleteLines = [];

    public $ship_date;
    public $actual_ship_date;
    public $delivery_date;

	public function rules()
    {
        return [
            'ship_date' => 'nullable|date',
            'actual_ship_date' => 'nullable|date',
            'delivery_date' => 'nullable|date',
			'orderEdit.suppliers_id'   => 'required|numeric|min:1',
			'orderEdit.invoice_to_id' => 'required|numeric|min:1',
			'orderEdit.deliver_to_id' => 'required|numeric|min:1',
			'orderEdit.seasons_id'    => 'nullable|numeric|min:1',
			'orderEdit.date'          => 'required|date',
			'orderEdit.customers_id'  => 'required|numeric|min:1',
			//FILES??
			'orderEdit.yarn_order_lines.*.yarn_id' => 'required|numeric|min:1',
			'orderEdit.yarn_order_lines.*.yarn_colours_id' => 'required|numeric|min:1',
			'orderEdit.yarn_order_lines.*.qty' => 'required|numeric|min:0',
			'orderEdit.yarn_order_lines.*.price' => 'required|numeric|min:0',
			'orderEdit.yarn_order_lines.*.ship_date' => 'nullable|date',
			'orderEdit.yarn_order_lines.*.actual_ship_date' => 'nullable|date',
			'orderEdit.yarn_order_lines.*.delivery_date' => 'nullable|date',
			'orderEdit.yarn_order_lines.*.yarn_order_line_styles.*.customer_orders_id' => 'required|numeric|min:1',
			'orderEdit.yarn_order_lines.*.yarn_order_line_styles.*.customer_order_lines_id' => 'required|numeric|min:1',
        ];
    }


	public function clear(){
		$this->reset();
		$this->gauge_id = "";
		$this->factory_id = "";
		$this->customers_id = "";
	}

	public function set(YarnOrder $order)
	{
		$this->order = $order;
		$this->orderEdit = $order->toArray();
		foreach($this->orderEdit['yarn_order_lines'] ?? [] as $l=>$line){
			$this->orderEdit['yarn_order_lines'][$l]['yarn_id'] = $line['yarn_colours']['yarn_id'];
			foreach($line['yarn_order_line_styles'] ?? [] as $s=>$style){
				$this->orderEdit['yarn_order_lines'][$l]['yarn_order_line_styles'][$s]['customer_orders_id'] = $style['customer_order_lines']['customer_orders_id'] ?? "";
			}
		}
	}

	public function new(){
		$this->orderEdit = [
			'suppliers_id' => NULL,
			'invoice_to_id' => NULL,
			'deliver_to_id' => NULL,
			'seasons_id' => NULL,
			'date' => NULL,
			'customers_id' => NULL,
			'yarn_order_lines' => []
		];
	}

	public function update()
	{
		session()->flash('message', 'Validation Error');
		session()->flash('alert-class', 'alert-warning');
		$this->validate();
		session()->flash('message', 'Error Saving');
		session()->flash('alert-class', 'alert-danger');

		if(empty($this->orderEdit['id'])){
			$order = new YarnOrder;
		}
		else{
			$order = $this->order;
		}

		$order->id            = empty($this->orderEdit['id']) ? NULL : $this->orderEdit['id'];
		$order->suppliers_id  = empty($this->orderEdit['suppliers_id']) ? '' : $this->orderEdit['suppliers_id'];
		$order->date          = empty($this->orderEdit['date']) ? '' : $this->orderEdit['date'];
		$order->comments      = empty($this->orderEdit['comments']) ? '' : $this->orderEdit['comments'];
		$order->invoice_to_id = empty($this->orderEdit['invoice_to_id']) ? '' : $this->orderEdit['invoice_to_id'];
		$order->deliver_to_id = empty($this->orderEdit['deliver_to_id']) ? '' : $this->orderEdit['deliver_to_id'];
		$order->customers_id  = empty($this->orderEdit['customers_id']) ? '' : $this->orderEdit['customers_id'];
		$order->seasons_id    = empty($this->orderEdit['seasons_id']) ? NULL : $this->orderEdit['seasons_id'];

		$order->save();

		if($this->uploadFiles != []){
			foreach($this->uploadFiles as $file){
				$path = NULL;
				$entry = new YarnOrderFiles();
				$path = Storage::disk('public')->put('yarnOrderFiles', $file);
				if($path != NULL){
					$entry->yarn_order_id = $order->id;
					$entry->filename = $file->getClientOriginalName();
					$entry->file = $path;
					$entry->save();
				}
			}
		}

		foreach($this->deleteLines as $line){
			YarnOrderLine::destroy($line);
		}

		foreach($this->orderEdit['yarn_order_lines'] ?? [] as $l=>$line){
			if(empty($line['id']))
				$newLine = new YarnOrderLine;
			else
			$newLine                   = $this->order->yarn_order_lines[$l];
			$newLine->yarn_order_id    = $order->id;
			$newLine->yarn_colours_id  = empty($line['yarn_colours_id']) ? '' : $line['yarn_colours_id'];
			$newLine->qty              = empty($line['qty']) ? '' : $line['qty'];
			$newLine->price            = empty($line['price']) ? '' : $line['price'];
            $newLine->ship_date        = getFirstNonEmpty($this->ship_date, $line['ship_date']);
            $newLine->actual_ship_date = getFirstNonEmpty($this->actual_ship_date, $line['actual_ship_date']);
            $newLine->delivery_date    = getFirstNonEmpty($this->delivery_date, $line['delivery_date']);
			$newLine->save();

			$styles = [];
			foreach($line['yarn_order_line_styles'] ?? [] as $s=>$style){
				$newStyle                          = new YarnOrderLineStyles;
				$newStyle->customer_order_lines_id = $style['customer_order_lines_id'];
				$styles[]                          = $newStyle;
			}
            YarnOrderLineStyles::where('yarn_order_lines_id', $newLine->id)->delete();
            $newLine->yarn_order_line_styles()->saveMany($styles);
		}

		session()->flash('message', 'Saved!');
		session()->flash('alert-class', 'alert-success');
	}

    /**
     * Copy the price on every draft YarnOrderLine into the related
     * colourway_yarns row(s).
     *
     *  – Called by the Livewire component: $this->form->syncColourPrices()
     *  – Uses the ColourwayYarn model if it exists; otherwise falls back
     *    to the query-builder so you still get a single UPDATE statement.
     */
    public function syncColourPrices(): void
    {
        if (empty($this->orderEdit['yarn_order_lines'])) {
            return;
        }

        // Track style versions that need cache invalidation
        $styleVersionsToInvalidate = [];

        foreach ($this->orderEdit['yarn_order_lines'] as $line) {

            if (
                empty($line['price']) ||
                empty($line['yarn_colours_id']) ||
                empty($line['yarn_order_line_styles'])
            ) {
                continue;
            }

            foreach ($line['yarn_order_line_styles'] as $style) {

                // Which colourway are we talking about?
                $colourwaysId = CustomerOrderLines::whereKey(
                    $style['customer_order_lines_id'] ?? null
                )->value('colourways_id');

                if (! $colourwaysId) {
                    continue;
                }

                /* ---------- ②  update with a single SQL statement ---------- */

                if (class_exists(ColourwayYarns::class)) {
                    // uses Eloquent but still only ONE UPDATE under the hood
                    ColourwayYarns::where('colourways_id',  $colourwaysId)
                        ->where('yarn_colours_id', $line['yarn_colours_id'])
                        ->update(['price' => $line['price']]);
                } else {
                    // fall back to the query builder if you don't have a model
                    DB::table('colourway_yarns')
                        ->where('colourways_id',  $colourwaysId)
                        ->where('yarn_colours_id', $line['yarn_colours_id'])
                        ->update(['price' => $line['price']]);
                }

                // Track this colourway's style version for cache invalidation
                $colourway = \App\Models\Colourways::find($colourwaysId);
                if ($colourway && $colourway->style_versions_id) {
                    $styleVersionsToInvalidate[$colourway->style_versions_id] = true;
                }
            }
        }

        // Clear cache for all affected style versions
        // This fixes the bug where mass updates bypass model events
        foreach (array_keys($styleVersionsToInvalidate) as $styleVersionId) {
            Cache::tags(['yarn_costs', "style_version:{$styleVersionId}"])->flush();
            Cache::tags(['prices', "styleversion:{$styleVersionId}"])->flush();
        }
    }

    /**
     * Re-calculate the "yarn_cost" shown on every style row, **in memory only**.
     * Call this immediately after you have written the new prices to the
     * colourway_yarns table.
     */
    public function refreshYarnCosts(): void
    {
        foreach ($this->orderEdit['yarn_order_lines'] as &$line) {

            if (
                empty($line['yarn_colours_id']) ||
                empty($line['yarn_order_line_styles'])
            ) {
                continue;
            }

            // ── What is the new average price for this colourway + yarn? ──
            $avgPrice = DB::table('colourway_yarns')
                ->whereIn(
                    'colourways_id',
                    collect($line['yarn_order_line_styles'])
                        ->pluck('customer_order_lines_id')
                        ->unique()
                        ->map(function ($colLineId) {
                            return DB::table('customer_order_lines')
                                ->where('id', $colLineId)
                                ->value('colourways_id');
                        })
                )
                ->where('yarn_colours_id', $line['yarn_colours_id'])
                ->avg('price');

            // ── Push that new cost into every style that belongs to this PO line ──
            foreach ($line['yarn_order_line_styles'] as &$style) {
                data_set(
                    $style,
                    'customer_order_lines.yarn_cost',
                    $avgPrice
                );
            }
        }
    }
}
