<?php

namespace App\Livewire\Forms;

use Livewire\Form;
use App\Models\Design;
use App\Models\Styles;
use App\Models\Colourways;
use App\Models\StyleVersions;
use App\Models\ColourwayYarns;
use Livewire\Attributes\Locked;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Gate;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\Storage;

class StyleEditFormNew extends Form
{
    #[Locked]
    public $id;
    public $customers_id;
    public $departments_id;
    public $seasons_id;
    public $designs_id;
    public $category;
    public $commodity_codes_id;
    public $customer_ref;
    public $description;
    public $cancelled = false;
    public $carryover = false;
    public $notes;
    public $intake_id;

    // Nested style versions, with a default value.
    public $style_versions = [
        [
            'id' => null,
            'factory_id' => null,
            'name' => '',
            'gauge_id' => null,
            'yarn_ends' => null,
            'issued' => null,
            'sample_yarn' => 'Not Ordered',
            'sample_yarn_date' => null,
            'sample_accessories' => 'Not Ordered',
            'sample_accessories_date' => null,
            'colourways' => [
                [
                    'id' => null,
                    'name' => '',
                    'comments' => '',
                    'cancelled' => false,
                    'accessories' => '',
                    'colour_type' => 'solid',
                    'image' => null,
                    'img_thumb' => null,
                    'new_image' => null,
                    'colourway_yarns' => [
                        [
                            'id' => null,
                            'suppliers_id' => null,
                            'yarn_id' => null,
                            'yarn_colours_id' => null,
                            'cust_ref' => '',
                            'percentage' => null,
                            'price' => null,
                            'not_full_factored' => false,
                        ],
                    ],
                ],
            ],
        ]
    ];

    /**
     * Define validation rules for the entire form, including nested arrays.
     */
    protected function rules(): array
    {
        return [
            // Top-level fields with existence checks
            'customers_id'       => 'required|integer|min:1|exists:customers,id',
            'departments_id'     => 'required|integer|min:1|exists:departments,id',
            'seasons_id'         => 'required|integer|min:1|exists:seasons,id',
            'designs_id'         => 'required|integer|min:1|exists:designs,id',
            'category'           => 'required|in:ladies,mens,childrens,accessories',
            'commodity_codes_id' => 'nullable|integer|min:1|exists:commodity_codes,id',
            'customer_ref'       => 'nullable|string|max:35',
            'description'        => 'nullable|string|max:35',
            'cancelled'          => 'nullable|boolean',
            'carryover'          => 'nullable|boolean',
            'notes'              => 'nullable|string|max:5000',
            'intake_id'          => 'nullable|integer|min:1|exists:intakes,id',

            // Validate the style_versions array
            'style_versions' => 'required|array',

            // Each style version
            'style_versions.*.factory_id' => 'required|integer|min:1|exists:suppliers,id',
            'style_versions.*.name'       => humanizeAttribute('required|string|max:35'),
            'style_versions.*.gauge_id'   => 'required|integer|min:1|exists:gauges,id',
            'style_versions.*.yarn_ends'  => 'nullable|numeric',
            'style_versions.*.issued'     => 'nullable|date',
            'style_versions.*.sample_yarn' => 'required|in:Not Ordered,Expected,Arrived',
            'style_versions.*.sample_yarn_date' => 'nullable|date',
            'style_versions.*.sample_accessories' => 'required|in:Not Ordered,Expected,Arrived',
            'style_versions.*.sample_accessories_date' => 'nullable|date',

            // Each colourway in a style version
            'style_versions.*.colourways' => 'required|array',
            'style_versions.*.colourways.*.name' => humanizeAttribute('required|string|max:35'),
            'style_versions.*.colourways.*.customer_description' => 'nullable|string|max:35',
            'style_versions.*.colourways.*.comments' => 'nullable|string',
            'style_versions.*.colourways.*.cancelled' => 'nullable|boolean',
            'style_versions.*.colourways.*.accessories' => 'nullable|string',
            'style_versions.*.colourways.*.colour_type' => 'required|in:solid,melange,mixed',
            'style_versions.*.colourways.*.image' => 'nullable|string|max:150',
            'style_versions.*.colourways.*.img_thumb' => 'nullable|string|max:150',
            'style_versions.*.colourways.*.new_image' => 'nullable|image|max:20000',

            // Each yarn in a colourway
            'style_versions.*.colourways.*.colourway_yarns' => 'nullable|array',
            'style_versions.*.colourways.*.colourway_yarns.*.suppliers_id' => 'nullable|integer|min:1|exists:suppliers,id',
            'style_versions.*.colourways.*.colourway_yarns.*.yarn_id' => 'nullable|integer|min:1|exists:yarns,id',
            'style_versions.*.colourways.*.colourway_yarns.*.yarn_colours_id' => 'required|integer|min:1|exists:yarn_colours,id',
            'style_versions.*.colourways.*.colourway_yarns.*.cust_ref' => 'nullable|string|max:35',
            'style_versions.*.colourways.*.colourway_yarns.*.percentage' => 'nullable|numeric|min:0|max:100',
            'style_versions.*.colourways.*.colourway_yarns.*.price' => 'nullable|numeric',
            'style_versions.*.colourways.*.colourway_yarns.*.not_full_factored' => 'nullable|boolean',
        ];
    }


    /**
     * Override the error message formatting so that the :attribute placeholder
     * uses your humanized attribute name.
     *
     * @param  string  $attribute  The original attribute key (e.g. "style_versions.0.name")
     * @param  string  $rule       The validation rule (e.g. "required")
     * @param  string  $message    The original error message template
     * @return string
     */
    protected function formatErrorMessage($attribute, $rule, $message)
    {
        // Replace :attribute with a humanized version of the attribute key.
        return str_replace(':attribute', humanizeAttribute($attribute), $message);
    }



    /**
     * We do not override the constructor here.
     * Livewire will call the parent constructor using the correct signature:
     *   __construct(BaseComponent $component, array $values = [], ?callable $callback = null)
     *
     * If you need to perform additional initialization, consider using a lifecycle hook
     * such as mount() or a custom initialize() method.
     */

    /**
     * Fill the form with an array of values.
     *
     * @param mixed $values
     */
    public function fill($values)
    {
        if (!is_array($values)) {
            $values = [];
        }
        $this->id                 = $values['id'] ?? null;
        $this->customers_id       = $values['customers_id'] ?? null;
        $this->departments_id     = $values['departments_id'] ?? null;
        $this->seasons_id         = $values['seasons_id'] ?? null;
        $this->designs_id         = $values['designs_id'] ?? null;
        $this->category           = $values['category'] ?? null;
        $this->description        = $values['description'] ?? null;
        $this->commodity_codes_id = $values['commodity_codes_id'] ?? null;
        $this->customer_ref       = $values['customer_ref'] ?? '';
        $this->cancelled          = $values['cancelled'] ?? false;
        $this->carryover          = $values['carryover'] ?? false;
        $this->notes              = $values['notes'] ?? '';
        $this->intake_id          = $values['intake_id'] ?? null;
        $this->style_versions     = $values['style_versions'] ?? $this->style_versions;
    }

    /**
     * Perform validation and update the Style and its nested items.
     */
    public function update()
    {
        try {
            // Validate the incoming data.
            $this->validate();

            // Update (or create) the main Style record.
            $style = Styles::updateOrCreate(
                ['id' => $this->id ?? null],
                [
                    'customers_id'       => $this->customers_id,
                    'departments_id'     => $this->departments_id,
                    'seasons_id'         => $this->seasons_id,
                    'designs_id'         => $this->designs_id,
                    'category'           => $this->category,
                    'commodity_codes_id' => $this->commodity_codes_id,
                    'customer_ref'       => $this->customer_ref,
                    'cancelled'          => $this->cancelled,
                    'carryover'          => $this->carryover,
                    'notes'              => $this->notes,
                    'intake_id'          => $this->intake_id,
                ]
            );

            // Update the Design record.
            $style->designs->update(['description' => $this->description]);

            // Process each style version.
            $existingVersionIds = [];
            foreach ($this->style_versions as $versionData) {
                $version = StyleVersions::updateOrCreate(
                    ['id' => $versionData['id'] ?? null],
                    [
                        'styles_id'         => $style->id,
                        'factory_id'        => $versionData['factory_id'],
                        'name'              => $versionData['name'],
                        'gauge_id'          => $versionData['gauge_id'],
                        'yarn_ends'         => $versionData['yarn_ends'] ?? 0,
                        'issued'            => $versionData['issued'] ?? null,
                        'sample_yarn'       => $versionData['sample_yarn'] ?? 'Not Ordered',
                        'sample_yarn_date'  => $versionData['sample_yarn_date'] ?? null,
                        'sample_accessories'=> $versionData['sample_accessories'] ?? 'Not Ordered',
                        'sample_accessories_date' => $versionData['sample_accessories_date'] ?? null,
                    ]
                );
                $existingVersionIds[] = $version->id;

                // Process nested colourways for this version.
                $existingColourwayIds = [];
                foreach ($versionData['colourways'] as $cwData) {
                    $colourway = Colourways::updateOrCreate(
                        ['id' => $cwData['id'] ?? null],
                        [
                            'style_versions_id' => $version->id,
                            'name'              => $cwData['name'],
                            'customer_description' => $cwData['customer_description'],
                            'comments'          => $cwData['comments'] ?? '',
                            'cancelled'         => $cwData['cancelled'] ?? false,
                            'accessories'       => $cwData['accessories'] ?? '',
                            'colour_type'       => $cwData['colour_type'] ?? 'solid',
                        ]
                    );
                    $existingColourwayIds[] = $colourway->id;

                    // Process new image upload if available.
                    if (!empty($cwData['new_image'])) {
                        $img = Image::make($cwData['new_image'])
                            ->resize(1000, 1000, function ($constraint) {
                                $constraint->aspectRatio();
                                $constraint->upsize();
                            })
                            ->encode('jpg', 80);



                        $tempInput = tempnam(sys_get_temp_dir(), 'input_') . '.jpg';
                        $tempOutput = tempnam(sys_get_temp_dir(), 'output_') . '.png';

                        // Save the uploaded image (assumed to be an Intervention Image instance)
                        file_put_contents($tempInput, $img->encode('png'));

                        // Build the command safely
                        $cmd = "NUMBA_CACHE_DIR=/tmp HOME=/tmp rembg i '$tempInput' '$tempOutput' 2>&1";
                        exec($cmd, $output, $result);

                        // Check for errors
                        if ($result !== 0 || !file_exists($tempOutput) || filesize($tempOutput) < 1000) {
                            throw new \Exception('rembg failed or output image not valid.');
                        }

                        // Now load the result (e.g., to store or manipulate further)
                        $thumb = Image::make($tempOutput);

                        // Optionally clean up
                        @unlink($tempInput);
                        @unlink($tempOutput);

                        // Resize + encode thumbnail
                        $thumb = $thumb
                            ->resize(200, 200, function ($constraint) {
                                $constraint->aspectRatio();
                                $constraint->upsize();
                            })
                            ->encode('png');


                        if (!empty($cwData['image'])) {
                            Storage::delete('storage/' . $cwData['image']);
                        }
                        if (!empty($cwData['img_thumb'])) {
                            Storage::delete('storage/' . $cwData['img_thumb']);
                        }

                        $hash = md5($img->__toString());
                        $paththumb = "images/{$hash}-thumb.jpg";
                        if (Storage::disk('public')->put($paththumb, $thumb)) {
                            $colourway->update(['img_thumb' => $paththumb]);
                        }
                        $path = "images/{$hash}.jpg";
                        if (Storage::disk('public')->put($path, $img)) {
                            $colourway->update(['image' => $path]);
                        }
                    }

                    // Process nested colourway yarns.
                    $existingYarnIds = [];
                    foreach ($cwData['colourway_yarns'] as $yarnData) {
                        $yarn = ColourwayYarns::updateOrCreate(
                            ['id' => $yarnData['id'] ?? null],
                            [
                                'colourways_id'    => $colourway->id,
                                'cust_ref'         => $yarnData['cust_ref'] ?? '',
                                'percentage'       => $yarnData['percentage'] ?? 0,
                                'price'            => $yarnData['price'] ?? 0,
                                'not_full_factored'=> $yarnData['not_full_factored'] ?? false,
                                'yarn_colours_id'  => $yarnData['yarn_colours_id'],
                            ]
                        );
                        $existingYarnIds[] = $yarn->id;
                    }
                    // Delete yarns no longer in the form.
                    foreach ($colourway->colourway_yarns as $DByarn) {
                        if (!in_array($DByarn->id, $existingYarnIds)) {
                            $DByarn->delete();
                        }
                    }
                }
                // Delete colourways no longer present.
                foreach ($version->colourways as $DBcw) {
                    if (!in_array($DBcw->id, $existingColourwayIds)) {
                        $DBcw->delete();
                    }
                }
            }
            // Delete style versions no longer present.
            foreach ($style->style_versions as $DBversion) {
                if (!in_array($DBversion->id, $existingVersionIds)) {
                    $DBversion->delete();
                }
            }

            $this->reset('images');

            // Commodity code suggestion is now handled automatically by model events:
            // - Styles::created triggers suggestion for new styles
            // - ColourwayYarns::saved/deleted triggers suggestion when yarns change

            session()->flash('message', 'Saved!');
            session()->flash('alert-class', 'alert-success');
        } catch (\Exception $e) {
            // Log the error for debugging.
            \Log::error('Error updating style: ' . $e->getMessage(), ['exception' => $e]);
            // Flash an error message.
            session()->flash('message', 'Error Saving: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
            return;
        }
    }


    // -- Helper Methods for Nested Data Manipulation --

    public function addVersion()
    {
        // Append a new version without colourways.
        $this->style_versions[] = [
            'id' => null,
            'factory_id' => null,
            'name' => 'X',
            'gauge_id' => null,
            'yarn_ends' => null,
            'issued' => null,
            'sample_yarn' => 'Not Ordered',
            'sample_yarn_date' => null,
            'sample_accessories' => 'Not Ordered',
            'sample_accessories_date' => null,
            'colourways' => [],
        ];

        // Add a default colourway for the new version.
        $this->addColourway(array_key_last($this->style_versions));
    }


    public function removeVersion($index)
    {
        unset($this->style_versions[$index]);
        $this->style_versions = array_values($this->style_versions);
    }

    public function cloneVersion($index)
    {
        $version = $this->style_versions[$index];
        unset($version['id']);
        $version['name'] = $version['name'] . ' (Clone)';
        foreach ($version['colourways'] as $i => $cw) {
            unset($version['colourways'][$i]['id']);
            foreach ($cw['colourway_yarns'] as $j => $yarn) {
                unset($version['colourways'][$i]['colourway_yarns'][$j]['id']);
            }
        }
        $this->style_versions[] = $version;
    }

    public function addColourway($versionIndex)
    {
        // Ensure the version exists and has a colourways array
        if (!isset($this->style_versions[$versionIndex]) || !is_array($this->style_versions[$versionIndex])) {
            return;
        }
        
        if (!isset($this->style_versions[$versionIndex]['colourways'])) {
            $this->style_versions[$versionIndex]['colourways'] = [];
        }
        
        $this->style_versions[$versionIndex]['colourways'][] = [
            'id' => null,
            'name' => 'X',
            'comments' => '',
            'cancelled' => false,
            'accessories' => '',
            'colour_type' => 'solid',
            'image' => null,
            'img_thumb' => null,
            'new_image' => null,
            'colourway_yarns' => [],
        ];
    }

    public function removeColourway($versionIndex, $colourwayIndex)
    {
        if (!isset($this->style_versions[$versionIndex]['colourways'][$colourwayIndex])) {
            return;
        }
        
        unset($this->style_versions[$versionIndex]['colourways'][$colourwayIndex]);
        $this->style_versions[$versionIndex]['colourways'] = array_values($this->style_versions[$versionIndex]['colourways']);
    }

    public function cloneColourway($versionIndex, $colourwayIndex)
    {
        if (!isset($this->style_versions[$versionIndex]['colourways'][$colourwayIndex])) {
            return;
        }
        
        $colourway = $this->style_versions[$versionIndex]['colourways'][$colourwayIndex];
        unset($colourway['id']);
        $colourway['name'] = ($colourway['name'] ?? 'Colourway') . ' (Clone)';
        foreach ($colourway['colourway_yarns'] ?? [] as $i => $yarn) {
            unset($colourway['colourway_yarns'][$i]['id']);
        }
        $this->style_versions[$versionIndex]['colourways'][] = $colourway;
    }

    public function addYarn($versionIndex, $colourwayIndex)
    {
        if (!isset($this->style_versions[$versionIndex]['colourways'][$colourwayIndex])) {
            return;
        }
        
        if (!isset($this->style_versions[$versionIndex]['colourways'][$colourwayIndex]['colourway_yarns'])) {
            $this->style_versions[$versionIndex]['colourways'][$colourwayIndex]['colourway_yarns'] = [];
        }
        
        $this->style_versions[$versionIndex]['colourways'][$colourwayIndex]['colourway_yarns'][] = [
            'id' => null,
            'suppliers_id' => null,
            'yarn_id' => null,
            'yarn_colours_id' => null,
            'cust_ref' => '',
            'percentage' => null,
            'price' => null,
            'not_full_factored' => false,
        ];
    }

    public function removeYarn($versionIndex, $colourwayIndex, $yarnIndex)
    {
        if (!isset($this->style_versions[$versionIndex]['colourways'][$colourwayIndex]['colourway_yarns'][$yarnIndex])) {
            return;
        }
        
        unset($this->style_versions[$versionIndex]['colourways'][$colourwayIndex]['colourway_yarns'][$yarnIndex]);
        $this->style_versions[$versionIndex]['colourways'][$colourwayIndex]['colourway_yarns'] =
            array_values($this->style_versions[$versionIndex]['colourways'][$colourwayIndex]['colourway_yarns']);
    }

    public function cloneYarn($versionIndex, $colourwayIndex, $yarnIndex)
    {
        if (!isset($this->style_versions[$versionIndex]['colourways'][$colourwayIndex]['colourway_yarns'][$yarnIndex])) {
            return;
        }
        
        $yarn = $this->style_versions[$versionIndex]['colourways'][$colourwayIndex]['colourway_yarns'][$yarnIndex];
        unset($yarn['id']);
        $this->style_versions[$versionIndex]['colourways'][$colourwayIndex]['colourway_yarns'][] = $yarn;
    }

    public function setCommodityCode($code){
        $this->commodity_codes_id = $code;
    }

    public function setDesigns($design){
        $this->designs_id = $design;
    }
}
