<?php

namespace App\Livewire\Forms;

use Livewire\Form;
use App\Models\CustomerOrders;
use App\Models\CustomerOrderFiles;
use App\Models\Sizes;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Rule;

class OrdersEditV2Form extends Form
{
    public ?CustomerOrders $order = null;
    
    public $isNew = false;
    
    // Standard order fields
    #[Rule('required_if:isNew,true|nullable|numeric|min:1')]
    public $departments_id = null;
    
    #[Rule('required|numeric|min:1')]
    public $seasons_id = null;
    
    #[Rule('required|numeric|min:1')]
    public $customers_id = null;
    
    #[Rule('required|in:EXW,FCA,CPT,CIP,DAP,DPU,DDP,FAS,FOB,CFR,CIF')]
    public $incoterms = 'EXW';
    
    #[Rule('nullable|numeric|min:1')]
    public $customer_addresses_id = null;
    
    #[Rule('required|string|max:35')]
    public $customer_po = '';
    
    #[Rule('nullable|string|max:35')]
    public $shipment_mode = null;
    
    #[Rule('nullable|date')]
    public $order_date = null;
    
    // Advanced fields
    #[Rule('nullable|string|max:35')]
    public $cust_ship_no = null;
    
    #[Rule('nullable|date')]
    public $sent_to_factory = null;
    
    #[Rule('nullable|numeric|min:1')]
    public $phase_id = null;
    
    #[Rule('required|boolean')]
    public $incomplete_flag = false;
    
    #[Rule('required|boolean')]
    public $cancelled = false;
    
    public $order_type = 'wholesale';
    
    // File uploads
    public $uploadFiles = [];
    
    /**
     * Open the form with an order or for a new order
     */
    public function open(?CustomerOrders $order = null)
    {
        if ($order) {
            $this->order = $order;
            $this->isNew = false;
            
            // Populate form fields
            $this->departments_id = $order->departments_id;
            $this->seasons_id = $order->seasons_id;
            $this->customers_id = $order->customers_id;
            $this->incoterms = $order->incoterms ?? 'EXW';
            $this->customer_addresses_id = $order->customer_addresses_id;
            $this->customer_po = $order->customer_po;
            $this->cust_ship_no = $order->cust_ship_no;
            $this->order_date = $order->order_date?->format('Y-m-d');
            $this->sent_to_factory = $order->sent_to_factory?->format('Y-m-d');
            $this->incomplete_flag = $order->incomplete_flag;
            $this->cancelled = $order->cancelled;
            $this->order_type = $order->order_type ?? 'wholesale';
            $this->shipment_mode = $order->shipment_mode;
            $this->phase_id = $order->phase_id;
        } else {
            $this->isNew = true;
            $this->order = null;
            
            // Set defaults for new order
            $this->reset();
            $this->incomplete_flag = false;
            $this->cancelled = false;
            $this->order_type = 'wholesale';
            $this->incoterms = 'EXW';
        }
    }
    
    /**
     * Close the form and reset
     */
    public function close()
    {
        $this->reset();
        $this->order = null;
    }
    
    /**
     * Check if the order is read-only (season locked)
     */
    public function readOnly(): bool
    {
        if (!$this->order) return false;
        return $this->order->seasons->locked ?? false;
    }
    
    /**
     * Save the order
     */
    public function save(): bool
    {
        // Validate
        $this->validate();
        
        // Check if season is locked (for existing orders)
        if ($this->order && $this->readOnly()) {
            session()->flash('message', 'Cannot edit - season is locked');
            session()->flash('alert-class', 'alert-danger');
            return false;
        }
        
        try {
            // Create or update order
            if ($this->order) {
                $order = $this->order;
            } else {
                $order = new CustomerOrders();
            }
            
            // Populate order fields
            $order->departments_id = $this->departments_id;
            $order->seasons_id = $this->seasons_id;
            $order->customers_id = $this->customers_id;
            $order->incoterms = $this->incoterms;
            $order->customer_addresses_id = $this->customer_addresses_id;
            $order->customer_po = $this->customer_po;
            $order->cust_ship_no = $this->cust_ship_no;
            $order->order_date = $this->order_date;
            $order->sent_to_factory = $this->sent_to_factory;
            $order->incomplete_flag = $this->incomplete_flag;
            $order->cancelled = $this->cancelled;
            $order->order_type = $this->order_type;
            $order->shipment_mode = $this->shipment_mode;
            $order->phase_id = $this->phase_id;
            
            $order->save();
            
            // Store order reference for new orders
            $this->order = $order;
            $this->isNew = false;
            
            // Handle file uploads
            if (!empty($this->uploadFiles)) {
                foreach ($this->uploadFiles as $file) {
                    $path = Storage::disk('public')->put('customerOrderFiles', $file);
                    
                    if ($path) {
                        $fileEntry = new CustomerOrderFiles();
                        $fileEntry->customer_orders_id = $order->id;
                        $fileEntry->filename = $file->getClientOriginalName();
                        $fileEntry->file = $path;
                        $fileEntry->save();
                    }
                }
                
                $this->uploadFiles = [];
            }
            
            return true;
            
        } catch (\Exception $e) {
            \Log::error('Error saving order: ' . $e->getMessage());
            session()->flash('message', 'Error saving order: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
            return false;
        }
    }
}














