<?php

namespace App\Livewire\Forms;

use Exception;
use Throwable;
use Livewire\Form;
use App\Models\Price;
use App\Models\Sizes;
use App\Models\Seasons;
use App\Helper\Functions;
use App\Helper\Conversions;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Livewire\Attributes\Rule;
use App\Models\CustomerOrders;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Validate;
use App\Models\CustomerOrderFiles;
use App\Models\CustomerOrderLines;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Storage;
use App\Models\CustomerOrderLineQuantities;

class OrdersEditLineForm extends Form
{
	public ?CustomerOrderLines $orderLine;
	public $orderLineEdit, $uploadFiles = [], $removeItems = [], $removeDrops = [], $removeSizes = [], $new = FALSE, $sizes, $orderType;

    protected function rules(): array
    {
        return [
			'orderLineEdit.colourways_id' => 'required|numeric|min:1',
			'orderLineEdit.factory_cust_date' => 'nullable|date',
			'orderLineEdit.wh_cust_date' => 'nullable|date',
			'orderLineEdit.start_knit' => 'nullable|date',

            'orderLineEdit.customer_order_line_quantities.*.price' => 'nullable|numeric|min:0',
			'orderLineEdit.customer_order_line_quantities.*.qty' => 'nullable|numeric',
			'orderLineEdit.customer_order_line_quantities.*.SKU' => 'nullable|string|max:35',
			'orderLineEdit.customer_order_line_quantities.*.barcode' => 'nullable|string|max:35',
			'orderLineEdit.customer_order_line_quantities.*.prices_id' => 'nullable|numeric|exists:prices,id',

			'orderLineEdit.shipment_lines.*.exfty' => 'required|date',
			'orderLineEdit.shipment_lines.*.shipment_line_sizes.*.qty' => 'nullable|numeric',
			'orderLineEdit.shipment_lines.*.shipment_line_sizes.*.shipped_qty' => 'nullable|numeric',
        ];
    }


	public function open($orderLine = NULL, $orderType = 'wholesale'){
        $this->sizes = Sizes::all();

        $this->orderType = $orderType;

		if($orderLine){
            // Add size names to customer_order_line_quantities
            foreach($orderLine->customer_order_line_quantities as $q=>$qty){
                $orderLine->customer_order_line_quantities[$q]->size = $this->sizes->where('id', $qty->sizes_id)->first()->name;
            }
            
            // Convert to array first
			$this->orderLineEdit = $orderLine->toArray();
            
            // Ensure prices_id is explicitly null for display if not set
            foreach($this->orderLineEdit['customer_order_line_quantities'] as $q => $qty){
                if(empty($qty['prices_id']) || $qty['prices_id'] === 0 || $qty['prices_id'] === '0'){
                    $this->orderLineEdit['customer_order_line_quantities'][$q]['prices_id'] = null;
                }
            }
            
            // Now reorganize shipment_line_sizes to match customer_order_line_quantities order by size_id
            foreach($this->orderLineEdit['shipment_lines'] as $sl => $shipmentLine){
                // Create a map of size_id => shipment_line_size data
                $sizeMap = [];
                foreach($shipmentLine['shipment_line_sizes'] ?? [] as $sls){
                    $sizeMap[$sls['sizes_id']] = $sls;
                }
                
                // Rebuild shipment_line_sizes array in the same order as customer_order_line_quantities
                $reorderedSizes = [];
                foreach($this->orderLineEdit['customer_order_line_quantities'] as $qty){
                    $sizesId = $qty['sizes_id'];
                    if(isset($sizeMap[$sizesId])){
                        // Use existing shipment line size
                        $reorderedSizes[] = $sizeMap[$sizesId];
                    } else {
                        // Create placeholder if this size doesn't exist in this drop yet
                        $reorderedSizes[] = [
                            'id' => null,
                            'shipment_line_id' => $shipmentLine['id'] ?? null,
                            'sizes_id' => $sizesId,
                            'qty' => 0,
                            'shipped_qty' => 0,
                            'sizes' => $qty['sizes'] ?? null,
                        ];
                    }
                }
                
                // Replace the shipment_line_sizes with the reordered array
                $this->orderLineEdit['shipment_lines'][$sl]['shipment_line_sizes'] = $reorderedSizes;
            }
		}
	}

    public function removeSize($s){
        unset($this->orderLineEdit['customer_order_line_quantities'][$s]);
        foreach($this->orderLineEdit['shipment_lines'] as $l=>$line){
            unset($this->orderLineEdit['shipment_lines'][$l]['shipment_line_sizes'][$s]);
        }
    }
    public function addSize(){
        $this->orderLineEdit['customer_order_line_quantities'][] = ['discount' => $this->orderLineEdit['customer_orders']['customers']['default_discount'], 'commission' => $this->orderLineEdit['customer_orders']['customers']['default_commission']];
    }

    public function addCustSizes($customer){
        $sizes = Sizes::whereIn('id', $customer->setting('size-defaults'))->orderBy('order')->get();
        // dd($sizes);
        $sizes_array = [];
        foreach($sizes as $i=>$s){
            $sizes_array[] = ['size' => $s->name];
        }
        $this->orderLineEdit['customer_order_line_quantities'] = $sizes_array;
    }

    public function addDrop(){
        $this->orderLineEdit['shipment_lines'][] = [];
    }

    public function deleteDrop($d, $drop){
        unset($this->orderLineEdit['shipment_lines'][$d]);
        if(!empty($drop))
            ShipmentLine::find($drop)->delete();
    }

    public function viewOnly(){
        return Seasons::whereRelation('customer_orders', 'id', $this->orderLineEdit['customer_orders_id'])->first()->locked;
    }

	public function save(){
        $this->validate();

        if(!$this->viewOnly() && Gate::check('order:update')){
            $orderLine = CustomerOrderLines::find($this->orderLineEdit['id']);

            $orderLine->factory_cust_date = $this->orderLineEdit['factory_cust_date'] ?? NULL;
            $orderLine->wh_cust_date = $this->orderLineEdit['wh_cust_date'] ?? NULL;
            $orderLine->start_knit = $this->orderLineEdit['start_knit'] ?? NULL;
            $orderLine->notes = $this->orderLineEdit['notes'] ?? NULL;
            $orderLine->save();

            $lines = [];
            foreach($this->orderLineEdit['customer_order_line_quantities'] ?? [] as $q=>$qty){
                if(($qty['sizes_id'] = $this->sizes->where('name', $qty['size'])->first()?->id) == NULL){
                    $qty['sizes_id'] = Sizes::firstOrCreate([
                        'name' => $qty['size'],
                    ])->id;
                }

                $orderLine->load('colourways.style_versions.prices', 'colourways.style_versions.styles.seasons');

                // Use price resolution cache instead of Functions::getPrice()
                // Create a temporary quantity object to get price model
                $tempQuantity = new CustomerOrderLineQuantities;
                $tempQuantity->customer_order_lines_id = $orderLine->id;
                $tempQuantity->sizes_id = $qty['sizes_id'];
                $tempQuantity->customer_order_lines = $orderLine;
                
                $priceModel = $tempQuantity->price_model;
                $price = $priceModel ? (object) $priceModel : null;

                $quantity = new CustomerOrderLineQuantities;

                $quantity->customer_order_lines_id = $orderLine->id;
                $quantity->sizes_id = $qty['sizes_id'];
                $quantity->price = $qty['price'] ?? 0;
                $quantity->discount = $qty['discount'] ?? 0;
                $quantity->commission = $qty['commission'] ?? 0;
                $quantity->qty = $qty['qty'] ?? 0;
                $quantity->SKU = $qty['SKU'] ?? NULL;
                $quantity->barcode = $qty['barcode'] ?? NULL;
                
                // Set specific price ID if provided (priority pricing)
                // Explicitly check for non-empty, non-zero values
                if(!empty($qty['prices_id']) && $qty['prices_id'] !== '' && $qty['prices_id'] !== '0' && $qty['prices_id'] !== 0){
                    $quantity->prices_id = $qty['prices_id'];
                } else {
                    $quantity->prices_id = NULL;
                }

                if(!is_string($price)){
                    // Price resolution system will handle caching automatically
                    // If prices_id is set, it will be used as priority
                }
                $lines[] = $quantity;
            }

            $orderLine->customer_order_line_quantities()->forceDelete();
            $orderLine->customer_order_line_quantities()->saveMany($lines);
            
            // Invalidate total cache for order line and parent order
            $totalCacheService = app(\App\Services\TotalCacheService::class);
            $totalCacheService->invalidateEntity('customer_order_line', $orderLine->id);
            $totalCacheService->invalidateEntity('customer_order', $orderLine->customer_orders_id);
            
            // Warm up price resolutions for all sizes in this colourway
            // This ensures the shipment schedule can find prices via SQL join
            $priceResolutionService = app(\App\Services\PriceResolutionService::class);
            $priceResolutionService->warmupColourway($orderLine->colourways_id);

            $exftyDates = array_filter(array_column($this->orderLineEdit['shipment_lines'] ?? [], 'exfty'));

            $duplicates = array_diff_assoc($exftyDates, array_unique($exftyDates));

            if (!empty($duplicates)) {
                throw new \Exception("Duplicate EXFTY dates found. Each drop must have a unique EXFTY.");
            }

            foreach($this->orderLineEdit['shipment_lines'] ?? [1] as $sl=>$shipmentLine){

                //IF ONLY SHIPMENT LINE - PREVENTS DUPLICATES BEING CREATED
                if(count($this->orderLineEdit['shipment_lines']) == 1){
                    $drop = ShipmentLine::updateOrCreate(
                        ['customer_order_lines_id' => $orderLine->id],
                        ['first_exfty' => $shipmentLine['exfty'],
                        'exfty' => $shipmentLine['exfty'],
                        ]
                    );
                }
                else{
                    $drop = ShipmentLine::updateOrCreate(
                        ['id' => $shipmentLine['id'] ?? 0],
                        ['customer_order_lines_id' => $orderLine->id,
                        'first_exfty' => $shipmentLine['exfty'],
                        'exfty' => $shipmentLine['exfty'],
                        ]
                    );
                }


                $lines = [];
                foreach($this->orderLineEdit['customer_order_line_quantities'] as $q=>$qty){
                    if(($qty['sizes_id'] = $this->sizes->where('name', $qty['size'])->first()?->id) == NULL){
                        $qty['sizes_id'] = Sizes::firstOrCreate([
                            'name' => $qty['size'],
                        ])->id;
                    }
                    $quantity = new ShipmentLineSizes;
                    $quantity->shipment_line_id = $drop->id;
                    $quantity->sizes_id = $qty['sizes_id'];
                    if(count($this->orderLineEdit['shipment_lines']) == 1){
                        $quantity->qty = $qty['qty'] ?? 0;
                    }
                    else{
                        $quantity->qty = $shipmentLine['shipment_line_sizes'][$q]['qty'] ?? 0;
                    }

                    $quantity->shipped_qty = empty($shipmentLine['shipment_line_sizes'][$q]['shipped_qty']) ? 0 : $shipmentLine['shipment_line_sizes'][$q]['shipped_qty'];
                    $lines[] = $quantity;
                }

                $drop->shipment_line_sizes()->forceDelete();
                $drop->shipment_line_sizes()->saveMany($lines);
                
                // Invalidate shipment line and shipment cache
                $totalCacheService->invalidateEntity('shipment_line', $drop->id);
                if ($drop->shipment_id) {
                    $totalCacheService->invalidateEntity('shipment', $drop->shipment_id);
                }
            }

            session()->flash('message', 'Line Saved');
            session()->flash('alert-class', 'alert-success');
        }
	}
}
