<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Services\TotalCacheService;
use App\Models\ShipmentLine;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrders;
use App\Models\Shipment;

class WarmupTotalCacheJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct(
        public readonly string $entityType,
        public readonly int $entityId
    ) {}

    /**
     * Execute the job.
     */
    public function handle(TotalCacheService $totalCacheService): void
    {
        try {
            switch ($this->entityType) {
                case 'shipment_line':
                    $entity = ShipmentLine::with([
                        'shipment_line_sizes',
                        'customer_order_lines.customer_order_line_quantities'
                    ])->find($this->entityId);
                    
                    if ($entity) {
                        $totalCacheService->warmupShipmentLine($entity);
                    }
                    break;

                case 'customer_order_line':
                    $entity = CustomerOrderLines::with([
                        'customer_order_line_quantities'
                    ])->find($this->entityId);
                    
                    if ($entity) {
                        $totalCacheService->warmupCustomerOrderLine($entity);
                    }
                    break;

                case 'customer_order':
                    $entity = CustomerOrders::with([
                        'customer_order_lines.customer_order_line_quantities'
                    ])->find($this->entityId);
                    
                    if ($entity) {
                        $totalCacheService->warmupCustomerOrder($entity);
                    }
                    break;

                case 'shipment':
                    $entity = Shipment::with([
                        'shipment_lines.shipment_line_sizes',
                        'shipment_lines.customer_order_lines.customer_order_line_quantities'
                    ])->find($this->entityId);
                    
                    if ($entity) {
                        $totalCacheService->warmupShipment($entity);
                    }
                    break;

                default:
                    \Log::warning("WarmupTotalCacheJob: Unknown entity type", [
                        'entity_type' => $this->entityType,
                        'entity_id' => $this->entityId
                    ]);
            }
        } catch (\Exception $e) {
            \Log::error('WarmupTotalCacheJob failed', [
                'entity_type' => $this->entityType,
                'entity_id' => $this->entityId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Re-throw to mark job as failed and allow for retry
            throw $e;
        }
    }
}



















