<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Services\PriceResolutionService;
use App\Services\TotalCacheService;
use App\Models\Price;

class WarmupPriceCacheJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct(
        public readonly int $priceId,
        public readonly bool $isStyleVersionWarmup = false
    ) {}

    /**
     * Execute the job.
     */
    public function handle(PriceResolutionService $priceResolutionService, TotalCacheService $totalCacheService): void
    {
        try {
            // Get the price to determine what needs to be warmed up
            $price = Price::find($this->priceId);
            if (!$price) {
                \Log::warning('WarmupPriceCacheJob: Price not found', ['price_id' => $this->priceId]);
                return;
            }

            // IMPORTANT: Warm up priority price resolutions FIRST
            // This must run before warmupColourway to ensure priority prices take precedence
            $this->warmupPriorityPriceResolutions($price, $priceResolutionService);
            
            // Then warm up dynamic price resolution cache for other sizes
            if ($this->isStyleVersionWarmup || $price->colourways_id === null) {
                // For general prices (NULL colourway) or style version updates,
                // warm up cache for all colourways of this style version
                $colourways = \App\Models\Colourways::where('style_versions_id', $price->style_versions_id)
                    ->pluck('id');
                    
                foreach ($colourways as $colourwayId) {
                    $priceResolutionService->warmupColourway($colourwayId);
                }
            } else {
                // For specific colourway prices, warm up cache for that colourway
                $priceResolutionService->warmupColourway($price->colourways_id);
            }
            
            // After price resolutions are warmed up, dispatch total cache warmup jobs
            // This will recalculate totals for affected entities
            $this->dispatchTotalCacheWarmup($price, $totalCacheService);
        } catch (\Exception $e) {
            \Log::error('WarmupPriceCacheJob failed', [
                'price_id' => $this->priceId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Re-throw to mark job as failed and allow for retry
            throw $e;
        }
    }
    
    /**
     * Dispatch total cache warmup jobs for affected entities.
     */
    private function dispatchTotalCacheWarmup(Price $price, TotalCacheService $totalCacheService): void
    {
        // Get all affected shipment lines
        $shipmentLineIds = \App\Models\ShipmentLine::whereHas('customer_order_lines.colourways', function($query) use ($price) {
            if ($price->colourways_id === null) {
                $query->where('style_versions_id', $price->style_versions_id);
            } else {
                $query->where('id', $price->colourways_id);
            }
        })->pluck('id')->toArray();

        if (!empty($shipmentLineIds)) {
            // Invalidate and dispatch warmup for affected shipment lines
            $totalCacheService->invalidateAndDispatchWarmupBatch('shipment_line', $shipmentLineIds);
        }
        
        // Get all affected customer order lines
        $orderLineIds = \App\Models\CustomerOrderLines::whereHas('colourways', function($query) use ($price) {
            if ($price->colourways_id === null) {
                $query->where('style_versions_id', $price->style_versions_id);
            } else {
                $query->where('id', $price->colourways_id);
            }
        })->pluck('id')->toArray();

        if (!empty($orderLineIds)) {
            // Invalidate and dispatch warmup for affected customer order lines
            $totalCacheService->invalidateAndDispatchWarmupBatch('customer_order_line', $orderLineIds);
        }
    }
    
    /**
     * Warm up priority price resolutions for order line quantities linked to this price.
     */
    private function warmupPriorityPriceResolutions(Price $price, PriceResolutionService $priceResolutionService): void
    {
        // Find all order line quantities that have this price linked as priority
        $linkedQuantities = \App\Models\CustomerOrderLineQuantities::with([
            'customer_order_lines.colourways',
            'customer_order_lines.customer_orders'
        ])
        ->where('prices_id', $price->id)
        ->get();
        
        // Group quantities by colourway/phase/season to batch invalidate
        $invalidationGroups = [];
        foreach ($linkedQuantities as $qty) {
            $colourwayId = $qty->customer_order_lines->colourways_id;
            $seasonId = $qty->customer_order_lines->customer_orders->seasons_id;
            $phaseId = $qty->customer_order_lines->phase_id;
            $key = "{$colourwayId}_{$seasonId}_{$phaseId}";
            
            if (!isset($invalidationGroups[$key])) {
                $invalidationGroups[$key] = [
                    'colourway_id' => $colourwayId,
                    'season_id' => $seasonId,
                    'phase_id' => $phaseId,
                    'size_ids' => []
                ];
            }
            $invalidationGroups[$key]['size_ids'][] = $qty->sizes_id;
        }
        
        // Delete ALL price_resolutions for these specific combinations
        // This ensures dynamic price_resolutions don't conflict with priority ones
        // and no stale data remains in the database
        foreach ($invalidationGroups as $group) {
            \App\Models\PriceResolution::where('colourways_id', $group['colourway_id'])
                ->where('season_id', $group['season_id'])
                ->where(function($q) use ($group) {
                    if ($group['phase_id'] === null) {
                        $q->whereNull('phase_id');
                    } else {
                        $q->where('phase_id', $group['phase_id']);
                    }
                })
                ->whereIn('sizes_id', $group['size_ids'])
                ->delete();
        }
        
        // Now create the priority price resolutions
        foreach ($linkedQuantities as $qty) {
            try {
                $colourway = $qty->customer_order_lines->colourways;
                $seasonId = $qty->customer_order_lines->customer_orders->seasons_id;
                $phaseId = $qty->customer_order_lines->phase_id;
                
                // Recompute the priority price resolution
                $priceResolutionService->resolveWithSpecificPrice(
                    $price->id,
                    $colourway->style_versions_id,
                    $colourway->id,
                    $qty->sizes_id,
                    $phaseId,
                    $seasonId
                );
            } catch (\Exception $e) {
                \Log::warning('Failed to warm up priority price resolution for quantity', [
                    'quantity_id' => $qty->id,
                    'price_id' => $price->id,
                    'error' => $e->getMessage()
                ]);
            }
        }
    }
}














