<?php

namespace App\Jobs;

use App\Models\Styles;
use App\Services\CommodityCodeSuggestionService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class SuggestCommodityCode implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $styleId;
    public bool $forceUpdate;

    /**
     * The number of times the job may be attempted.
     */
    public int $tries = 2;

    /**
     * The number of seconds to wait before retrying the job.
     */
    public int $backoff = 30;

    /**
     * Create a new job instance.
     */
    public function __construct(int $styleId, bool $forceUpdate = false)
    {
        $this->styleId = $styleId;
        $this->forceUpdate = $forceUpdate;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $style = Styles::find($this->styleId);

        if (!$style) {
            Log::warning('SuggestCommodityCode: Style not found', ['style_id' => $this->styleId]);
            return;
        }

        // Skip if commodity code already set and not forcing update
        if ($style->commodity_codes_id && !$this->forceUpdate) {
            Log::info('SuggestCommodityCode: Style already has commodity code', [
                'style_id' => $this->styleId,
                'commodity_code_id' => $style->commodity_codes_id,
            ]);
            return;
        }

        try {
            $service = new CommodityCodeSuggestionService();
            $service->suggestAndUpdate($style);
        } catch (\Exception $e) {
            Log::error('SuggestCommodityCode: Failed to suggest commodity code', [
                'style_id' => $this->styleId,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }
}
