<?php

namespace App\Jobs;

use App\Models\CommissionImportBatch;
use App\Models\CommissionImportBatchFile;
use App\Services\CommissionImports\CommissionExtractorFactory;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use App\Support\StoredUploadedFile;

class ProcessCommissionFile implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $timeout = 600; // 10 minutes per file (PDF conversion + AI can be slow)
    public $tries = 1; // Don't auto-retry - user can manually retry from UI

    public function __construct(
        public CommissionImportBatchFile $batchFile
    ) {}

    public function handle(): void
    {
        // Refresh the model to get latest status
        $this->batchFile->refresh();
        
        // Skip if already extracted or imported (prevent duplicate processing)
        if (in_array($this->batchFile->status, ['extracted', 'imported'])) {
            Log::info('Skipping already processed file', [
                'file_id' => $this->batchFile->id,
                'status' => $this->batchFile->status,
            ]);
            return;
        }
        
        // Check if this is a retry (marked in metadata by the retry action)
        $isRetry = ($this->batchFile->metadata['is_retry'] ?? false) === true;
        
        try {
            // Mark as processing
            $this->batchFile->update(['status' => 'processing']);
            
            $batch = $this->batchFile->batch;
            
            Log::info('Processing commission file', [
                'batch_id' => $batch->id,
                'file_id' => $this->batchFile->id,
                'filename' => $this->batchFile->original_filename,
            ]);

            // Get the stored file
            $storedPath = $this->batchFile->file_path;
            
            if (!Storage::exists($storedPath)) {
                throw new \RuntimeException("File not found in storage: {$storedPath}");
            }

            // Create a temporary uploaded file object
            $tempFile = $this->createTemporaryUploadedFile($storedPath);

            // Get the appropriate extractor
            $extractor = CommissionExtractorFactory::make($batch->customer_name);

            // Extract data from the file
            $extractedData = $extractor->extractData($tempFile);

            // Store the extracted data (and clear the retry flag)
            $metadata = $extractedData['metadata'] ?? [];
            unset($metadata['is_retry']);
            $this->batchFile->markAsExtracted($extractedData, $metadata);

            // Only increment counters if this is not a retry
            if (!$isRetry) {
                $batch->increment('processed_files');
                $batch->increment('successful_files');
            } else {
                // If this is a successful retry, decrement the failed count
                $batch->decrement('failed_files');
                $batch->increment('successful_files');
            }

            // Check if batch is complete
            if ($batch->isComplete()) {
                $batch->markAsCompleted();
            }

            Log::info('Commission file processed successfully', [
                'batch_id' => $batch->id,
                'file_id' => $this->batchFile->id,
                'orders' => $this->batchFile->total_orders,
                'lines' => $this->batchFile->total_lines,
                'quantity' => $this->batchFile->total_quantity,
            ]);

        } catch (\Throwable $e) {
            Log::error('Failed to process commission file', [
                'batch_id' => $this->batchFile->batch_id,
                'file_id' => $this->batchFile->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            $this->batchFile->markAsFailed($e->getMessage());
            
            // Only increment counters if this is not a retry
            if (!$isRetry) {
                $this->batchFile->batch->increment('processed_files');
                $this->batchFile->batch->increment('failed_files');
            }

            // Don't rethrow - we've handled the error
        }
    }

    protected function createTemporaryUploadedFile(string $storagePath): StoredUploadedFile
    {
        // Get the full path from local disk
        $fullPath = Storage::disk('local')->path($storagePath);
        
        // Create a stored file wrapper
        return new StoredUploadedFile($fullPath, $this->batchFile->original_filename);
    }
}
