<?php

namespace App\Jobs;

use App\Models\CommissionImportBatch;
use App\Models\User;
use App\Services\CommissionImports\MultiAgentImportService;
use App\Notifications\CommissionImportComplete;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class ProcessCommissionBatchImport implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $timeout = 1800; // 30 minutes
    public int $tries = 1;

    public function __construct(
        public CommissionImportBatch $batch
    ) {}

    public function handle(MultiAgentImportService $importService): void
    {
        Log::info("ProcessCommissionBatchImport - Starting job", [
            'batch_id' => $this->batch->id,
            'batch_name' => $this->batch->batch_name,
        ]);

        try {
            // Process through multi-agent pipeline
            $importService->processBatch($this->batch);

            // Notify the uploader
            $this->notifyUploader();

            Log::info("ProcessCommissionBatchImport - Job complete", [
                'batch_id' => $this->batch->id,
            ]);

        } catch (\Exception $e) {
            Log::error("ProcessCommissionBatchImport - Job failed", [
                'batch_id' => $this->batch->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            $this->batch->update([
                'status' => 'failed',
                'workflow_status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    protected function notifyUploader(): void
    {
        $user = User::find($this->batch->user_id);
        
        if (!$user || !$user->email) {
            Log::warning("ProcessCommissionBatchImport - Cannot notify user", [
                'user_id' => $this->batch->user_id
            ]);
            return;
        }

        try {
            $user->notify(new CommissionImportComplete($this->batch));
            
            Log::info("ProcessCommissionBatchImport - User notified", [
                'user_id' => $user->id,
                'email' => $user->email,
            ]);
        } catch (\Exception $e) {
            Log::error("ProcessCommissionBatchImport - Notification failed", [
                'error' => $e->getMessage()
            ]);
        }
    }

    public function failed(\Throwable $exception): void
    {
        Log::error("ProcessCommissionBatchImport - Job failed permanently", [
            'batch_id' => $this->batch->id,
            'error' => $exception->getMessage(),
        ]);

        $this->batch->update([
            'status' => 'failed',
            'workflow_status' => 'failed',
            'error_message' => $exception->getMessage(),
        ]);
    }
}

