<?php

namespace App\Http\Livewire\Zoho;

use App\ZohoInvoice;
use Livewire\Component;
use App\Models\Customer;
use App\Helper\Functions;
use App\Facades\ZohoFacade;
use App\Helper\Conversions;
use App\Models\ShipmentLine;
use Livewire\WithPagination;
use App\Models\InvoiceLayout;
use App\Services\ZohoService;
use Illuminate\Support\Carbon;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;

use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use League\OAuth2\Client\Token\AccessToken;
use App\Http\Livewire\Traits\HandlesZohoOAuth;
use League\OAuth2\Client\Provider\Exception\IdentityProviderException;

class ZohoInvoiceManager extends Component
{
    use HandlesZohoOAuth;
    public $customerInvoice;
    public $supplier;
    public $date;
    private $zoho;
    public bool $zohoConnected = false;
    public $createdInvoices = [];
    public $totalInvoices = 0;
    public $addressWarnings = [];
    public $tagWarnings = [];
    public $completionMessage = '';
    public $showCompletionModal = false;

    public function refresh(){
        $this->reset(['completionMessage', 'showCompletionModal', 'createdInvoices', 'addressWarnings', 'tagWarnings']);
        $this->dispatch('refresh');
    }
    
    /**
     * Clear all Zoho cached data (accounts, customers, templates, etc.)
     * Use this if you're getting stale data errors or have made changes in Zoho.
     */
    public function clearZohoCache()
    {
        try {
            $this->zoho = app(ZohoService::class);
            $this->zoho->clearCache();
            Session::flash('message', 'Zoho cache cleared successfully. Data will be refreshed on next request.');
            Session::flash('alert-class', 'alert-success');
        } catch (\Exception $e) {
            Log::error('Failed to clear Zoho cache', ['error' => $e->getMessage()]);
            Session::flash('message', 'Failed to clear Zoho cache: ' . $e->getMessage());
            Session::flash('alert-class', 'alert-danger');
        }
    }

    public function hydrate(){
        $this->zoho = app(ZohoService::class);

        if ($this->zoho->isConnected()) {
            $this->zohoConnected = true;
        } else {
            $this->authUrl = $this->zoho->getAuthUrl();
            $this->zohoConnected = false;
        }
    }

    public function render()
    {

        $this->zoho = app(ZohoService::class);

        if ($this->zoho->isConnected()) {
            $this->zohoConnected = true;
        } else {
            $this->authUrl = $this->zoho->getAuthUrl();
            $this->zohoConnected = false;
        }

        return view('livewire.zoho.zoho-invoice-manager', [
            'customers' => $this->getCustomers(),
        ]);
    }

    public function getCustomers()
    {
        // Return a list of customers to populate the dropdown
        return Customer::all()->sortBy('name');
    }

    #[Computed]
    public function suppliers()
    {
        return $this->getSuppliers();
    }

    public function sendInvoices()
    {
        try {
            // Reset created invoices array and clear any previous messages
            $this->createdInvoices = [];
            $this->addressWarnings = [];
            $this->tagWarnings = [];
            Session::forget(['message', 'alert-class']);
            
            if($this->customerInvoice == ""){
                Session::flash('message', 'Please select a customer first');
                Session::flash('alert-class', 'alert-warning');
                return;
            }
            
            $invoices = $this->generateInvoices();
            if (empty($invoices)) {
                Session::flash('message', 'No invoices to process');
                Session::flash('alert-class', 'alert-info');
                return;
            }
            
            $this->totalInvoices = count($invoices);
            // dd($invoices);

            // Verify customers exist and get IDs
            $customers = $this->getCustomerIDs($invoices);
            if ($this->isError($customers)) {
                Session::flash('message', $customers);
                Session::flash('alert-class', 'alert-danger');
                return;
            }
            $customers = collect($customers);
            
            $templates = $this->getTemplateIDs($invoices);
            if ($this->isError($templates)) {
                Session::flash('message', $templates);
                Session::flash('alert-class', 'alert-danger');
                return;
            }
            $templates = collect($templates);
            
            $accounts = $this->getAccountIDs($invoices);
            if ($this->isError($accounts)) {
                Session::flash('message', $accounts);
                Session::flash('alert-class', 'alert-danger');
                return;
            }
            $accounts = collect($accounts);
            
            $tax = $this->getTaxID();
            if ($this->isError($tax)) {
                Session::flash('message', $tax);
                Session::flash('alert-class', 'alert-danger');
                return;
            }
            // $reportingTags = $this->getReportingTags();
            // $addresses = collect($this->getAddressIDs($invoices, $customers));
            // dd($addresses);

            // foreach($reportingTags as $r=>$rtag){
            //     $reportingTags[$r]['tag_options'] = collect(str_getcsv($rtag['tag_options']));
            // }
            // $reportingTags = collect($reportingTags);

            foreach ($invoices as $invoice) {
            try {
                // dd($accounts);
                // dd($templates->where('template_name', $invoice->invoiceLayout)->first());
                // dd($templates->pluck('template_name'),$templates->where('name', $invoice->invoiceLayout), $invoice->invoiceLayout);
                // Prepare the invoice data payload
                $customer = $customers->where('contact_name', $invoice->customerName)->first();
                if (!$customer) {
                    Session::flash('message', "Error: Customer '{$invoice->customerName}' not found in Zoho");
                    Session::flash('alert-class', 'alert-danger');
                    continue;
                }
                $template = $templates->where('template_name', $invoice->invoiceLayout)->first();
                if (!$template) {
                    Session::flash('message', "Error: Template '{$invoice->invoiceLayout}' not found in Zoho");
                    Session::flash('alert-class', 'alert-danger');
                    continue;
                }
                
                $shippingAddressId = $this->getAddressIDs($customer['contact_id'], $invoice->invoiceShippingAddress);
                $addressWarning = null;
                
                if ($this->isError($shippingAddressId)) {
                    // Address not found, but continue without it
                    $shippingAddressId = null;
                    $addressWarning = "Address '{$invoice->invoiceShippingAddress}' not found for customer - invoice created without shipping address";
                }
                
                $lineItems = $this->buildLineItems($invoice, $tax, $accounts);
                
                // Skip invoices with no line items (all shipped quantities are 0)
                if (empty($lineItems)) {
                    $warningMessage = "Invoice {$invoice->invoicePO} ({$invoice->customerName}) skipped - all shipped quantities are 0. Please check the shipment data.";
                    if (!in_array($warningMessage, $this->addressWarnings)) {
                        $this->addressWarnings[] = $warningMessage;
                    }
                    Log::warning('Skipping invoice submission - all shipped quantities are 0', [
                        'customer_name' => $invoice->customerName,
                        'reference_number' => $invoice->invoicePO,
                        'original_lines_count' => count($invoice->lines),
                    ]);
                    Session::flash('message', "Invoice for PO '{$invoice->invoicePO}' ({$invoice->customerName}) cannot be created - all shipped quantities are 0. Please check the shipment data.");
                    Session::flash('alert-class', 'alert-danger');
                    continue;
                }
                
                $payload = [
                    "customer_id" => $customer['contact_id'],
                    "template_id" => $template['template_id'],
                    "reference_number" => $invoice->invoicePO,
                    "date" => $invoice->invoiceDate,
                    "notes" => $invoice->invoiceComments,
                    "custom_fields" => $this->buildCustomFields($invoice),
                    "line_items" => $lineItems,
                    "status" => "draft"
                ];
                
                // Only add shipping_address_id if we have a valid one
                if ($shippingAddressId) {
                    $payload["shipping_address_id"] = $shippingAddressId;
                }
                
                $invoiceData = $this->zoho->createInvoice($payload, '20079258157');
                // dd($invoiceData);

                if ($invoiceData) {
                    $invoiceNo = $invoiceData['invoice_number'] ?? 'UNKNOWN';
                    
                    // Add to created invoices array
                    $this->createdInvoices[] = $invoiceNo;
                    
                    // Track address warning if any
                    if ($addressWarning) {
                        $this->addressWarnings[] = "Invoice {$invoiceNo}: {$addressWarning}";
                    }

                    foreach ($invoice->drops as $dropId) {
                        $drop = ShipmentLine::find($dropId);
                        if ($drop) {
                            $drop->rt_invoice = $invoiceNo;
                            $drop->save();
                        }
                    }
                } else {
                    Session::flash('message', 'Zoho API Error: Invoice creation failed.');
                    Session::flash('alert-class', 'alert-danger');
                }

            } catch (IdentityProviderException $e) {
                // Handle OAuth errors
                Log::error('Zoho OAuth Error:', [
                    'message' => $e->getMessage(),
                    'response' => $e->getResponseBody(),
                ]);
                Session::flash('message', 'Zoho OAuth Error: ' . $e->getMessage());
                Session::flash('alert-class', 'alert-danger');
            }
            
            // Show final success message with all created invoice numbers
            if (!empty($this->createdInvoices)) {
                $invoiceCount = count($this->createdInvoices);
                $invoiceList = implode(', ', $this->createdInvoices);
                
                $message = $invoiceCount === 1 
                    ? "Invoice created successfully: {$invoiceList}"
                    : "{$invoiceCount} invoices created successfully: {$invoiceList}";
                
                // Add address warnings to the message if any
                if (!empty($this->addressWarnings)) {
                    $message .= "\n\nAddress Warnings:\n" . implode("\n", $this->addressWarnings);
                }
                
                // Add tag warnings to the message if any
                if (!empty($this->tagWarnings)) {
                    $message .= "\n\nReporting Tag Warnings:\n" . implode("\n", $this->tagWarnings);
                }
                
                // Store the message and warnings in component properties instead of session
                $this->completionMessage = $message;
                $this->showCompletionModal = true;
            }
        }
        } catch (\Exception $e) {
            // Handle any other unexpected errors at the method level
            Log::error('Unexpected error in sendInvoices method:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            Session::flash('message', 'Unexpected error: ' . $e->getMessage());
            Session::flash('alert-class', 'alert-danger');
        }
    }

    public function generateInvoices()
    {
        if ($this->zoho->isConnected()) {
            $this->zohoConnected = true;
        } else {
            $this->authUrl = $this->zoho->getAuthUrl();
            $this->zohoConnected = false;
        }

        $query = ShipmentLine::with([
            'customer_order_lines.customer_orders.customers.customer_invoice_layouts.invoice_layouts',
            'customer_order_lines.colourways.style_versions.styles.designs',
            'customer_order_lines.customer_order_line_quantities',
            'shipment_line_sizes.sizes',
        ]);

        if ($this->customerInvoice) {
            $query->whereRelation('customer_order_lines.customer_orders.customers', 'id', $this->customerInvoice);
        }

        $query->whereRelation('customer_order_lines.customer_orders', 'order_type', "wholesale");
        $query->whereRelation('customer_order_lines.customer_orders', 'cancelled', 0);
        $query->where('complete', 1);
        $query->where(function($q) {
            $q->whereNull('shipment_lines.rt_invoice')
              ->orWhere('shipment_lines.rt_invoice', '');
        });

        // $query->where('id', 7712);

        $lines = $query->get()->groupBy('customer_order_lines.customer_orders.customers_id');

        // dd($query->get()->pluck('id'));

        $invoices = [];

        foreach ($lines as $customerGroup) {
            $customer = $customerGroup->first()->customer_order_lines->customer_orders->customers;
            $layout = $customer->customer_invoice_layouts
                ->where('incoterm', $customerGroup->first()->customer_order_lines->customer_orders->incoterms)
                ->first()
                ->invoice_layouts ?? InvoiceLayout::where('default', 1)->first();

            $groupedData = match ($layout->invoice_by) {
                'order' => $customerGroup->groupBy('customer_order_lines.customer_orders_id'),
                'style' => $customerGroup->groupBy('customer_order_lines.colourways.style_versions.styles_id'),
                'factory' => $customerGroup->groupBy('customer_order_lines.colourways.style_versions.factory_id'),
                'drop' => $customerGroup->groupBy('id'),
                default => $customerGroup->groupBy('collection_date'),
            };

            foreach ($groupedData as $orderGroup) {
                $invoice = new ZohoInvoice();

                // Initialize $lineData with default values from first drop to avoid undefined variable error
                $firstDrop = $orderGroup->first();
                $lineData = [
                    'line_customer_ref' => $firstDrop->customer_order_lines->colourways->style_versions->styles->customer_ref ?? '',
                    'line_description' => $firstDrop->customer_order_lines->colourways->style_versions->styles->designs->description ?? '',
                    'line_designs_id' => $firstDrop->customer_order_lines->colourways->style_versions->styles->designs_id ?? '',
                    'line_style_version' => $firstDrop->customer_order_lines->colourways->style_versions->name ?? '',
                    'line_colourway' => $firstDrop->customer_order_lines->colourways->name ?? '',
                    'line_customer_po' => $firstDrop->customer_order_lines->customer_orders->customer_po ?? '',
                    'line_qty' => 0,
                    'line_price' => 0,
                    'line_vat' => false,
                    'line_gross_weight' => $firstDrop->gross_weight ?? 0,
                    'line_net_weight' => $firstDrop->net_weight ?? 0,
                    'line_no_cartons' => $firstDrop->no_cartons ?? 0,
                    'line_shipment_mode' => $firstDrop->customer_order_lines->customer_orders->shipment_mode ?? '',
                ];

                foreach ($orderGroup as $drop) {
                    $quote = $drop->customer_order_lines->customer_order_line_quantities->first()->price_model['discount_price'] ?? 0;

                    if(!$layout->per_size){
                        $tax = !($drop->customer_order_lines->colourways->style_versions->styles->category == 'childrens' ||
                            $drop->customer_order_lines->customer_orders->incoterms == 'FOB');

                        $lineData = [
                            'line_customer_ref' => $drop->customer_order_lines->colourways->style_versions->styles->customer_ref,
                            'line_description' => $drop->customer_order_lines->colourways->style_versions->styles->designs->description,
                            'line_designs_id' => $drop->customer_order_lines->colourways->style_versions->styles->designs_id,
                            'line_style_version' => $drop->customer_order_lines->colourways->style_versions->name,
                            'line_colourway' => $drop->customer_order_lines->colourways->name,
                            'line_customer_po' => $drop->customer_order_lines->customer_orders->customer_po,
                            'line_qty' => $drop->shipment_line_sizes->sum('shipped_qty'),
                            'line_price' => $quote ?? 0,
                            'line_vat' => $tax,

                            'line_gross_weight' => $drop->gross_weight,
                            'line_net_weight' => $drop->net_weight,
                            'line_no_cartons' => $drop->no_cartons,

                            'line_shipment_mode' => $drop->customer_order_lines->customer_orders->shipment_mode,
                        ];

                        // Skip lines with 0 quantity
                        if ($lineData['line_qty'] > 0) {
                            $invoice->addLine([
                                'item_title' => $this->formatInvoiceField($layout->line_title, $lineData),
                                'lineDescription' => $this->formatInvoiceField($layout->line_description, $lineData),
                                'price' => $this->formatInvoiceField($layout->line_price, $lineData),
                                'qty' => $lineData['line_qty'],
                                'vat' => $lineData['line_vat'],
                            ]);
                        }
                    }
                    else{
                        foreach ($drop->shipment_line_sizes as $size) {
                            // Skip sizes with 0 shipped quantity
                            if ($size->shipped_qty <= 0) {
                                continue;
                            }
                            
                            $orderSize = $drop->customer_order_lines->customer_order_line_quantities
                                ->where('sizes_id', $size->sizes_id)
                                ->first();

                            $tax = !($drop->customer_order_lines->colourways->style_versions->styles->category == 'childrens' ||
                                $drop->customer_order_lines->customer_orders->incoterms == 'FOB');
                            $lineData = [
                                'line_customer_ref' => $drop->customer_order_lines->colourways->style_versions->styles->customer_ref,
                                'line_customer_description' => $drop->customer_order_lines->colourways->style_versions->styles->customer_description,
                                'line_sku' => $orderSize->SKU,
                                'line_size' => $size->sizes->name,
                                'line_description' => $drop->customer_order_lines->colourways->style_versions->styles->designs->description,
                                'line_designs_id' => $drop->customer_order_lines->colourways->style_versions->styles->designs_id,
                                'line_style_version' => $drop->customer_order_lines->colourways->style_versions->name,
                                'line_colourway' => $drop->customer_order_lines->colourways->name,
                                'line_customer_po' => $drop->customer_order_lines->customer_orders->customer_po,
                                'line_composition' => $drop->customer_order_lines->colourways->composition,
                                'line_commodity_code' => $drop->customer_order_lines->colourways->style_versions->styles->commodity_codes_id,
                                'line_qty' => $size->shipped_qty,
                                'line_price' => $orderSize->price_model['discount_price'] ?? 0,
                                'line_asn' => $drop->asn,
                                'line_category' => ucfirst($drop->customer_order_lines->colourways->style_versions->styles->category),
                                'line_exfty_date' => Carbon::parse($this->date ?? $drop->exfty)->format('d/m/Y'),

                                'line_vat' => $tax,

                                'line_gross_weight' => $drop->gross_weight,
                                'line_net_weight' => $drop->net_weight,
                                'line_no_cartons' => $drop->no_cartons,

                                'line_shipment_mode' => $drop->customer_order_lines->customer_orders->shipment_mode,
                            ];

                            $invoice->addLine([
                                'item_title' => $this->formatInvoiceField($layout->line_title, $lineData),
                                'lineDescription' => $this->formatInvoiceField($layout->line_description, $lineData),
                                'price' => $this->formatInvoiceField($layout->line_price, $lineData),
                                'qty' => $lineData['line_qty'],
                                'vat' => $lineData['line_vat'],
                            ]);
                        }
                    }

                    $invoice->drops[] = $drop->id;
                }

                $customer = $orderGroup->first()->customer_order_lines->customer_orders->customers;
                
                $invoice->customerName = $customer->zoho_customer ?? $customer->name;
                $invoice->zohoCustomer = $customer->zoho_customer;
                $invoice->invoiceDate = $this->date ?? $orderGroup->first()->collection_date?->format('Y-m-d') ?? $orderGroup->first()->exfty?->format('Y-m-d');
                
                // Collect all unique POs from the order group
                $allPOs = $orderGroup
                    ->pluck('customer_order_lines.customer_orders.customer_po')
                    ->filter()
                    ->unique()
                    ->values()
                    ->implode(', ');
                $invoice->invoicePO = $allPOs;
                $invoice->invoiceLayout = $layout['invoice_layout'];
                $invoice->invoiceComments = $this->formatInvoiceField($layout->invoice_comments, $lineData);
                $invoice->invoiceShippingAddress = $orderGroup->first()->customer_order_lines->customer_orders->customer_addresses?->name ?? '';
                $invoice->zohoCustomerTag = $customer->zoho_customer_tag ?? $customer->zoho_customer ?? $customer->name;
                $invoice->zohoAccount = $layout->zoho_account;
                $invoice->zohoDepartment = $orderGroup->first()->customer_order_lines->customer_orders->departments?->zoho_name ?? '';
                $invoice->zohoFactored = $customer->zoho_factored;
                $invoice->factory_invoice = $orderGroup->first()->factory_invoice;

                // Skip invoices with no line items (all shipped quantities are 0)
                if (empty($invoice->lines)) {
                    Log::warning('Skipping invoice generation - all shipped quantities are 0', [
                        'customer_name' => $customer->name,
                        'customer_po' => $invoice->invoicePO,
                        'drops' => $invoice->drops,
                    ]);
                    Session::flash('message', "Invoice for PO '{$invoice->invoicePO}' ({$customer->name}) cannot be created - all shipped quantities are 0. Please check the shipment data.");
                    Session::flash('alert-class', 'alert-danger');
                    continue;
                }

                $invoices[] = $invoice;
            }
        }
        // dd($invoices);
        // dd($invoices);
        return $invoices;
    }

    private function buildCustomFields($invoice)
    {
        $customFields = [];
        
        // PI Number
        $piNumberFieldId = $this->getInvoiceCustomFieldId('PI Number');
        if (!$this->isError($piNumberFieldId) && $invoice->factory_invoice !== null) {
            $customFields[] = [
                "customfield_id" => $piNumberFieldId,
                "value" => $invoice->factory_invoice,
            ];
        }

        // Product Type
        $productTypeFieldId = $this->getInvoiceCustomFieldId('Product Type');
        if (!$this->isError($productTypeFieldId)) {
            $customFields[] = [
                "customfield_id" => $productTypeFieldId,
                "value" => "Production",
            ];
        }

        // Supplier
        $supplierFieldId = $this->getInvoiceCustomFieldId('Supplier');
        if (!$this->isError($supplierFieldId) && !empty($this->supplier)) {
            $customFields[] = [
                "customfield_id" => $supplierFieldId,
                "value" => $this->supplier,
            ];
        }
        
        return $customFields;
    }
    
    private function buildReportingTags($invoice)
    {
        $tags = [];
        $missingData = [];
        
        try {
            // Check if required data is missing
            if (empty($invoice->zohoCustomerTag)) {
                $missingData[] = 'Customer tag is empty';
            }
            if (empty($invoice->zohoDepartment)) {
                $missingData[] = 'Department is empty';
            }
            if (empty($invoice->zohoFactored)) {
                $missingData[] = 'Factored status is empty';
            }
            
            // Customer tag
            $customerTagId = $this->getReportingTagId('Customer');
            if ($this->isError($customerTagId)) {
                Log::warning('Failed to get Customer reporting tag ID', ['error' => $customerTagId]);
                $missingData[] = 'Customer tag not configured in Zoho';
            } else {
                $customerTagOptions = $this->reportingTagCustomers();
                if ($this->isError($customerTagOptions)) {
                    Log::warning('Failed to get Customer tag options', ['error' => $customerTagOptions]);
                    $missingData[] = 'Customer tag options unavailable';
                } else if (!empty($invoice->zohoCustomerTag)) {
                    $customerTagOption = collect($customerTagOptions)->where('tag_option_name', $invoice->zohoCustomerTag)->first();
                    if ($customerTagOption && !empty($customerTagOption['tag_option_id'])) {
                        $tags[] = [
                            "tag_id" => $customerTagId,
                            "tag_option_id" => $customerTagOption['tag_option_id'],
                        ];
                    } else {
                        Log::warning('Customer tag option not found in Zoho', [
                            'looking_for' => $invoice->zohoCustomerTag,
                            'available_options' => collect($customerTagOptions)->pluck('tag_option_name')->take(10)->toArray(),
                            'total_options' => collect($customerTagOptions)->count(),
                        ]);
                        $missingData[] = "Customer '{$invoice->zohoCustomerTag}' not found in Zoho";
                    }
                }
            }
            
            // Department tag
            $departmentTagId = $this->getReportingTagId('Department');
            if ($this->isError($departmentTagId)) {
                Log::warning('Failed to get Department reporting tag ID', ['error' => $departmentTagId]);
                $missingData[] = 'Department tag not configured in Zoho';
            } else {
                $departmentTagOptions = $this->reportingTagDepartments();
                if ($this->isError($departmentTagOptions)) {
                    Log::warning('Failed to get Department tag options', ['error' => $departmentTagOptions]);
                    $missingData[] = 'Department tag options unavailable';
                } else if (!empty($invoice->zohoDepartment)) {
                    $departmentTagOption = collect($departmentTagOptions)->where('tag_option_name', $invoice->zohoDepartment)->first();
                    if ($departmentTagOption && !empty($departmentTagOption['tag_option_id'])) {
                        $tags[] = [
                            "tag_id" => $departmentTagId,
                            "tag_option_id" => $departmentTagOption['tag_option_id'],
                        ];
                    } else {
                        Log::warning('Department tag option not found in Zoho', [
                            'looking_for' => $invoice->zohoDepartment,
                            'available_options' => collect($departmentTagOptions)->pluck('tag_option_name')->take(10)->toArray(),
                        ]);
                        $missingData[] = "Department '{$invoice->zohoDepartment}' not found in Zoho";
                    }
                }
            }
            
            // Factored tag
            $factoredTagId = $this->getReportingTagId('Factored');
            if ($this->isError($factoredTagId)) {
                Log::warning('Failed to get Factored reporting tag ID', ['error' => $factoredTagId]);
                $missingData[] = 'Factored tag not configured in Zoho';
            } else {
                $factoredTagOptions = $this->reportingTagFactored();
                if ($this->isError($factoredTagOptions)) {
                    Log::warning('Failed to get Factored tag options', ['error' => $factoredTagOptions]);
                    $missingData[] = 'Factored tag options unavailable';
                } else if (!empty($invoice->zohoFactored)) {
                    $factoredTagOption = collect($factoredTagOptions)->where('tag_option_name', $invoice->zohoFactored)->first();
                    if ($factoredTagOption && !empty($factoredTagOption['tag_option_id'])) {
                        $tags[] = [
                            "tag_id" => $factoredTagId,
                            "tag_option_id" => $factoredTagOption['tag_option_id'],
                        ];
                    } else {
                        Log::warning('Factored tag option not found in Zoho', [
                            'looking_for' => $invoice->zohoFactored,
                            'available_options' => collect($factoredTagOptions)->pluck('tag_option_name')->take(10)->toArray(),
                        ]);
                        $missingData[] = "Factored '{$invoice->zohoFactored}' not found in Zoho";
                    }
                }
            }
            
            // Add warnings if any data is missing
            if (!empty($missingData)) {
                $warningMessage = "Invoice {$invoice->invoicePO}: " . implode(', ', $missingData);
                if (!in_array($warningMessage, $this->tagWarnings)) {
                    $this->tagWarnings[] = $warningMessage;
                }
            }
            
        } catch (\Exception $e) {
            Log::error('Exception in buildReportingTags', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
        }
        
        // Filter out any tags with empty values (defensive)
        return array_filter($tags, function($tag) {
            return !empty($tag['tag_id']) && !empty($tag['tag_option_id']);
        });
    }
    
    private function buildLineItems($invoice, $tax, $accounts)
    {
        $lineItems = [];
        
        // Resolve the correct account ID for invoice line items
        $accountId = $this->resolveAccountId($invoice->zohoAccount, $accounts);
        if (!$accountId) {
            Session::flash('message', "Error: No valid Income-type account found for '{$invoice->zohoAccount}'. Invoice line items require an Income account in Zoho Books.");
            Session::flash('alert-class', 'alert-danger');
            return [];
        }
        
        // Build reporting tags once (same for all line items in this invoice)
        $reportingTags = $this->buildReportingTags($invoice);
        
        foreach ($invoice->lines as $line) {
            // Skip lines with 0 quantity
            $quantity = $line['qty'] ?? 1;
            if ($quantity <= 0) {
                continue;
            }
            
            $lineItem = [
                "name" => $line['item_title'],
                "description" => $line['lineDescription'],
                "rate" => $line['price'],
                "quantity" => $quantity,
                "account_id" => $accountId,
            ];
            
            // Only add tax_id if VAT applies and we have a valid tax ID
            if ($line['vat'] && !empty($tax['tax_id'])) {
                $lineItem["tax_id"] = $tax['tax_id'];
            }
            
            // Only add tags if we have valid reporting tags
            if (!empty($reportingTags)) {
                $lineItem["tags"] = $reportingTags;
            }
            
            $lineItems[] = $lineItem;
        }
        
        return $lineItems;
    }
    
    /**
     * Resolve a Zoho account_id from an account code or name.
     * Validates that the account is an Income-type account (required for invoice line items).
     * Falls back to the first Income/Sales account if the preference is invalid.
     */
    private function resolveAccountId(?string $preference, $accounts): ?string
    {
        try {
            if ($accounts->isEmpty()) {
                Log::error('Zoho accounts list is empty');
                return null;
            }

            $pref = trim((string)($preference ?? ''));
            
            if ($pref !== '') {
                // Try by account_id first
                $acc = $accounts->firstWhere('account_id', $pref);
                if ($acc && $this->isIncomeAccount($acc)) {
                    return $acc['account_id'];
                }
                
                // Try by account_code
                $acc = $accounts->firstWhere('account_code', $pref);
                if ($acc && $this->isIncomeAccount($acc)) {
                    return $acc['account_id'];
                }
                
                // Try by exact account_name
                $acc = $accounts->first(function ($a) use ($pref) {
                    return trim((string)($a['account_name'] ?? '')) === $pref;
                });
                if ($acc && $this->isIncomeAccount($acc)) {
                    return $acc['account_id'];
                }
                
                // Try case-insensitive account_name
                $lp = mb_strtolower($pref);
                $acc = $accounts->first(function ($a) use ($lp) {
                    return mb_strtolower(trim((string)($a['account_name'] ?? ''))) === $lp;
                });
                if ($acc && $this->isIncomeAccount($acc)) {
                    return $acc['account_id'];
                }
                
                // If account found but not Income type, log warning
                $foundAcc = $accounts->firstWhere('account_code', $pref) 
                    ?? $accounts->first(fn($a) => mb_strtolower(trim((string)($a['account_name'] ?? ''))) === $lp);
                if ($foundAcc) {
                    Log::warning("Account '{$pref}' found but is not an Income type", [
                        'account_type' => $foundAcc['account_type'] ?? 'unknown',
                        'account_name' => $foundAcc['account_name'] ?? '',
                    ]);
                }
            }

            // Fallback: find an income/sales account
            $acc = $accounts->first(function ($a) {
                return $this->isIncomeAccount($a);
            });
            
            if ($acc) {
                Log::info("Using fallback Income account", [
                    'account_id' => $acc['account_id'],
                    'account_name' => $acc['account_name'] ?? '',
                    'original_preference' => $pref,
                ]);
                return $acc['account_id'];
            }

            Log::error('No Income-type account found in Zoho Books');
            return null;
        } catch (\Throwable $e) {
            Log::error('resolveAccountId failed', ['error' => $e->getMessage()]);
            return null;
        }
    }
    
    /**
     * Check if an account is an Income-type account (valid for invoice line items).
     * In Zoho Books, only 'income' and 'other_income' account types are valid for invoices.
     */
    private function isIncomeAccount(array $account): bool
    {
        $type = mb_strtolower((string)($account['account_type'] ?? ''));
        
        // Valid account types for invoice line items in Zoho Books
        // Must be income or other_income - NOT asset, expense, equity, etc.
        return $type === 'income' || $type === 'other_income';
    }






    private function isError($value)
    {
        return is_string($value) && str_starts_with($value, 'Error:');
    }
    
    private function formatInvoiceField($template, $data)
    {
        preg_match_all('/\{([^}]+)\}/', $template, $matches);
        $variableNames = $matches[1];

        $formattedField = $template;
        foreach ($variableNames as $varName) {
            $formattedField = str_replace('{' . $varName . '}', $data[$varName] ?? '', $formattedField);
        }

        return $formattedField;
    }

    public function getInvoiceCustomFieldValues($name){
        try {
            $fields = $this->zoho->getCustomFields('20079258157');
            
            if ($fields->isEmpty()) {
                return []; // Return empty array instead of error string
            }
            
            // The API returns fields grouped by module, so we need to look in the 'invoice' array
            if ($fields->has('invoice')) {
                $invoiceFields = collect($fields['invoice']);
            } else {
                // If not in module structure, try to find fields by label directly
                $invoiceFields = $fields->filter(function($field) {
                    return is_array($field) && isset($field['label']) && isset($field['search_entity']) && $field['search_entity'] === 'invoice';
                });
            }
            
            if ($invoiceFields->isEmpty()) {
                return []; // Return empty array instead of error string
            }
            
            $invoiceField = $invoiceFields->firstWhere('label', $name);
            if (!$invoiceField) {
                return []; // Return empty array instead of error string
            }
            
            // Ensure we always return an array
            $values = $invoiceField['values'] ?? [];
            return is_array($values) ? $values : [];
        } catch (\Exception $e) {
            Log::error("Error retrieving custom field values for '{$name}'", ['error' => $e->getMessage()]);
            return []; // Return empty array instead of error string
        }
    }
    
    /**
     * Get suppliers for the dropdown - this method specifically handles supplier data
     * and ensures it returns an array that can be iterated over in the Blade template
     */
    public function getSuppliers(){
        try {
            // Check if Zoho is connected first
            if (!$this->zohoConnected) {
                return [];
            }
            
            // Try to get suppliers from custom fields first
            $supplierField = $this->getInvoiceCustomFieldValues('Supplier');
            if (!empty($supplierField)) {
                return $supplierField;
            }
            
            // If no custom field values, return a default empty array
            // You can modify this to fetch suppliers from other sources if needed
            return [];
        } catch (\Exception $e) {
            return []; // Return empty array instead of error string
        }
    }

    public function getInvoiceCustomFieldId($name){
        try {
            $fields = $this->zoho->getCustomFields('20079258157');
            
            if (!$fields || !isset($fields['invoice'])) {
                return "Error: Invoice custom fields not available from Zoho API";
            }
            
            $invoiceField = collect($fields['invoice'])->firstWhere('label', $name);
            if (!$invoiceField) {
                return "Error: Custom field '{$name}' not found in invoice fields";
            }
            
            return $invoiceField['customfield_id'] ?? "Error: No ID found for custom field '{$name}'";
        } catch (\Exception $e) {
            return "Error: Failed to retrieve invoice custom field ID - " . $e->getMessage();
        }
    }

    public function getCustomerIDs(){
        try {
            $customers = $this->zoho->getCustomers('20079258157');
            if (!$customers) {
                return "Error: No customers found from Zoho API";
            }
            return $customers;
        } catch (\Exception $e) {
            return "Error: Failed to retrieve customers - " . $e->getMessage();
        }
    }

    public function getReportingTagId($name){
        try {
            $tag = $this->zoho->getReportingTags('20079258157');
            if (!$tag || $tag->isEmpty()) {
                Log::warning('No reporting tags available from Zoho API');
                return "Error: No reporting tags available from Zoho API";
            }
            
            $tagData = collect($tag)->where('tag_name', $name)->first();
            if (!$tagData) {
                Log::warning("Reporting tag not found", [
                    'looking_for' => $name,
                    'available_tags' => $tag->pluck('tag_name')->toArray(),
                ]);
                return "Error: Reporting tag '{$name}' not found";
            }
            
            return $tagData['tag_id'] ?? "Error: No ID found for reporting tag '{$name}'";
        } catch (\Exception $e) {
            Log::error('Failed to retrieve reporting tag ID', [
                'tag_name' => $name,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return "Error: Failed to retrieve reporting tag ID - " . $e->getMessage();
        }
    }

    public function reportingTagCustomers(){
        try {
            $tag_id = $this->getReportingTagId('Customer');
            if ($this->isError($tag_id)) {
                Log::warning('Failed to get Customer reporting tag ID', ['error' => $tag_id]);
                return $tag_id; // Return the error message
            }
            
            $tag_values = $this->zoho->getReportingTagOptions('20079258157', $tag_id);
            if (!$tag_values || $tag_values->isEmpty()) {
                Log::warning('No customer tag options found', ['tag_id' => $tag_id]);
                return "Error: No customer tag options found";
            }
            
            return $tag_values;
        } catch (\Exception $e) {
            Log::error('Failed to retrieve customer tag options', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return "Error: Failed to retrieve customer tag options - " . $e->getMessage();
        }
    }

    public function reportingTagDepartments(){
        try {
            $tag_id = $this->getReportingTagId('Department');
            if ($this->isError($tag_id)) {
                Log::warning('Failed to get Department reporting tag ID', ['error' => $tag_id]);
                return $tag_id; // Return the error message
            }
            
            $tag_values = $this->zoho->getReportingTagOptions('20079258157', $tag_id);
            if (!$tag_values || $tag_values->isEmpty()) {
                Log::warning('No department tag options found', ['tag_id' => $tag_id]);
                return "Error: No department tag options found";
            }
            
            return $tag_values;
        } catch (\Exception $e) {
            Log::error('Failed to retrieve department tag options', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return "Error: Failed to retrieve department tag options - " . $e->getMessage();
        }
    }

    public function reportingTagFactored(){
        try {
            $tag_id = $this->getReportingTagId('Factored');
            if ($this->isError($tag_id)) {
                Log::warning('Failed to get Factored reporting tag ID', ['error' => $tag_id]);
                return $tag_id; // Return the error message
            }
            
            $tag_values = $this->zoho->getReportingTagOptions('20079258157', $tag_id);
            if (!$tag_values || $tag_values->isEmpty()) {
                Log::warning('No factored tag options found', ['tag_id' => $tag_id]);
                return "Error: No factored tag options found";
            }
            
            return $tag_values;
        } catch (\Exception $e) {
            Log::error('Failed to retrieve factored tag options', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return "Error: Failed to retrieve factored tag options - " . $e->getMessage();
        }
    }

    public function getReportingTags(){
        try {
            $tags = $this->zoho->getReportingTags('20079258157');
            if (!$tags) {
                return "Error: No reporting tags available from Zoho API";
            }
            return $tags;
        } catch (\Exception $e) {
            return "Error: Failed to retrieve reporting tags - " . $e->getMessage();
        }
    }

    public function getAddressIDs($customer, $address){
        try {
            if (!$customer) {
                return "Error: Customer ID is required to get addresses";
            }
            
            $addresses = $this->zoho->getAddresses('20079258157', $customer);
            if (!$addresses) {
                return "Error: No addresses found for customer";
            }
          
            $addressData = $addresses->firstWhere('attention', $address);
            if (!$addressData) {
                return "Error: Address '{$address}' not found for customer";
            }
            
            return $addressData['address_id'] ?? "Error: No ID found for address '{$address}'";
        } catch (\Exception $e) {
            return "Error: Failed to retrieve address ID - " . $e->getMessage();
        }
    }

    public function getAccountIDs(){
        try {
            $accounts = $this->zoho->getChartOfAccounts('20079258157');
            if (!$accounts) {
                return "Error: No chart of accounts available from Zoho API";
            }
            return $accounts;
        } catch (\Exception $e) {
            return "Error: Failed to retrieve chart of accounts - " . $e->getMessage();
        }
    }

    private function getTemplateIDs(array $invoices)
    {
        try {
            $templates = $this->zoho->getInvoiceTemplates('20079258157');
            if (!$templates) {
                return "Error: No invoice templates available from Zoho API";
            }
            return $templates;
        } catch (\Exception $e) {
            return "Error: Failed to retrieve invoice templates - " . $e->getMessage();
        }
    }

    public function getTaxID(){
        try {
            $taxes = $this->zoho->getTaxes('20079258157');
            if (!$taxes) {
                return "Error: No taxes available from Zoho API";
            }
            
            $tax = collect($taxes)->where('tax_name', 'Standard Rate')->first();
            if (!$tax) {
                return "Error: Standard Rate tax not found";
            }
            
            return $tax;
        } catch (\Exception $e) {
            return "Error: Failed to retrieve tax information - " . $e->getMessage();
        }
    }

    private function getAccessTokenFromYourDataStore()
    {
        $user = Auth::user();

        if ($user && $user->zoho_access_token) {
            return unserialize($user->zoho_access_token);
        } else {
            return null;
        }
    }

    // public function initiateZohoOAuth()
    // {
    //     $authorizationUrl = ZohoFacade::getAuthorizationUrl([
    //         'scope' => [
    //             'ZohoBooks.invoices.CREATE',
    //             'ZohoBooks.invoices.READ',
    //             'ZohoBooks.contacts.READ',
    //             'ZohoBooks.accountants.READ',
    //             'ZohoBooks.settings.READ',
    //             'ZohoBooks.contacts.READ'
    //         ],
    //         'access_type' => 'offline',
    //         'prompt' => 'consent',
    //     ]);

    //     $state = ZohoFacade::getState();
    //     session(['oauth2state' => $state]);

    //     $this->dispatch('open-zoho-oauth-popup', url: $authorizationUrl);
    // }
}

